#!/usr/bin/env python3
"""Convert markdown files to html."""
import argparse
from datetime import datetime, timezone
import io
import json
import os
import shutil

from mako.template import Template
from mako.lookup import TemplateLookup
import markdown

parser = argparse.ArgumentParser(prog='thera')
parser.add_argument('files', nargs='*')
parser.add_argument('-b', '--blog', action='store_true', help='Build blog files')
parser.add_argument('-c', '--config', help='Configuration file path')
parser.add_argument('-C', '--clean',
                    action='store_true', help='Delete distribution directory')
parser.add_argument('-s', '--static', default=[],
                    nargs='*', help='Include these static directories')
parser.add_argument('-t', '--template', help='Template file path')

cfg = {'DISTRIBUTION_DIRECTORY': 'dist', }


class Converter(object):
    def __init__(self):
        self.dist_dir = None
        self.blog_dist_dir = None
        self.mdconverter = markdown.Markdown(
            extensions=[
                'footnotes',
                'meta',
                'smarty',
                'tables',
                'toc'],
            output_format='html5')

    def build(self):
        cwd = os.getcwd()
        self.config = cfg
        self.read_config()
        self.dist_dir = os.path.join(
            cwd,
            self.config['DISTRIBUTION_DIRECTORY'])
        blog_dist = self.config.get('BLOG_DISTRIBUTION_DIRECTORY')
        if blog_dist:
            self.blog_dist_dir = os.path.join(cwd, blog_dist)
        if args.clean:
            print(f'Cleaning {self.dist_dir}')
            shutil.rmtree(self.dist_dir, ignore_errors=True)
            if self.blog_dist_dir:
                shutil.rmtree(self.blog_dist_dir, ignore_errors=True)
            return
        os.makedirs(self.dist_dir, exist_ok=True)

        # Copy static assets to dist directory
        for dir in args.static:
            dest = os.path.join(self.dist_dir, dir)
            print(f'{dir} -> {dest}')
            shutil.rmtree(dest, ignore_errors=True)
            shutil.copytree(dir, dest)

        articles = []
        for path in args.files:
            articles.append(self.create_page(path))
        if args.blog:
            datefmt = self.config['SOURCE_DATE_FORMAT']
            # Sort articles newest first.
            articles.sort(
                key=lambda r: datetime.strptime(r['date'], datefmt),
                reverse=True)
            self.create_RSS_feed(articles)
            self.create_blog_archive(articles)

    def convert(self, mdpath, temppath=''):
        """Convert markdown file to HTML.

        Parameters:
            mdpath (s): path to markdown file.
            temppath (s): path to template file.

        Returns:
            data (dict), html (s): html with template.
        """
        with io.open(mdpath, 'r', encoding='ISO-8859-1') as f:
            md = f.read()
        content = self.mdconverter.reset().convert(md)
        metadata = self.mdconverter.Meta
        data = self.normalize(metadata)
        data['title'] = self.create_title(data, content)
        data['slug'] = self.create_slug(data, mdpath)
        data['path'] = os.path.join(self.dist_dir, os.path.dirname(mdpath))

        if data.get('date'):
            rss_date_format = self.config.get('RSS_DATE_FORMAT')
            if rss_date_format:
                data['utc-date'] = self.convert_utc(
                    data['date'],
                    self.config['SOURCE_DATE_FORMAT'],
                    rss_date_format)
            data['display-date'] = self.convert_utc(
                data['date'],
                self.config['SOURCE_DATE_FORMAT'],
                self.config['DISPLAY_DATE_FORMAT'])

        data['content'] = content
        if temppath:
            lookup = TemplateLookup(directories=[os.getcwd()])
            temp = Template(filename=temppath, lookup=lookup)
            html = temp.render(data=data)
        else:
            html = content
        return data, html

    def convert_datetime(self, datestring, fin, fout):
        """Convert datetime string from old format to new format.

        Parameters:
            datestring (s): date time string.
            fin (s): input date string format.
            fout (s): output date string format.
        """
        return datetime.strptime(
            datestring, fin).strftime(fout)

    def convert_utc(self, datestring, fin, fout):
        """Convert datetime string to UTC.

        Parameters:
            datestring (s): date time string.
            fin (s): input date string format.
            fout (s): output date string format.

        Note:
            The input format string must use %z for offset (-0400) instead
            of %Z for abbreviation (EDT). The output format string can
            use either. Per Python datetime docs: "Changed in version 3.2:
            When the %z directive is provided to the strptime() method,
            an aware datetime object will be produced."
        """
        din = datetime.strptime(datestring, fin)
        utc = din.astimezone(timezone.utc)
        return utc.strftime(fout)

    def create_blog_archive(self, articles):
        """Create blog archive page.

        Parameters:
            articles (list): presorted newest to oldest.

        Data format:
            data[year] = [list of articles]
        """
        temppath = self.config['BLOG_ARCHIVE_TEMPLATE']
        if not temppath:
            return
        print('Creating blog archive page')
        datefmt = self.config['SOURCE_DATE_FORMAT']
        data = {}
        for article in articles:
            year = datetime.strptime(article['date'], datefmt).year
            if year in data:
                data[year].append(article)
            else:
                data[year] = [article]
        lookup = TemplateLookup(directories=[os.getcwd()])
        temp = Template(filename=temppath, lookup=lookup)
        html = temp.render(data=data)
        dest = os.path.join(self.blog_dist_dir, 'index.html')
        print(f'Creating {dest}')
        self.write(dest, html)

    def create_page(self, path):
        """Create HTML file.

        Parameters:
            path (s): markdown file path.

        Returns:
            Page data in dictionary.
        """
        print(f'Converting {path}')
        data, html = self.convert(path, args.template)
        os.makedirs(data['path'], exist_ok=True)
        dest = os.path.join(data['path'], data['slug'] + '.html')
        print(f'Creating {dest}')
        self.write(dest, html)
        return data

    def create_RSS_feed(self, articles):
        rss_temp = self.config.get('RSS_FEED_TEMPLATE')
        if not rss_temp:
            return
        print('Creating RSS feed')
        max_count = self.config.get('RSS_ARTICLE_COUNT')
        if max_count:
            articles = articles[:max_count]
        os.makedirs(self.blog_dist_dir, exist_ok=True)
        lookup = TemplateLookup(directories=[os.getcwd()])
        temp = Template(filename=rss_temp, lookup=lookup)
        xml = temp.render(now=self.now(), articles=articles)
        dest = os.path.join(self.blog_dist_dir, 'rss.xml')
        print(f'Creating {dest}')
        self.write(dest, xml)

    def now(self):
        """Return now timestamp in UTC."""
        fmt = self.config['RSS_DATE_FORMAT']
        return datetime.now(timezone.utc).strftime(fmt)

    def create_slug(self, metadata, filename):
        """Return metadata slug or file name."""
        slug = metadata.get('slug', '')
        if slug:
            return slug
        name, ext = os.path.splitext(os.path.basename(filename))
        return name

    def create_title(self, metadata, html):
        """Return metadata title or h1."""
        title = metadata.get('title', '')
        if title:
            return title
        h1start = html.find('<h1')
        h1stop = html.find('>', h1start)
        start = h1stop + 1
        end = html.find('</h1>')
        if start != -1 and end != -1:
            title = html[start:end]
        return title

    def normalize(self, d):
        """Return a copy of dict d, but without lists.

        The Python markdown meta extension creates a dictionary of
        lists.
        """
        return {key: d[key][0] for key in d}

    def read_config(self):
        path = args.config
        if path:
            with open(path, 'r') as f:
                d = json.load(f)
            self.config.update(d)

    def write(self, path, s):
        """Write string s to file only if the file string has changed."""
        olds = None
        if os.path.exists(path):
            with io.open(path, 'r', encoding='ISO-8859-1') as f:
                olds = f.read()
        if s != olds:
            with io.open(path, 'w', encoding='ISO-8859-1') as f:
                f.write(s)


if __name__ == "__main__":
    args = parser.parse_args()
    converter = Converter()
    converter.build()

