"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const yaml = require("yaml");
const __1 = require("..");
const deps_1 = require("../deps");
const logging = require("../logging");
const node_package_1 = require("../node-package");
const util_1 = require("./util");
logging.disable();
test('license file is added by default', () => {
    // WHEN
    const project = new TestNodeProject();
    // THEN
    expect(util_1.synthSnapshot(project).LICENSE).toContain('Apache License');
});
test('license file is not added if licensed is false', () => {
    // WHEN
    const project = new TestNodeProject({
        licensed: false,
    });
    // THEN
    const snapshot = util_1.synthSnapshot(project);
    expect(snapshot.LICENSE).toBeUndefined();
    expect(snapshot['.gitignore']).not.toContain('LICENSE');
    expect(snapshot['package.json'].license).toEqual('UNLICENSED');
});
describe('deps', () => {
    test('runtime deps', () => {
        // GIVEN
        const project = new TestNodeProject({
            deps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.RUNTIME);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.dependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        expect(pkgjson.peerDependencies).toStrictEqual({});
    });
    test('dev dependencies', () => {
        // GIVEN
        const project = new TestNodeProject({
            devDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addDevDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.TEST);
        project.deps.addDependency('eee@^1', deps_1.DependencyType.DEVENV);
        project.deps.addDependency('fff@^2', deps_1.DependencyType.BUILD);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.devDependencies.aaa).toStrictEqual('^1.2.3');
        expect(pkgjson.devDependencies.bbb).toStrictEqual('~4.5.6');
        expect(pkgjson.devDependencies.ccc).toStrictEqual('*');
        expect(pkgjson.devDependencies.ddd).toStrictEqual('*');
        expect(pkgjson.devDependencies.eee).toStrictEqual('^1');
        expect(pkgjson.devDependencies.fff).toStrictEqual('^2');
        expect(pkgjson.peerDependencies).toStrictEqual({});
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('peerDependencies', () => {
        // GIVEN
        const project = new TestNodeProject({
            peerDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addPeerDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.PEER);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.peerDependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        // devDependencies are added with pinned versions
        expect(pkgjson.devDependencies.aaa).toStrictEqual('1.2.3');
        expect(pkgjson.devDependencies.bbb).toStrictEqual('4.5.6');
        expect(pkgjson.devDependencies.ccc).toStrictEqual('*');
        expect(pkgjson.devDependencies.ddd).toStrictEqual('*');
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('peerDependencies without pinnedDevDep', () => {
        // GIVEN
        const project = new TestNodeProject({
            peerDependencyOptions: {
                pinnedDevDependency: false,
            },
            peerDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addPeerDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.PEER);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.peerDependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        // sanitize
        ['jest', 'jest-junit', 'projen', 'standard-version'].forEach(d => delete pkgjson.devDependencies[d]);
        expect(pkgjson.devDependencies).toStrictEqual({});
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('bundled deps are automatically added as normal deps', () => {
        // GIVEN
        const project = new TestNodeProject({
            bundledDeps: ['hey@2.1.1'],
        });
        // WHEN
        project.addBundledDeps('foo@^1.2.3');
        project.deps.addDependency('bar@~1.0.0', deps_1.DependencyType.BUNDLED);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.dependencies).toStrictEqual({
            hey: '2.1.1',
            foo: '^1.2.3',
            bar: '~1.0.0',
        });
        expect(pkgjson.bundledDependencies).toStrictEqual([
            'bar',
            'foo',
            'hey',
        ]);
    });
});
describe('npm publishing options', () => {
    test('defaults', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'my-package',
        });
        // THEN
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.PUBLIC);
        expect(npm.npmDistTag).toStrictEqual('latest');
        expect(npm.npmRegistry).toStrictEqual('registry.npmjs.org');
        expect(npm.npmRegistryUrl).toStrictEqual('https://registry.npmjs.org/');
        // since these are all defaults, publishConfig is not defined.
        expect(util_1.synthSnapshot(project)['package.json'].publishConfig).toBeUndefined();
    });
    test('scoped packages default to RESTRICTED access', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
        });
        // THEN
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.RESTRICTED);
        // since these are all defaults, publishConfig is not defined.
        expect(packageJson(project).publishConfig).toBeUndefined();
    });
    test('non-scoped package cannot be RESTRICTED', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // THEN
        expect(() => new node_package_1.NodePackage(project, {
            packageName: 'my-package',
            npmAccess: node_package_1.NpmAccess.RESTRICTED,
        })).toThrow(/"npmAccess" cannot be RESTRICTED for non-scoped npm package/);
    });
    test('custom settings', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
            npmDistTag: 'next',
            npmRegistryUrl: 'https://foo.bar',
            npmAccess: node_package_1.NpmAccess.PUBLIC,
        });
        // THEN
        expect(npm.npmDistTag).toStrictEqual('next');
        expect(npm.npmRegistry).toStrictEqual('foo.bar');
        expect(npm.npmRegistryUrl).toStrictEqual('https://foo.bar/');
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.PUBLIC);
        expect(packageJson(project).publishConfig).toStrictEqual({
            access: 'public',
            registry: 'https://foo.bar/',
            tag: 'next',
        });
    });
    test('deprecated npmRegistry can be used instead of npmRegistryUrl and then https:// is assumed', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
            npmRegistry: 'foo.bar.com',
        });
        // THEN
        expect(npm.npmRegistry).toStrictEqual('foo.bar.com');
        expect(npm.npmRegistryUrl).toStrictEqual('https://foo.bar.com/');
        expect(packageJson(project).publishConfig).toStrictEqual({
            registry: 'https://foo.bar.com/',
        });
    });
});
test('extend github release workflow', () => {
    var _a;
    const project = new TestNodeProject();
    (_a = project.releaseWorkflow) === null || _a === void 0 ? void 0 : _a.addJobs({
        publish_docker_hub: {
            'runs-on': 'ubuntu-latest',
            'env': {
                CI: 'true',
            },
            'steps': [
                {
                    name: 'Check out the repo',
                    uses: 'actions/checkout@v2',
                },
                {
                    name: 'Push to Docker Hub',
                    uses: 'docker/build-push-action@v1',
                    with: {
                        username: '${{ secrets.DOCKER_USERNAME }}',
                        password: '${{ secrets.DOCKER_PASSWORD }}',
                        repository: 'projen/projen-docker',
                        tag_with_ref: true,
                    },
                },
            ],
        },
    });
    const workflow = util_1.synthSnapshot(project)['.github/workflows/release.yml'];
    expect(workflow).toContain('publish_docker_hub:\n    runs-on: ubuntu-latest\n');
    expect(workflow).toContain('username: ${{ secrets.DOCKER_USERNAME }}\n          password: ${{ secrets.DOCKER_PASSWORD }}');
});
test('buildWorkflowMutable will push changes to PR branches', () => {
    // WHEN
    const project = new TestNodeProject({
        mutableBuild: true,
    });
    // THEN
    const workflowYaml = util_1.synthSnapshot(project)['.github/workflows/build.yml'];
    const workflow = yaml.parse(workflowYaml);
    expect(workflow.jobs.build.steps).toMatchSnapshot();
});
function packageJson(project) {
    return util_1.synthSnapshot(project)['package.json'];
}
class TestNodeProject extends __1.NodeProject {
    constructor(options = {}) {
        super({
            outdir: util_1.mkdtemp(),
            name: 'test-node-project',
            logging: {
                level: __1.LogLevel.OFF,
            },
            defaultReleaseBranch: 'master',
            ...options,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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