"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TaskRuntime = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
const util_1 = require("util");
const chalk = require("chalk");
const logging = require("../logging");
const tasks_1 = require("./tasks");
/**
 * (experimental) The runtime component of the tasks engine.
 *
 * @experimental
 */
class TaskRuntime {
    /**
     * @experimental
     */
    constructor(workdir) {
        this.workdir = path_1.resolve(workdir);
        const manifestPath = path_1.join(this.workdir, tasks_1.Tasks.MANIFEST_FILE);
        this.manifest = fs_1.existsSync(manifestPath)
            ? JSON.parse(fs_1.readFileSync(manifestPath, 'utf-8'))
            : { tasks: {} };
    }
    /**
     * (experimental) The tasks in this project.
     *
     * @experimental
     */
    get tasks() {
        var _b;
        return Object.values((_b = this.manifest.tasks) !== null && _b !== void 0 ? _b : {});
    }
    /**
     * (experimental) Find a task by name, or `undefined` if not found.
     *
     * @experimental
     */
    tryFindTask(name) {
        if (!this.manifest.tasks) {
            return undefined;
        }
        return this.manifest.tasks[name];
    }
    /**
     * (experimental) Runs the task.
     *
     * @param name The task name.
     * @experimental
     */
    runTask(name, parents = []) {
        const task = this.tryFindTask(name);
        if (!task) {
            throw new Error(`cannot find command ${task}`);
        }
        new RunTask(this, task, parents);
    }
}
exports.TaskRuntime = TaskRuntime;
_a = JSII_RTTI_SYMBOL_1;
TaskRuntime[_a] = { fqn: "projen.tasks.TaskRuntime", version: "0.17.92" };
class RunTask {
    constructor(runtime, task, parents = []) {
        var _b, _c, _d;
        this.runtime = runtime;
        this.task = task;
        this.env = {};
        this.workdir = (_b = task.cwd) !== null && _b !== void 0 ? _b : this.runtime.workdir;
        this.parents = parents;
        this.env = { ...process.env };
        this.env = this.resolveEnvironment();
        // evaluate condition
        if (!this.evalCondition(task)) {
            this.log('condition exited with non-zero - skipping');
            return;
        }
        for (const step of (_c = task.steps) !== null && _c !== void 0 ? _c : []) {
            if (step.say) {
                logging.info(this.fmtLog(step.say));
            }
            if (step.spawn) {
                this.runtime.runTask(step.spawn, [...this.parents, this.task.name]);
            }
            if (step.exec) {
                let command = '';
                let hasError = false;
                const cmd = step.exec.split(' ')[0];
                if (os_1.platform() == 'win32' && ['mkdir', 'mv', 'rm'].includes(cmd)) {
                    command = `shx ${step.exec}`;
                }
                else {
                    command = step.exec;
                }
                const cwd = step.cwd;
                try {
                    const result = this.shell({
                        command,
                        cwd,
                    });
                    hasError = result.status !== 0;
                }
                catch (e) {
                    // This is the error 'shx' will throw
                    if ((_d = e === null || e === void 0 ? void 0 : e.message) === null || _d === void 0 ? void 0 : _d.startsWith('non-zero exit code:')) {
                        hasError = true;
                    }
                    throw e;
                }
                if (hasError) {
                    throw new Error(`Task "${this.fullname}" failed when executing "${command}" (cwd: ${path_1.resolve(cwd !== null && cwd !== void 0 ? cwd : this.workdir)})`);
                }
            }
        }
    }
    /**
     * Determines if a task should be executed based on "condition".
     *
     * @returns true if the task should be executed or false if the condition
     * evaluates to false (exits with non-zero), indicating that the task should
     * be skipped.
     */
    evalCondition(task) {
        // no condition, carry on
        if (!task.condition) {
            return true;
        }
        this.log(`condition: ${task.condition}`);
        const result = this.shell({
            command: task.condition,
            logprefix: 'condition: ',
            quiet: true,
        });
        if (result.status === 0) {
            return true;
        }
        else {
            return false;
        }
    }
    /**
     * Renders the runtime environment for a task. Namely, it supports this syntax
     * `$(xx)` for allowing environment to be evaluated by executing a shell
     * command and obtaining its result.
     *
     * @param env The user-defined environment
     */
    resolveEnvironment() {
        var _b, _c, _d, _e;
        const env = {
            ...(_b = this.runtime.manifest.env) !== null && _b !== void 0 ? _b : {},
            ...(_c = this.task.env) !== null && _c !== void 0 ? _c : {},
        };
        const output = {
            ...process.env,
        };
        for (const [key, value] of Object.entries(env !== null && env !== void 0 ? env : {})) {
            if (value.startsWith('$(') && value.endsWith(')')) {
                const query = value.substring(2, value.length - 1);
                const result = this.shellEval({ command: query });
                if (result.status !== 0) {
                    const error = result.error
                        ? result.error.stack
                        : (_e = (_d = result.stderr) === null || _d === void 0 ? void 0 : _d.toString()) !== null && _e !== void 0 ? _e : 'unknown error';
                    throw new Error(`unable to evaluate environment variable ${key}=${value}: ${error}`);
                }
                output[key] = result.stdout.toString('utf-8').trim();
            }
            else {
                output[key] = value;
            }
        }
        return output;
    }
    /**
     * Returns the "full name" of the task which includes all it's parent task names concatenated by chevrons.
     */
    get fullname() {
        return [...this.parents, this.task.name].join(' » ');
    }
    log(...args) {
        logging.verbose(this.fmtLog(...args));
    }
    fmtLog(...args) {
        return util_1.format(`${chalk.underline(this.fullname)} |`, ...args);
    }
    shell(options) {
        var _b, _c;
        const quiet = (_b = options.quiet) !== null && _b !== void 0 ? _b : false;
        if (!quiet) {
            const log = new Array();
            if (options.logprefix) {
                log.push(options.logprefix);
            }
            log.push(options.command);
            if (options.cwd) {
                log.push(`(cwd: ${options.cwd})`);
            }
            this.log(log.join(' '));
        }
        const cwd = (_c = options.cwd) !== null && _c !== void 0 ? _c : this.workdir;
        if (!fs_1.existsSync(cwd) || !fs_1.statSync(cwd).isDirectory()) {
            throw new Error(`invalid workdir (cwd): ${cwd} must be an existing directory`);
        }
        return child_process_1.spawnSync(options.command, {
            ...options,
            cwd,
            shell: true,
            stdio: 'inherit',
            env: this.env,
            ...options.spawnOptions,
        });
    }
    shellEval(options) {
        return this.shell({
            quiet: true,
            ...options,
            spawnOptions: {
                stdio: ['inherit', 'pipe', 'inherit'],
            },
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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