"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isPropertyDifference = exports.ResourceDifference = exports.ResourceImpact = exports.ParameterDifference = exports.OutputDifference = exports.MetadataDifference = exports.MappingDifference = exports.ConditionDifference = exports.DifferenceCollection = exports.PropertyDifference = exports.Difference = exports.TemplateDiff = void 0;
const assert_1 = require("assert");
const cfnspec = require("@aws-cdk/cfnspec");
const iam_changes_1 = require("../iam/iam-changes");
const security_group_changes_1 = require("../network/security-group-changes");
const util_1 = require("./util");
/** Semantic differences between two CloudFormation templates. */
class TemplateDiff {
    constructor(args) {
        if (args.awsTemplateFormatVersion !== undefined) {
            this.awsTemplateFormatVersion = args.awsTemplateFormatVersion;
        }
        if (args.description !== undefined) {
            this.description = args.description;
        }
        if (args.transform !== undefined) {
            this.transform = args.transform;
        }
        this.conditions = args.conditions || new DifferenceCollection({});
        this.mappings = args.mappings || new DifferenceCollection({});
        this.metadata = args.metadata || new DifferenceCollection({});
        this.outputs = args.outputs || new DifferenceCollection({});
        this.parameters = args.parameters || new DifferenceCollection({});
        this.resources = args.resources || new DifferenceCollection({});
        this.unknown = args.unknown || new DifferenceCollection({});
        this.iamChanges = new iam_changes_1.IamChanges({
            propertyChanges: this.scrutinizablePropertyChanges(iam_changes_1.IamChanges.IamPropertyScrutinies),
            resourceChanges: this.scrutinizableResourceChanges(iam_changes_1.IamChanges.IamResourceScrutinies),
        });
        this.securityGroupChanges = new security_group_changes_1.SecurityGroupChanges({
            egressRulePropertyChanges: this.scrutinizablePropertyChanges([cfnspec.schema.PropertyScrutinyType.EgressRules]),
            ingressRulePropertyChanges: this.scrutinizablePropertyChanges([cfnspec.schema.PropertyScrutinyType.IngressRules]),
            egressRuleResourceChanges: this.scrutinizableResourceChanges([cfnspec.schema.ResourceScrutinyType.EgressRuleResource]),
            ingressRuleResourceChanges: this.scrutinizableResourceChanges([cfnspec.schema.ResourceScrutinyType.IngressRuleResource]),
        });
    }
    get differenceCount() {
        let count = 0;
        if (this.awsTemplateFormatVersion !== undefined) {
            count += 1;
        }
        if (this.description !== undefined) {
            count += 1;
        }
        if (this.transform !== undefined) {
            count += 1;
        }
        count += this.conditions.differenceCount;
        count += this.mappings.differenceCount;
        count += this.metadata.differenceCount;
        count += this.outputs.differenceCount;
        count += this.parameters.differenceCount;
        count += this.resources.differenceCount;
        count += this.unknown.differenceCount;
        return count;
    }
    get isEmpty() {
        return this.differenceCount === 0;
    }
    /**
     * Return true if any of the permissions objects involve a broadening of permissions
     */
    get permissionsBroadened() {
        return this.iamChanges.permissionsBroadened || this.securityGroupChanges.rulesAdded;
    }
    /**
     * Return true if any of the permissions objects have changed
     */
    get permissionsAnyChanges() {
        return this.iamChanges.hasChanges || this.securityGroupChanges.hasChanges;
    }
    /**
     * Return all property changes of a given scrutiny type
     *
     * We don't just look at property updates; we also look at resource additions and deletions (in which
     * case there is no further detail on property values), and resource type changes.
     */
    scrutinizablePropertyChanges(scrutinyTypes) {
        const ret = new Array();
        for (const [resourceLogicalId, resourceChange] of Object.entries(this.resources.changes)) {
            if (resourceChange.resourceTypeChanged) {
                // we ignore resource type changes here, and handle them in scrutinizableResourceChanges()
                continue;
            }
            const props = cfnspec.scrutinizablePropertyNames(resourceChange.newResourceType, scrutinyTypes);
            for (const propertyName of props) {
                ret.push({
                    resourceLogicalId,
                    propertyName,
                    resourceType: resourceChange.resourceType,
                    scrutinyType: cfnspec.propertySpecification(resourceChange.resourceType, propertyName).ScrutinyType,
                    oldValue: resourceChange.oldProperties && resourceChange.oldProperties[propertyName],
                    newValue: resourceChange.newProperties && resourceChange.newProperties[propertyName],
                });
            }
        }
        return ret;
    }
    /**
     * Return all resource changes of a given scrutiny type
     *
     * We don't just look at resource updates; we also look at resource additions and deletions (in which
     * case there is no further detail on property values), and resource type changes.
     */
    scrutinizableResourceChanges(scrutinyTypes) {
        const ret = new Array();
        const scrutinizableTypes = new Set(cfnspec.scrutinizableResourceTypes(scrutinyTypes));
        for (const [resourceLogicalId, resourceChange] of Object.entries(this.resources.changes)) {
            if (!resourceChange) {
                continue;
            }
            const commonProps = {
                oldProperties: resourceChange.oldProperties,
                newProperties: resourceChange.newProperties,
                resourceLogicalId,
            };
            // changes to the Type of resources can happen when migrating from CFN templates that use Transforms
            if (resourceChange.resourceTypeChanged) {
                // Treat as DELETE+ADD
                if (scrutinizableTypes.has(resourceChange.oldResourceType)) {
                    ret.push({
                        ...commonProps,
                        newProperties: undefined,
                        resourceType: resourceChange.oldResourceType,
                        scrutinyType: cfnspec.resourceSpecification(resourceChange.oldResourceType).ScrutinyType,
                    });
                }
                if (scrutinizableTypes.has(resourceChange.newResourceType)) {
                    ret.push({
                        ...commonProps,
                        oldProperties: undefined,
                        resourceType: resourceChange.newResourceType,
                        scrutinyType: cfnspec.resourceSpecification(resourceChange.newResourceType).ScrutinyType,
                    });
                }
            }
            else {
                if (scrutinizableTypes.has(resourceChange.resourceType)) {
                    ret.push({
                        ...commonProps,
                        resourceType: resourceChange.resourceType,
                        scrutinyType: cfnspec.resourceSpecification(resourceChange.resourceType).ScrutinyType,
                    });
                }
            }
        }
        return ret;
    }
}
exports.TemplateDiff = TemplateDiff;
/**
 * Models an entity that changed between two versions of a CloudFormation template.
 */
class Difference {
    /**
     * @param oldValue the old value, cannot be equal (to the sense of +deepEqual+) to +newValue+.
     * @param newValue the new value, cannot be equal (to the sense of +deepEqual+) to +oldValue+.
     */
    constructor(oldValue, newValue) {
        this.oldValue = oldValue;
        this.newValue = newValue;
        if (oldValue === undefined && newValue === undefined) {
            throw new assert_1.AssertionError({ message: 'oldValue and newValue are both undefined!' });
        }
        this.isDifferent = !util_1.deepEqual(oldValue, newValue);
    }
    /** @returns +true+ if the element is new to the template. */
    get isAddition() {
        return this.oldValue === undefined;
    }
    /** @returns +true+ if the element was removed from the template. */
    get isRemoval() {
        return this.newValue === undefined;
    }
    /** @returns +true+ if the element was already in the template and is updated. */
    get isUpdate() {
        return this.oldValue !== undefined
            && this.newValue !== undefined;
    }
}
exports.Difference = Difference;
class PropertyDifference extends Difference {
    constructor(oldValue, newValue, args) {
        super(oldValue, newValue);
        this.changeImpact = args.changeImpact;
    }
}
exports.PropertyDifference = PropertyDifference;
class DifferenceCollection {
    constructor(diffs) {
        this.diffs = diffs;
    }
    get changes() {
        return onlyChanges(this.diffs);
    }
    get differenceCount() {
        return Object.values(this.changes).length;
    }
    get(logicalId) {
        const ret = this.diffs[logicalId];
        if (!ret) {
            throw new Error(`No object with logical ID '${logicalId}'`);
        }
        return ret;
    }
    get logicalIds() {
        return Object.keys(this.changes);
    }
    /**
     * Returns a new TemplateDiff which only contains changes for which `predicate`
     * returns `true`.
     */
    filter(predicate) {
        const newChanges = {};
        for (const id of Object.keys(this.changes)) {
            const diff = this.changes[id];
            if (predicate(diff)) {
                newChanges[id] = diff;
            }
        }
        return new DifferenceCollection(newChanges);
    }
    /**
     * Invokes `cb` for all changes in this collection.
     *
     * Changes will be sorted as follows:
     *  - Removed
     *  - Added
     *  - Updated
     *  - Others
     *
     * @param cb
     */
    forEachDifference(cb) {
        const removed = new Array();
        const added = new Array();
        const updated = new Array();
        const others = new Array();
        for (const logicalId of this.logicalIds) {
            const change = this.changes[logicalId];
            if (change.isAddition) {
                added.push({ logicalId, change });
            }
            else if (change.isRemoval) {
                removed.push({ logicalId, change });
            }
            else if (change.isUpdate) {
                updated.push({ logicalId, change });
            }
            else if (change.isDifferent) {
                others.push({ logicalId, change });
            }
        }
        removed.forEach(v => cb(v.logicalId, v.change));
        added.forEach(v => cb(v.logicalId, v.change));
        updated.forEach(v => cb(v.logicalId, v.change));
        others.forEach(v => cb(v.logicalId, v.change));
    }
}
exports.DifferenceCollection = DifferenceCollection;
class ConditionDifference extends Difference {
}
exports.ConditionDifference = ConditionDifference;
class MappingDifference extends Difference {
}
exports.MappingDifference = MappingDifference;
class MetadataDifference extends Difference {
}
exports.MetadataDifference = MetadataDifference;
class OutputDifference extends Difference {
}
exports.OutputDifference = OutputDifference;
class ParameterDifference extends Difference {
}
exports.ParameterDifference = ParameterDifference;
var ResourceImpact;
(function (ResourceImpact) {
    /** The existing physical resource will be updated */
    ResourceImpact["WILL_UPDATE"] = "WILL_UPDATE";
    /** A new physical resource will be created */
    ResourceImpact["WILL_CREATE"] = "WILL_CREATE";
    /** The existing physical resource will be replaced */
    ResourceImpact["WILL_REPLACE"] = "WILL_REPLACE";
    /** The existing physical resource may be replaced */
    ResourceImpact["MAY_REPLACE"] = "MAY_REPLACE";
    /** The existing physical resource will be destroyed */
    ResourceImpact["WILL_DESTROY"] = "WILL_DESTROY";
    /** The existing physical resource will be removed from CloudFormation supervision */
    ResourceImpact["WILL_ORPHAN"] = "WILL_ORPHAN";
    /** There is no change in this resource */
    ResourceImpact["NO_CHANGE"] = "NO_CHANGE";
})(ResourceImpact = exports.ResourceImpact || (exports.ResourceImpact = {}));
/**
 * This function can be used as a reducer to obtain the resource-level impact of a list
 * of property-level impacts.
 * @param one the current worst impact so far.
 * @param two the new impact being considered (can be undefined, as we may not always be
 *      able to determine some peroperty's impact).
 */
function worstImpact(one, two) {
    if (!two) {
        return one;
    }
    const badness = {
        [ResourceImpact.NO_CHANGE]: 0,
        [ResourceImpact.WILL_UPDATE]: 1,
        [ResourceImpact.WILL_CREATE]: 2,
        [ResourceImpact.WILL_ORPHAN]: 3,
        [ResourceImpact.MAY_REPLACE]: 4,
        [ResourceImpact.WILL_REPLACE]: 5,
        [ResourceImpact.WILL_DESTROY]: 6,
    };
    return badness[one] > badness[two] ? one : two;
}
/**
 * Change to a single resource between two CloudFormation templates
 *
 * This class can be mutated after construction.
 */
class ResourceDifference {
    constructor(oldValue, newValue, args) {
        this.oldValue = oldValue;
        this.newValue = newValue;
        this.resourceTypes = args.resourceType;
        this.propertyDiffs = args.propertyDiffs;
        this.otherDiffs = args.otherDiffs;
        this.isAddition = oldValue === undefined;
        this.isRemoval = newValue === undefined;
    }
    get oldProperties() {
        return this.oldValue && this.oldValue.Properties;
    }
    get newProperties() {
        return this.newValue && this.newValue.Properties;
    }
    /**
     * Whether this resource was modified at all
     */
    get isDifferent() {
        return this.differenceCount > 0 || this.oldResourceType !== this.newResourceType;
    }
    /**
     * Whether the resource was updated in-place
     */
    get isUpdate() {
        return this.isDifferent && !this.isAddition && !this.isRemoval;
    }
    get oldResourceType() {
        return this.resourceTypes.oldType;
    }
    get newResourceType() {
        return this.resourceTypes.newType;
    }
    /**
     * All actual property updates
     */
    get propertyUpdates() {
        return onlyChanges(this.propertyDiffs);
    }
    /**
     * All actual "other" updates
     */
    get otherChanges() {
        return onlyChanges(this.otherDiffs);
    }
    /**
     * Return whether the resource type was changed in this diff
     *
     * This is not a valid operation in CloudFormation but to be defensive we're going
     * to be aware of it anyway.
     */
    get resourceTypeChanged() {
        return (this.resourceTypes.oldType !== undefined
            && this.resourceTypes.newType !== undefined
            && this.resourceTypes.oldType !== this.resourceTypes.newType);
    }
    /**
     * Return the resource type if it was unchanged
     *
     * If the resource type was changed, it's an error to call this.
     */
    get resourceType() {
        if (this.resourceTypeChanged) {
            throw new Error('Cannot get .resourceType, because the type was changed');
        }
        return this.resourceTypes.oldType || this.resourceTypes.newType;
    }
    /**
     * Replace a PropertyChange in this object
     *
     * This affects the property diff as it is summarized to users, but it DOES
     * NOT affect either the "oldValue" or "newValue" values; those still contain
     * the actual template values as provided by the user (they might still be
     * used for downstream processing).
     */
    setPropertyChange(propertyName, change) {
        this.propertyDiffs[propertyName] = change;
    }
    get changeImpact() {
        // Check the Type first
        if (this.resourceTypes.oldType !== this.resourceTypes.newType) {
            if (this.resourceTypes.oldType === undefined) {
                return ResourceImpact.WILL_CREATE;
            }
            if (this.resourceTypes.newType === undefined) {
                return this.oldValue.DeletionPolicy === 'Retain'
                    ? ResourceImpact.WILL_ORPHAN
                    : ResourceImpact.WILL_DESTROY;
            }
            return ResourceImpact.WILL_REPLACE;
        }
        // Base impact (before we mix in the worst of the property impacts);
        // WILL_UPDATE if we have "other" changes, NO_CHANGE if there are no "other" changes.
        const baseImpact = Object.keys(this.otherChanges).length > 0 ? ResourceImpact.WILL_UPDATE : ResourceImpact.NO_CHANGE;
        return Object.values(this.propertyDiffs)
            .map(elt => elt.changeImpact)
            .reduce(worstImpact, baseImpact);
    }
    /**
     * Count of actual differences (not of elements)
     */
    get differenceCount() {
        return Object.values(this.propertyUpdates).length
            + Object.values(this.otherChanges).length;
    }
    /**
     * Invoke a callback for each actual difference
     */
    forEachDifference(cb) {
        for (const key of Object.keys(this.propertyUpdates).sort()) {
            cb('Property', key, this.propertyUpdates[key]);
        }
        for (const key of Object.keys(this.otherChanges).sort()) {
            cb('Other', key, this.otherDiffs[key]);
        }
    }
}
exports.ResourceDifference = ResourceDifference;
function isPropertyDifference(diff) {
    return diff.changeImpact !== undefined;
}
exports.isPropertyDifference = isPropertyDifference;
/**
 * Filter a map of IDifferences down to only retain the actual changes
 */
function onlyChanges(xs) {
    const ret = {};
    for (const [key, diff] of Object.entries(xs)) {
        if (diff.isDifferent) {
            ret[key] = diff;
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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