"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.normalize = exports.massageSpec = void 0;
const lib_1 = require("../lib");
const scrutiny_1 = require("./scrutiny");
function massageSpec(spec) {
    scrutiny_1.detectScrutinyTypes(spec);
    replaceIncompleteTypes(spec);
    dropTypelessAttributes(spec);
}
exports.massageSpec = massageSpec;
/**
 * Fix incomplete type definitions in PropertyTypes
 *
 * Some user-defined types are defined to not have any properties, and not
 * be a collection of other types either. They have no definition at all.
 *
 * Add a property object type with empty properties.
 */
function replaceIncompleteTypes(spec) {
    for (const [name, definition] of Object.entries(spec.PropertyTypes)) {
        if (!lib_1.schema.isRecordType(definition)
            && !lib_1.schema.isCollectionProperty(definition)
            && !lib_1.schema.isScalarProperty(definition)
            && !lib_1.schema.isPrimitiveProperty(definition)) {
            // eslint-disable-next-line no-console
            console.log(`[${name}] Incomplete type, adding empty "Properties" field`);
            definition.Properties = {};
        }
    }
}
/**
 * Drop Attributes specified with the different ResourceTypes that have
 * no type specified.
 */
function dropTypelessAttributes(spec) {
    const resourceTypes = spec.ResourceTypes;
    Object.values(resourceTypes).forEach((resourceType) => {
        var _a;
        const attributes = (_a = resourceType.Attributes) !== null && _a !== void 0 ? _a : {};
        Object.keys(attributes).forEach((attrKey) => {
            const attrVal = attributes[attrKey];
            if (Object.keys(attrVal).length === 0) {
                delete attributes[attrKey];
            }
        });
    });
}
/**
 * Modifies the provided specification so that ``ResourceTypes`` and ``PropertyTypes`` are listed in alphabetical order.
 *
 * @param spec an AWS CloudFormation Resource Specification document.
 *
 * @returns ``spec``, after having sorted the ``ResourceTypes`` and ``PropertyTypes`` sections alphabetically.
 */
function normalize(spec) {
    spec.ResourceTypes = normalizeSection(spec.ResourceTypes);
    if (spec.PropertyTypes) {
        spec.PropertyTypes = normalizeSection(spec.PropertyTypes);
    }
    return spec;
    function normalizeSection(section) {
        const result = {};
        for (const key of Object.keys(section).sort()) {
            result[key] = section[key];
        }
        return result;
    }
}
exports.normalize = normalize;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWFzc2FnZS1zcGVjLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibWFzc2FnZS1zcGVjLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLGdDQUFnQztBQUNoQyx5Q0FBaUQ7QUFFakQsU0FBZ0IsV0FBVyxDQUFDLElBQTBCO0lBQ3BELDhCQUFtQixDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzFCLHNCQUFzQixDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzdCLHNCQUFzQixDQUFDLElBQUksQ0FBQyxDQUFDO0FBQy9CLENBQUM7QUFKRCxrQ0FJQztBQUVEOzs7Ozs7O0dBT0c7QUFDSCxTQUFTLHNCQUFzQixDQUFDLElBQTBCO0lBQ3hELEtBQUssTUFBTSxDQUFDLElBQUksRUFBRSxVQUFVLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRTtRQUNuRSxJQUFJLENBQUMsWUFBTSxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUM7ZUFDakMsQ0FBQyxZQUFNLENBQUMsb0JBQW9CLENBQUMsVUFBVSxDQUFDO2VBQ3hDLENBQUMsWUFBTSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsQ0FBQztlQUNwQyxDQUFDLFlBQU0sQ0FBQyxtQkFBbUIsQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUMxQyxzQ0FBc0M7WUFDdEMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxJQUFJLElBQUksb0RBQW9ELENBQUMsQ0FBQztZQUV6RSxVQUErQyxDQUFDLFVBQVUsR0FBRyxFQUFFLENBQUM7U0FDbEU7S0FDRjtBQUNILENBQUM7QUFFRDs7O0dBR0c7QUFDSCxTQUFTLHNCQUFzQixDQUFDLElBQTBCO0lBQ3hELE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUM7SUFDekMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxZQUFZLEVBQUUsRUFBRTs7UUFDcEQsTUFBTSxVQUFVLFNBQUcsWUFBWSxDQUFDLFVBQVUsbUNBQUksRUFBRSxDQUFDO1FBQ2pELE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxFQUFFLEVBQUU7WUFDMUMsTUFBTSxPQUFPLEdBQUcsVUFBVSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQ3BDLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO2dCQUNyQyxPQUFPLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQzthQUM1QjtRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDO0FBRUQ7Ozs7OztHQU1HO0FBQ0gsU0FBZ0IsU0FBUyxDQUFDLElBQTBCO0lBQ2xELElBQUksQ0FBQyxhQUFhLEdBQUcsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQzFELElBQUksSUFBSSxDQUFDLGFBQWEsRUFBRTtRQUN0QixJQUFJLENBQUMsYUFBYSxHQUFHLGdCQUFnQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztLQUMzRDtJQUNELE9BQU8sSUFBSSxDQUFDO0lBRVosU0FBUyxnQkFBZ0IsQ0FBSSxPQUE4QjtRQUN6RCxNQUFNLE1BQU0sR0FBMEIsRUFBRSxDQUFDO1FBQ3pDLEtBQUssTUFBTSxHQUFHLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRTtZQUM3QyxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQzVCO1FBQ0QsT0FBTyxNQUFNLENBQUM7SUFDaEIsQ0FBQztBQUNILENBQUM7QUFkRCw4QkFjQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IHNjaGVtYSB9IGZyb20gJy4uL2xpYic7XG5pbXBvcnQgeyBkZXRlY3RTY3J1dGlueVR5cGVzIH0gZnJvbSAnLi9zY3J1dGlueSc7XG5cbmV4cG9ydCBmdW5jdGlvbiBtYXNzYWdlU3BlYyhzcGVjOiBzY2hlbWEuU3BlY2lmaWNhdGlvbikge1xuICBkZXRlY3RTY3J1dGlueVR5cGVzKHNwZWMpO1xuICByZXBsYWNlSW5jb21wbGV0ZVR5cGVzKHNwZWMpO1xuICBkcm9wVHlwZWxlc3NBdHRyaWJ1dGVzKHNwZWMpO1xufVxuXG4vKipcbiAqIEZpeCBpbmNvbXBsZXRlIHR5cGUgZGVmaW5pdGlvbnMgaW4gUHJvcGVydHlUeXBlc1xuICpcbiAqIFNvbWUgdXNlci1kZWZpbmVkIHR5cGVzIGFyZSBkZWZpbmVkIHRvIG5vdCBoYXZlIGFueSBwcm9wZXJ0aWVzLCBhbmQgbm90XG4gKiBiZSBhIGNvbGxlY3Rpb24gb2Ygb3RoZXIgdHlwZXMgZWl0aGVyLiBUaGV5IGhhdmUgbm8gZGVmaW5pdGlvbiBhdCBhbGwuXG4gKlxuICogQWRkIGEgcHJvcGVydHkgb2JqZWN0IHR5cGUgd2l0aCBlbXB0eSBwcm9wZXJ0aWVzLlxuICovXG5mdW5jdGlvbiByZXBsYWNlSW5jb21wbGV0ZVR5cGVzKHNwZWM6IHNjaGVtYS5TcGVjaWZpY2F0aW9uKSB7XG4gIGZvciAoY29uc3QgW25hbWUsIGRlZmluaXRpb25dIG9mIE9iamVjdC5lbnRyaWVzKHNwZWMuUHJvcGVydHlUeXBlcykpIHtcbiAgICBpZiAoIXNjaGVtYS5pc1JlY29yZFR5cGUoZGVmaW5pdGlvbilcbiAgICAmJiAhc2NoZW1hLmlzQ29sbGVjdGlvblByb3BlcnR5KGRlZmluaXRpb24pXG4gICAgJiYgIXNjaGVtYS5pc1NjYWxhclByb3BlcnR5KGRlZmluaXRpb24pXG4gICAgJiYgIXNjaGVtYS5pc1ByaW1pdGl2ZVByb3BlcnR5KGRlZmluaXRpb24pKSB7XG4gICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgbm8tY29uc29sZVxuICAgICAgY29uc29sZS5sb2coYFske25hbWV9XSBJbmNvbXBsZXRlIHR5cGUsIGFkZGluZyBlbXB0eSBcIlByb3BlcnRpZXNcIiBmaWVsZGApO1xuXG4gICAgICAoZGVmaW5pdGlvbiBhcyB1bmtub3duIGFzIHNjaGVtYS5SZWNvcmRQcm9wZXJ0eSkuUHJvcGVydGllcyA9IHt9O1xuICAgIH1cbiAgfVxufVxuXG4vKipcbiAqIERyb3AgQXR0cmlidXRlcyBzcGVjaWZpZWQgd2l0aCB0aGUgZGlmZmVyZW50IFJlc291cmNlVHlwZXMgdGhhdCBoYXZlXG4gKiBubyB0eXBlIHNwZWNpZmllZC5cbiAqL1xuZnVuY3Rpb24gZHJvcFR5cGVsZXNzQXR0cmlidXRlcyhzcGVjOiBzY2hlbWEuU3BlY2lmaWNhdGlvbikge1xuICBjb25zdCByZXNvdXJjZVR5cGVzID0gc3BlYy5SZXNvdXJjZVR5cGVzO1xuICBPYmplY3QudmFsdWVzKHJlc291cmNlVHlwZXMpLmZvckVhY2goKHJlc291cmNlVHlwZSkgPT4ge1xuICAgIGNvbnN0IGF0dHJpYnV0ZXMgPSByZXNvdXJjZVR5cGUuQXR0cmlidXRlcyA/PyB7fTtcbiAgICBPYmplY3Qua2V5cyhhdHRyaWJ1dGVzKS5mb3JFYWNoKChhdHRyS2V5KSA9PiB7XG4gICAgICBjb25zdCBhdHRyVmFsID0gYXR0cmlidXRlc1thdHRyS2V5XTtcbiAgICAgIGlmIChPYmplY3Qua2V5cyhhdHRyVmFsKS5sZW5ndGggPT09IDApIHtcbiAgICAgICAgZGVsZXRlIGF0dHJpYnV0ZXNbYXR0cktleV07XG4gICAgICB9XG4gICAgfSk7XG4gIH0pO1xufVxuXG4vKipcbiAqIE1vZGlmaWVzIHRoZSBwcm92aWRlZCBzcGVjaWZpY2F0aW9uIHNvIHRoYXQgYGBSZXNvdXJjZVR5cGVzYGAgYW5kIGBgUHJvcGVydHlUeXBlc2BgIGFyZSBsaXN0ZWQgaW4gYWxwaGFiZXRpY2FsIG9yZGVyLlxuICpcbiAqIEBwYXJhbSBzcGVjIGFuIEFXUyBDbG91ZEZvcm1hdGlvbiBSZXNvdXJjZSBTcGVjaWZpY2F0aW9uIGRvY3VtZW50LlxuICpcbiAqIEByZXR1cm5zIGBgc3BlY2BgLCBhZnRlciBoYXZpbmcgc29ydGVkIHRoZSBgYFJlc291cmNlVHlwZXNgYCBhbmQgYGBQcm9wZXJ0eVR5cGVzYGAgc2VjdGlvbnMgYWxwaGFiZXRpY2FsbHkuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBub3JtYWxpemUoc3BlYzogc2NoZW1hLlNwZWNpZmljYXRpb24pOiBzY2hlbWEuU3BlY2lmaWNhdGlvbiB7XG4gIHNwZWMuUmVzb3VyY2VUeXBlcyA9IG5vcm1hbGl6ZVNlY3Rpb24oc3BlYy5SZXNvdXJjZVR5cGVzKTtcbiAgaWYgKHNwZWMuUHJvcGVydHlUeXBlcykge1xuICAgIHNwZWMuUHJvcGVydHlUeXBlcyA9IG5vcm1hbGl6ZVNlY3Rpb24oc3BlYy5Qcm9wZXJ0eVR5cGVzKTtcbiAgfVxuICByZXR1cm4gc3BlYztcblxuICBmdW5jdGlvbiBub3JtYWxpemVTZWN0aW9uPFQ+KHNlY3Rpb246IHsgW25hbWU6IHN0cmluZ106IFQgfSk6IHsgW25hbWU6IHN0cmluZ106IFQgfSB7XG4gICAgY29uc3QgcmVzdWx0OiB7IFtuYW1lOiBzdHJpbmddOiBUIH0gPSB7fTtcbiAgICBmb3IgKGNvbnN0IGtleSBvZiBPYmplY3Qua2V5cyhzZWN0aW9uKS5zb3J0KCkpIHtcbiAgICAgIHJlc3VsdFtrZXldID0gc2VjdGlvbltrZXldO1xuICAgIH1cbiAgICByZXR1cm4gcmVzdWx0O1xuICB9XG59XG4iXX0=