"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceProvider = exports.CustomResourceProviderRuntime = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const asset_staging_1 = require("../asset-staging");
const assets_1 = require("../assets");
const cfn_resource_1 = require("../cfn-resource");
const duration_1 = require("../duration");
const size_1 = require("../size");
const stack_1 = require("../stack");
const token_1 = require("../token");
const ENTRYPOINT_FILENAME = '__entrypoint__';
const ENTRYPOINT_NODEJS_SOURCE = path.join(__dirname, 'nodejs-entrypoint.js');
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("../construct-compat");
/**
 * The lambda runtime to use for the resource provider. This also indicates
 * which language is used for the handler.
 */
var CustomResourceProviderRuntime;
(function (CustomResourceProviderRuntime) {
    /**
     * Node.js 12.x
     *
     * @deprecated Use {@link NODEJS_14_X}
     */
    CustomResourceProviderRuntime["NODEJS_12"] = "deprecated_nodejs12.x";
    /**
     * Node.js 12.x
     */
    CustomResourceProviderRuntime["NODEJS_12_X"] = "nodejs12.x";
    /**
     * Node.js 14.x
     */
    CustomResourceProviderRuntime["NODEJS_14_X"] = "nodejs14.x";
    /**
     * Node.js 16.x
     */
    CustomResourceProviderRuntime["NODEJS_16_X"] = "nodejs16.x";
})(CustomResourceProviderRuntime = exports.CustomResourceProviderRuntime || (exports.CustomResourceProviderRuntime = {}));
/**
 * An AWS-Lambda backed custom resource provider, for CDK Construct Library constructs
 *
 * This is a provider for `CustomResource` constructs, backed by an AWS Lambda
 * Function. It only supports NodeJS runtimes.
 *
 * **This is not a generic custom resource provider class**. It is specifically
 * intended to be used only by constructs in the AWS CDK Construct Library, and
 * only exists here because of reverse dependency issues (for example, it cannot
 * use `iam.PolicyStatement` objects, since the `iam` library already depends on
 * the CDK `core` library and we cannot have cyclic dependencies).
 *
 * If you are not writing constructs for the AWS Construct Library, you should
 * use the `Provider` class in the `custom-resources` module instead, which has
 * a better API and supports all Lambda runtimes, not just Node.
 *
 * N.B.: When you are writing Custom Resource Providers, there are a number of
 * lifecycle events you have to pay attention to. These are documented in the
 * README of the `custom-resources` module. Be sure to give the documentation
 * in that module a read, regardless of whether you end up using the Provider
 * class in there or this one.
 */
class CustomResourceProvider extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_core_CustomResourceProviderProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CustomResourceProvider);
            }
            throw error;
        }
        const stack = stack_1.Stack.of(scope);
        // copy the entry point to the code directory
        fs.copyFileSync(ENTRYPOINT_NODEJS_SOURCE, path.join(props.codeDirectory, `${ENTRYPOINT_FILENAME}.js`));
        // verify we have an index file there
        if (!fs.existsSync(path.join(props.codeDirectory, 'index.js'))) {
            throw new Error(`cannot find ${props.codeDirectory}/index.js`);
        }
        const staging = new asset_staging_1.AssetStaging(this, 'Staging', {
            sourcePath: props.codeDirectory,
        });
        const assetFileName = staging.relativeStagedPath(stack);
        const asset = stack.synthesizer.addFileAsset({
            fileName: assetFileName,
            sourceHash: staging.assetHash,
            packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY,
        });
        const policies = !props.policyStatements ? undefined : [
            {
                PolicyName: 'Inline',
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: props.policyStatements,
                },
            },
        ];
        const role = new cfn_resource_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'lambda.amazonaws.com' } }],
                },
                ManagedPolicyArns: [
                    { 'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole' },
                ],
                Policies: policies,
            },
        });
        this.roleArn = token_1.Token.asString(role.getAtt('Arn'));
        const timeout = props.timeout ?? duration_1.Duration.minutes(15);
        const memory = props.memorySize ?? size_1.Size.mebibytes(128);
        const handler = new cfn_resource_1.CfnResource(this, 'Handler', {
            type: 'AWS::Lambda::Function',
            properties: {
                Code: {
                    S3Bucket: asset.bucketName,
                    S3Key: asset.objectKey,
                },
                Timeout: timeout.toSeconds(),
                MemorySize: memory.toMebibytes(),
                Handler: `${ENTRYPOINT_FILENAME}.handler`,
                Role: role.getAtt('Arn'),
                Runtime: customResourceProviderRuntimeToString(props.runtime),
                Environment: this.renderEnvironmentVariables(props.environment),
                Description: props.description ?? undefined,
            },
        });
        handler.addDependsOn(role);
        if (this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            handler.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PATH_KEY, assetFileName);
            handler.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY, 'Code');
        }
        this.serviceToken = token_1.Token.asString(handler.getAtt('Arn'));
        this.codeHash = staging.assetHash;
    }
    /**
     * Returns a stack-level singleton ARN (service token) for the custom resource
     * provider.
     *
     * @param scope Construct scope
     * @param uniqueid A globally unique id that will be used for the stack-level
     * construct.
     * @param props Provider properties which will only be applied when the
     * provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     */
    static getOrCreate(scope, uniqueid, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_CustomResourceProviderProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getOrCreate);
            }
            throw error;
        }
        return this.getOrCreateProvider(scope, uniqueid, props).serviceToken;
    }
    /**
     * Returns a stack-level singleton for the custom resource provider.
     *
     * @param scope Construct scope
     * @param uniqueid A globally unique id that will be used for the stack-level
     * construct.
     * @param props Provider properties which will only be applied when the
     * provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     */
    static getOrCreateProvider(scope, uniqueid, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_CustomResourceProviderProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getOrCreateProvider);
            }
            throw error;
        }
        const id = `${uniqueid}CustomResourceProvider`;
        const stack = stack_1.Stack.of(scope);
        const provider = stack.node.tryFindChild(id)
            ?? new CustomResourceProvider(stack, id, props);
        return provider;
    }
    renderEnvironmentVariables(env) {
        if (!env || Object.keys(env).length === 0) {
            return undefined;
        }
        // Sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it)
        const variables = {};
        const keys = Object.keys(env).sort();
        for (const key of keys) {
            variables[key] = env[key];
        }
        return { Variables: variables };
    }
}
exports.CustomResourceProvider = CustomResourceProvider;
_a = JSII_RTTI_SYMBOL_1;
CustomResourceProvider[_a] = { fqn: "@aws-cdk/core.CustomResourceProvider", version: "1.186.1" };
function customResourceProviderRuntimeToString(x) {
    switch (x) {
        case CustomResourceProviderRuntime.NODEJS_12:
        case CustomResourceProviderRuntime.NODEJS_12_X:
            return 'nodejs12.x';
        case CustomResourceProviderRuntime.NODEJS_14_X:
            return 'nodejs14.x';
        case CustomResourceProviderRuntime.NODEJS_16_X:
            return 'nodejs16.x';
    }
}
//# sourceMappingURL=data:application/json;base64,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