"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.writeSorted = exports.applyAndWrite = exports.applyPatchSet = exports.evaluatePatchSet = exports.loadPatchSet = void 0;
/**
 * Apply a JSON patch set into the given target file
 *
 * The sources can be taken from one or more directories.
 */
const path = require("path");
const fastJsonPatch = require("fast-json-patch");
const fs = require("fs-extra");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const sortJson = require('sort-json');
async function loadPatchSet(sourceDirectory, relativeTo = process.cwd()) {
    const ret = {};
    const files = await fs.readdir(sourceDirectory);
    for (const file of files) {
        const fullFile = path.join(sourceDirectory, file);
        const relName = path.relative(relativeTo, fullFile);
        if ((await fs.stat(fullFile)).isDirectory()) {
            ret[relName] = {
                type: 'set',
                sources: await loadPatchSet(fullFile, sourceDirectory),
            };
        }
        else if (file.endsWith('.json')) {
            ret[relName] = {
                type: file.indexOf('patch') === -1 ? 'fragment' : 'patch',
                data: await fs.readJson(fullFile),
            };
        }
    }
    return ret;
}
exports.loadPatchSet = loadPatchSet;
function evaluatePatchSet(sources, options = {}) {
    const targetObject = {};
    for (const key of Object.keys(sources).sort()) {
        const value = sources[key];
        switch (value.type) {
            case 'fragment':
                log(key);
                merge(targetObject, value.data, []);
                break;
            case 'patch':
                patch(targetObject, value.data, (m) => log(`${key}: ${m}`));
                break;
            case 'set':
                const evaluated = evaluatePatchSet(value.sources, options);
                log(key);
                merge(targetObject, evaluated, []);
                break;
        }
    }
    return targetObject;
    function log(x) {
        if (!options.quiet) {
            // eslint-disable-next-line no-console
            console.log(x);
        }
    }
}
exports.evaluatePatchSet = evaluatePatchSet;
/**
 * Load a patch set from a directory
 */
async function applyPatchSet(sourceDirectory, options = {}) {
    const patches = await loadPatchSet(sourceDirectory);
    return evaluatePatchSet(patches, options);
}
exports.applyPatchSet = applyPatchSet;
/**
 * Load a patch set and write it out to a file
 */
async function applyAndWrite(targetFile, sourceDirectory, options = {}) {
    const model = await applyPatchSet(sourceDirectory, options);
    await writeSorted(targetFile, model);
}
exports.applyAndWrite = applyAndWrite;
async function writeSorted(targetFile, data) {
    await fs.mkdirp(path.dirname(targetFile));
    await fs.writeJson(targetFile, sortJson(data), { spaces: 2 });
}
exports.writeSorted = writeSorted;
function printSorted(data) {
    process.stdout.write(JSON.stringify(sortJson(data), undefined, 2));
}
function merge(target, fragment, jsonPath) {
    if (!fragment) {
        return;
    }
    if (!target || typeof target !== 'object' || Array.isArray(target)) {
        throw new Error(`Expected object, found: '${target}' at '$.${jsonPath.join('.')}'`);
    }
    for (const key of Object.keys(fragment)) {
        if (key.startsWith('$')) {
            continue;
        }
        if (key in target) {
            const specVal = target[key];
            const fragVal = fragment[key];
            if (typeof specVal !== typeof fragVal) {
                // eslint-disable-next-line max-len
                throw new Error(`Attempted to merge ${JSON.stringify(fragVal)} into incompatible ${JSON.stringify(specVal)} at path ${jsonPath.join('/')}/${key}`);
            }
            if (specVal == fragVal) {
                continue;
            }
            if (typeof specVal !== 'object') {
                // eslint-disable-next-line max-len
                throw new Error(`Conflict when attempting to merge ${JSON.stringify(fragVal)} into ${JSON.stringify(specVal)} at path ${jsonPath.join('/')}/${key}`);
            }
            merge(specVal, fragVal, [...jsonPath, key]);
        }
        else {
            target[key] = fragment[key];
        }
    }
}
function patch(target, fragment, log) {
    var _a;
    if (!fragment) {
        return;
    }
    const patches = findPatches(target, fragment);
    for (const p of patches) {
        log((_a = p.description) !== null && _a !== void 0 ? _a : '');
        try {
            fastJsonPatch.applyPatch(target, p.operations);
        }
        catch (e) {
            throw new Error(`error applying patch: ${JSON.stringify(p, undefined, 2)}: ${e.message}`);
        }
    }
}
/**
 * Find the sets of patches to apply in a document
 *
 * Adjusts paths to be root-relative, which makes it possible to have paths
 * point outside the patch scope.
 */
function findPatches(data, patchSource) {
    const ret = [];
    recurse(data, patchSource, []);
    return ret;
    function recurse(actualData, fragment, jsonPath) {
        if (!fragment) {
            return;
        }
        if ('patch' in fragment) {
            const p = fragment.patch;
            if (!p.operations) {
                throw new Error(`Patch needs 'operations' key, got: ${JSON.stringify(p)}`);
            }
            ret.push({
                description: p.description,
                operations: p.operations.map((op) => adjustPaths(op, jsonPath)),
            });
        }
        else if ('patch:each' in fragment) {
            const p = fragment['patch:each'];
            if (typeof actualData !== 'object') {
                throw new Error(`Patch ${p.description}: expecting object in data, found '${actualData}'`);
            }
            if (!p.operations) {
                throw new Error(`Patch needs 'operations' key, got: ${JSON.stringify(p)}`);
            }
            for (const key in actualData) {
                ret.push({
                    description: `${key}: ${p.description}`,
                    operations: p.operations.map((op) => adjustPaths(op, [...jsonPath, key])),
                });
            }
        }
        else {
            for (const key of Object.keys(fragment)) {
                if (!(key in actualData)) {
                    actualData[key] = {};
                }
                recurse(actualData[key], fragment[key], [...jsonPath, key]);
            }
        }
    }
    function adjustPaths(op, jsonPath) {
        return {
            ...op,
            ...op.path ? { path: adjustPath(op.path, jsonPath) } : undefined,
            ...op.from ? { from: adjustPath(op.from, jsonPath) } : undefined,
        };
    }
    /**
     * Adjust path
     *
     * '$/' means from the root, otherwise interpret as relative path.
     */
    function adjustPath(originalPath, jsonPath) {
        if (typeof originalPath !== 'string') {
            throw new Error(`adjustPath: expected string, got ${JSON.stringify(originalPath)}`);
        }
        if (originalPath.startsWith('$/')) {
            return originalPath.slice(1);
        }
        return jsonPath.map(p => `/${p}`).join('') + originalPath;
    }
}
/**
 * Run this file as a CLI tool, to apply a patch set from the command line
 */
async function main(args) {
    const quiet = eatArg('-q', args) || eatArg('--quiet', args);
    if (args.length < 1) {
        throw new Error('Usage: patch-set <DIR> [<FILE>]');
    }
    const [dir, targetFile] = args;
    const model = await applyPatchSet(dir, { quiet });
    if (targetFile) {
        await writeSorted(targetFile, model);
    }
    else {
        printSorted(model);
    }
}
function eatArg(arg, args) {
    for (let i = 0; i < args.length; i++) {
        if (args[i] === arg) {
            args.splice(i, 1);
            return true;
        }
    }
    return false;
}
if (require.main === module) {
    main(process.argv.slice(2)).catch(e => {
        process.exitCode = 1;
        // eslint-disable-next-line no-console
        console.error(e.message);
    });
}
//# sourceMappingURL=data:application/json;base64,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