"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Annotations = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const constructs_1 = require("constructs");
const DEPRECATIONS_SYMBOL = Symbol.for('@aws-cdk/core.deprecations');
/**
 * Includes API for attaching annotations such as warning messages to constructs.
 */
class Annotations {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the annotations API for a construct scope.
     * @param scope The scope
     */
    static of(scope) {
        return new Annotations(scope);
    }
    /**
     * Adds a warning metadata entry to this construct.
     *
     * The CLI will display the warning when an app is synthesized, or fail if run
     * in --strict mode.
     *
     * @param message The warning message.
     */
    addWarning(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.WARN, message);
    }
    /**
     * Adds an info metadata entry to this construct.
     *
     * The CLI will display the info message when apps are synthesized.
     *
     * @param message The info message.
     */
    addInfo(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.INFO, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     * The toolkit will fail deployment of any stack that has errors reported against it.
     * @param message The error message.
     */
    addError(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.ERROR, message);
    }
    /**
     * Adds a deprecation warning for a specific API.
     *
     * Deprecations will be added only once per construct as a warning and will be
     * deduplicated based on the `api`.
     *
     * If the environment variable `CDK_BLOCK_DEPRECATIONS` is set, this method
     * will throw an error instead with the deprecation message.
     *
     * @param api The API being deprecated in the format `module.Class.property`
     * (e.g. `@aws-cdk/core.Construct.node`).
     * @param message The deprecation message to display, with information about
     * alternatives.
     */
    addDeprecation(api, message) {
        const text = `The API ${api} is deprecated: ${message}. This API will be removed in the next major release`;
        // throw if CDK_BLOCK_DEPRECATIONS is set
        if (process.env.CDK_BLOCK_DEPRECATIONS) {
            throw new Error(`${constructs_1.Node.of(this.scope).path}: ${text}`);
        }
        // de-dup based on api key
        const set = this.deprecationsReported;
        if (set.has(api)) {
            return;
        }
        this.addWarning(text);
        set.add(api);
    }
    /**
     * Adds a message metadata entry to the construct node, to be displayed by the CDK CLI.
     * @param level The message level
     * @param message The message itself
     */
    addMessage(level, message) {
        constructs_1.Node.of(this.scope).addMetadata(level, message);
    }
    /**
     * Returns the set of deprecations reported on this construct.
     */
    get deprecationsReported() {
        let set = this.scope[DEPRECATIONS_SYMBOL];
        if (!set) {
            set = new Set();
            Object.defineProperty(this.scope, DEPRECATIONS_SYMBOL, { value: set });
        }
        return set;
    }
}
exports.Annotations = Annotations;
_a = JSII_RTTI_SYMBOL_1;
Annotations[_a] = { fqn: "@aws-cdk/core.Annotations", version: "1.181.1" };
//# sourceMappingURL=data:application/json;base64,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