"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParameter = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cfn_element_1 = require("./cfn-element");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * A CloudFormation parameter.
 *
 * Use the optional Parameters section to customize your templates.
 * Parameters enable you to input custom values to your template each time you create or
 * update a stack.
 */
class CfnParameter extends cfn_element_1.CfnElement {
    /**
     * Creates a parameter construct.
     * Note that the name (logical ID) of the parameter will derive from it's `coname` and location
     * within the stack. Therefore, it is recommended that parameters are defined at the stack level.
     *
     * @param scope The parent construct.
     * @param props The parameter properties.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_core_CfnParameterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CfnParameter);
            }
            throw error;
        }
        this._type = props.type || 'String';
        this._default = props.default;
        this._allowedPattern = props.allowedPattern;
        this._allowedValues = props.allowedValues;
        this._constraintDescription = props.constraintDescription;
        this._description = props.description;
        this._maxLength = props.maxLength;
        this._maxValue = props.maxValue;
        this._minLength = props.minLength;
        this._minValue = props.minValue;
        this._noEcho = props.noEcho;
    }
    /**
     * The data type for the parameter (DataType).
     *
     * @default String
     */
    get type() {
        return this._type;
    }
    set type(type) {
        this._type = type;
    }
    /**
     * A value of the appropriate type for the template to use if no value is specified
     * when a stack is created. If you define constraints for the parameter, you must specify
     * a value that adheres to those constraints.
     *
     * @default - No default value for parameter.
     */
    get default() {
        return this._default;
    }
    set default(value) {
        this._default = value;
    }
    /**
     * A regular expression that represents the patterns to allow for String types.
     *
     * @default - No constraints on patterns allowed for parameter.
     */
    get allowedPattern() {
        return this._allowedPattern;
    }
    set allowedPattern(pattern) {
        this._allowedPattern = pattern;
    }
    /**
     * An array containing the list of values allowed for the parameter.
     *
     * @default - No constraints on values allowed for parameter.
     */
    get allowedValues() {
        return this._allowedValues;
    }
    set allowedValues(values) {
        this._allowedValues = values;
    }
    /**
     * A string that explains a constraint when the constraint is violated.
     * For example, without a constraint description, a parameter that has an allowed
     * pattern of [A-Za-z0-9]+ displays the following error message when the user specifies
     * an invalid value:
     *
     * @default - No description with customized error message when user specifies invalid values.
     */
    get constraintDescription() {
        return this._constraintDescription;
    }
    set constraintDescription(desc) {
        this._constraintDescription = desc;
    }
    /**
     * A string of up to 4000 characters that describes the parameter.
     *
     * @default - No description for the parameter.
     */
    get description() {
        return this._description;
    }
    set description(desc) {
        this._description = desc;
    }
    /**
     * An integer value that determines the largest number of characters you want to allow for String types.
     *
     * @default - None.
     */
    get maxLength() {
        return this._maxLength;
    }
    set maxLength(len) {
        this._maxLength = len;
    }
    /**
     * An integer value that determines the smallest number of characters you want to allow for String types.
     *
     * @default - None.
     */
    get minLength() {
        return this._minLength;
    }
    set minLength(len) {
        this._minLength = len;
    }
    /**
     * A numeric value that determines the largest numeric value you want to allow for Number types.
     *
     * @default - None.
     */
    get maxValue() {
        return this._maxValue;
    }
    set maxValue(len) {
        this._maxValue = len;
    }
    /**
     * A numeric value that determines the smallest numeric value you want to allow for Number types.
     *
     * @default - None.
     */
    get minValue() {
        return this._minValue;
    }
    set minValue(len) {
        this._minValue = len;
    }
    /**
     * Indicates if this parameter is configured with "NoEcho" enabled.
     */
    get noEcho() {
        return !!this._noEcho;
    }
    set noEcho(echo) {
        this._noEcho = echo;
    }
    /**
     * The parameter value as a Token
     */
    get value() {
        return cfn_reference_1.CfnReference.for(this, 'Ref');
    }
    /**
     * The parameter value, if it represents a string.
     */
    get valueAsString() {
        if (!isStringType(this.type) && !isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string or number type`);
        }
        return token_1.Token.asString(this.value);
    }
    /**
     * The parameter value, if it represents a string list.
     */
    get valueAsList() {
        if (!isListType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string list type`);
        }
        return token_1.Token.asList(this.value);
    }
    /**
     * The parameter value, if it represents a number.
     */
    get valueAsNumber() {
        if (!isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a number type`);
        }
        return token_1.Token.asNumber(this.value);
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Parameters: {
                [this.logicalId]: {
                    Type: this.type,
                    Default: this.default,
                    AllowedPattern: this.allowedPattern,
                    AllowedValues: this.allowedValues,
                    ConstraintDescription: this.constraintDescription,
                    Description: this.description,
                    MaxLength: this.maxLength,
                    MaxValue: this.maxValue,
                    MinLength: this.minLength,
                    MinValue: this.minValue,
                    NoEcho: this._noEcho,
                },
            },
        };
    }
    resolve(_context) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_IResolveContext(_context);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.resolve);
            }
            throw error;
        }
        return this.value;
    }
}
exports.CfnParameter = CfnParameter;
_a = JSII_RTTI_SYMBOL_1;
CfnParameter[_a] = { fqn: "@aws-cdk/core.CfnParameter", version: "1.180.0" };
/**
 * Whether the given parameter type looks like a list type
 */
function isListType(type) {
    return type.indexOf('List<') >= 0 || type.indexOf('CommaDelimitedList') >= 0;
}
/**
 * Whether the given parameter type looks like a number type
 */
function isNumberType(type) {
    return type === 'Number';
}
/**
 * Whether the given parameter type looks like a string type
 */
function isStringType(type) {
    return !isListType(type) && !isNumberType(type);
}
//# sourceMappingURL=data:application/json;base64,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