"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Default = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_entities_1 = require("./aws-entities");
/**
 * Provides default values for certain regional information points.
 */
class Default {
    constructor() { }
    /**
     * Computes a "standard" AWS Service principal for a given service, region and suffix. This is useful for example when
     * you need to compute a service principal name, but you do not have a synthesize-time region literal available (so
     * all you have is `{ "Ref": "AWS::Region" }`). This way you get the same defaulting behavior that is normally used
     * for built-in data.
     *
     * @param serviceFqn the name of the service (s3, s3.amazonaws.com, ...)
     * @param region    the region in which the service principal is needed.
     * @param urlSuffix deprecated and ignored.
     */
    static servicePrincipal(serviceFqn, region, urlSuffix) {
        const serviceName = extractSimpleName(serviceFqn);
        if (!serviceName) {
            // Return "service" if it does not look like any of the following:
            // - s3
            // - s3.amazonaws.com
            // - s3.amazonaws.com.cn
            // - s3.c2s.ic.gov
            // - s3.sc2s.sgov.gov
            return serviceFqn;
        }
        function determineConfiguration(service) {
            function universal(s) { return `${s}.amazonaws.com`; }
            function partitional(s, _, u) { return `${s}.${u}`; }
            function regional(s, r) { return `${s}.${r}.amazonaws.com`; }
            function regionalPartitional(s, r, u) { return `${s}.${r}.${u}`; }
            // Exceptions for Service Principals in us-iso-*
            const US_ISO_EXCEPTIONS = new Set([
                'cloudhsm',
                'config',
                'states',
                'workspaces',
            ]);
            // Account for idiosyncratic Service Principals in `us-iso-*` regions
            if (region.startsWith('us-iso-') && US_ISO_EXCEPTIONS.has(service)) {
                switch (service) {
                    // Services with universal principal
                    case ('states'):
                        return universal;
                    // Services with a partitional principal
                    default:
                        return partitional;
                }
            }
            // Exceptions for Service Principals in us-isob-*
            const US_ISOB_EXCEPTIONS = new Set([
                'dms',
                'states',
            ]);
            // Account for idiosyncratic Service Principals in `us-isob-*` regions
            if (region.startsWith('us-isob-') && US_ISOB_EXCEPTIONS.has(service)) {
                switch (service) {
                    // Services with universal principal
                    case ('states'):
                        return universal;
                    // Services with a partitional principal
                    default:
                        return partitional;
                }
            }
            switch (service) {
                // SSM turned from global to regional at some point
                case 'ssm':
                    return aws_entities_1.before(region, aws_entities_1.RULE_SSM_PRINCIPALS_ARE_REGIONAL)
                        ? universal
                        : regional;
                // CodeDeploy is regional+partitional in CN, only regional everywhere else
                case 'codedeploy':
                    return region.startsWith('cn-')
                        ? regionalPartitional
                        // ...except in the isolated regions, where it's universal
                        : (region.startsWith('us-iso') ? universal : regional);
                // Services with a regional AND partitional principal
                case 'logs':
                    return regionalPartitional;
                // Services with a regional principal
                case 'states':
                    return regional;
                // Services with a partitional principal
                case 'ec2':
                    return partitional;
                case 'elasticmapreduce':
                    return region.startsWith('cn-')
                        ? partitional
                        : universal;
                // Services with a universal principal across all regions/partitions (the default case)
                default:
                    return universal;
            }
        }
        ;
        const configuration = determineConfiguration(serviceName);
        return configuration(serviceName, region, urlSuffix);
    }
}
exports.Default = Default;
_a = JSII_RTTI_SYMBOL_1;
Default[_a] = { fqn: "@aws-cdk/region-info.Default", version: "1.176.0" };
/**
 * The default value for a VPC Endpoint Service name prefix, useful if you do
 * not have a synthesize-time region literal available (all you have is
 * `{ "Ref": "AWS::Region" }`)
 */
Default.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'com.amazonaws.vpce';
function extractSimpleName(serviceFqn) {
    const matches = serviceFqn.match(/^([^.]+)(?:(?:\.amazonaws\.com(?:\.cn)?)|(?:\.c2s\.ic\.gov)|(?:\.sc2s\.sgov\.gov))?$/);
    return matches ? matches[1] : undefined;
}
//# sourceMappingURL=data:application/json;base64,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