"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resource = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const arn_1 = require("./arn");
const cfn_resource_1 = require("./cfn-resource");
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const physical_name_generator_1 = require("./private/physical-name-generator");
const reference_1 = require("./reference");
const stack_1 = require("./stack");
const token_1 = require("./token");
const RESOURCE_SYMBOL = Symbol.for('@aws-cdk/core.Resource');
/**
 * A construct which represents an AWS resource.
 */
class Resource extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_core_ResourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Resource);
            }
            throw error;
        }
        if ((props.account !== undefined || props.region !== undefined) && props.environmentFromArn !== undefined) {
            throw new Error(`Supply at most one of 'account'/'region' (${props.account}/${props.region}) and 'environmentFromArn' (${props.environmentFromArn})`);
        }
        Object.defineProperty(this, RESOURCE_SYMBOL, { value: true });
        this.stack = stack_1.Stack.of(this);
        const parsedArn = props.environmentFromArn ?
            // Since we only want the region and account, NO_RESOURE_NAME is good enough
            this.stack.splitArn(props.environmentFromArn, arn_1.ArnFormat.NO_RESOURCE_NAME)
            : undefined;
        this.env = {
            account: props.account ?? parsedArn?.account ?? this.stack.account,
            region: props.region ?? parsedArn?.region ?? this.stack.region,
        };
        let physicalName = props.physicalName;
        if (props.physicalName && physical_name_generator_1.isGeneratedWhenNeededMarker(props.physicalName)) {
            // auto-generate only if cross-env is required
            this._physicalName = undefined;
            this._allowCrossEnvironment = true;
            physicalName = lazy_1.Lazy.string({ produce: () => this._physicalName });
        }
        else if (props.physicalName && !token_1.Token.isUnresolved(props.physicalName)) {
            // concrete value specified by the user
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = true;
        }
        else {
            // either undefined (deploy-time) or has tokens, which means we can't use for cross-env
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = false;
        }
        if (physicalName === undefined) {
            physicalName = token_1.Token.asString(undefined);
        }
        this.physicalName = physicalName;
    }
    /**
     * Check whether the given construct is a Resource
     */
    static isResource(construct) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_IConstruct(construct);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.isResource);
            }
            throw error;
        }
        return construct !== null && typeof (construct) === 'object' && RESOURCE_SYMBOL in construct;
    }
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment() {
        if (!this._allowCrossEnvironment) {
            // error out - a deploy-time name cannot be used across environments
            throw new Error(`Cannot use resource '${this.node.path}' in a cross-environment fashion, ` +
                "the resource's physical name must be explicit set or use `PhysicalName.GENERATE_IF_NEEDED`");
        }
        if (!this._physicalName) {
            this._physicalName = this.generatePhysicalName();
        }
    }
    /**
     * Apply the given removal policy to this resource
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be deleted (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     */
    applyRemovalPolicy(policy) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_RemovalPolicy(policy);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.applyRemovalPolicy);
            }
            throw error;
        }
        const child = this.node.defaultChild;
        if (!child || !cfn_resource_1.CfnResource.isCfnResource(child)) {
            throw new Error('Cannot apply RemovalPolicy: no child or not a CfnResource. Apply the removal policy on the CfnResource directly.');
        }
        child.applyRemovalPolicy(policy);
    }
    generatePhysicalName() {
        return physical_name_generator_1.generatePhysicalName(this);
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "name" attribute (e.g. `bucket.bucketName`).
     *
     * Normally, this token will resolve to `nameAttr`, but if the resource is
     * referenced across environments, it will be resolved to `this.physicalName`,
     * which will be a concrete name.
     *
     * @param nameAttr The CFN attribute which resolves to the resource's name.
     * Commonly this is the resource's `ref`.
     */
    getResourceNameAttribute(nameAttr) {
        return mimicReference(nameAttr, {
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.physicalName;
                }
                else {
                    return nameAttr;
                }
            },
        });
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "ARN" attribute (e.g. `bucket.bucketArn`).
     *
     * Normally, this token will resolve to `arnAttr`, but if the resource is
     * referenced across environments, `arnComponents` will be used to synthesize
     * a concrete ARN with the resource's physical name. Make sure to reference
     * `this.physicalName` in `arnComponents`.
     *
     * @param arnAttr The CFN attribute which resolves to the ARN of the resource.
     * Commonly it will be called "Arn" (e.g. `resource.attrArn`), but sometimes
     * it's the CFN resource's `ref`.
     * @param arnComponents The format of the ARN of this resource. You must
     * reference `this.physicalName` somewhere within the ARN in order for
     * cross-environment references to work.
     *
     */
    getResourceArnAttribute(arnAttr, arnComponents) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_ArnComponents(arnComponents);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getResourceArnAttribute);
            }
            throw error;
        }
        return mimicReference(arnAttr, {
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.stack.formatArn(arnComponents);
                }
                else {
                    return arnAttr;
                }
            },
        });
    }
}
exports.Resource = Resource;
_a = JSII_RTTI_SYMBOL_1;
Resource[_a] = { fqn: "@aws-cdk/core.Resource", version: "1.176.0" };
/**
 * Produce a Lazy that is also a Reference (if the base value is a Reference).
 *
 * If the given value is a Reference (or resolves to a Reference), return a new
 * Reference that mimics the same target and display name, but resolves using
 * the logic of the passed lazy.
 *
 * If the given value is NOT a Reference, just return a simple Lazy.
 */
function mimicReference(refSource, producer) {
    const reference = token_1.Tokenization.reverse(refSource, {
        // If this is an ARN concatenation, just fail to extract a reference.
        failConcat: false,
    });
    if (!reference_1.Reference.isReference(reference)) {
        return lazy_1.Lazy.uncachedString(producer);
    }
    return token_1.Token.asString(new class extends reference_1.Reference {
        resolve(context) {
            return producer.produce(context);
        }
    }(reference, reference.target, reference.displayName));
}
//# sourceMappingURL=data:application/json;base64,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