#!/usr/bin/env node
"use strict";
/**
 * automatically creates a module for any CloudFormation namespaces that do not
 * have an AWS construct library.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const pkglint = require("@aws-cdk/pkglint");
const fs = require("fs-extra");
const cfnspec = require("../lib");
// don't be a prude:
/* eslint-disable no-console */
/* eslint-disable quote-props */
async function main() {
    const root = path.join(__dirname, '..', '..');
    if (path.basename(root) !== '@aws-cdk') {
        throw new Error(`Something went wrong. We expected ${root} to be the "packages/@aws-cdk" directory. Did you move me?`);
    }
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const cfnSpecPkgJson = require('../package.json');
    const version = cfnSpecPkgJson.version;
    const jestTypesVersion = cfnSpecPkgJson.devDependencies['@types/jest'];
    // iterate over all cloudformation namespaces
    for (const namespace of cfnspec.namespaces()) {
        const module = pkglint.createModuleDefinitionFromCfnNamespace(namespace);
        const lowcaseModuleName = module.moduleBaseName.toLocaleLowerCase();
        const packagePath = path.join(root, module.moduleName);
        // we already have a module for this namesapce, move on.
        if (await fs.pathExists(packagePath)) {
            const packageJsonPath = path.join(packagePath, 'package.json');
            // eslint-disable-next-line @typescript-eslint/no-require-imports
            const packageJson = require(packageJsonPath);
            let scopes = packageJson['cdk-build'].cloudformation;
            if (typeof scopes === 'string') {
                scopes = [scopes];
            }
            if (scopes.indexOf(namespace) !== -1) {
                // V2-style module is already modeled in the root package, nothing to be done!
                continue;
            }
            else if (await fs.pathExists(path.join(root, `${module.moduleFamily}-${module.moduleBaseName}`.toLocaleLowerCase()))) {
                // V2-style package already has it's own package (legacy behavior), nothing to be done!
                continue;
            }
            else {
                // V2-style package needs to be added to it's "V1" package... Get down to business!
                console.error(`Adding ${namespace} to ${module.packageName}`);
                scopes.push(namespace);
                packageJson['cdk-build'].cloudformation = scopes;
                await fs.writeJson(packageJsonPath, packageJson, { encoding: 'utf-8', spaces: 2 });
                const indexTsPath = path.join(packagePath, 'lib', 'index.ts');
                const indexTs = [
                    (await fs.readFile(indexTsPath, { encoding: 'utf8' })).trimRight(),
                    `// ${namespace} CloudFormation Resources:`,
                    `export * from './${lowcaseModuleName}.generated';`,
                ].join('\n');
                await fs.writeFile(indexTsPath, indexTs, { encoding: 'utf8' });
                continue;
            }
        }
        async function write(relativePath, contents) {
            const fullPath = path.join(packagePath, relativePath);
            const dir = path.dirname(fullPath);
            await fs.mkdirp(dir);
            let data;
            if (typeof contents === 'string') {
                data = contents.trimLeft(); // trim first newline
            }
            else if (Array.isArray(contents)) {
                data = contents.join('\n');
            }
            else if (typeof contents === 'object') {
                data = JSON.stringify(contents, undefined, 2);
            }
            else {
                throw new Error('Invalid type of contents: ' + contents);
            }
            await fs.writeFile(fullPath, data + '\n');
        }
        console.log(`generating module for ${module.packageName}...`);
        const description = `${namespace} Construct Library`;
        await write('package.json', {
            name: module.packageName,
            version,
            description,
            main: 'lib/index.js',
            types: 'lib/index.d.ts',
            jsii: {
                outdir: 'dist',
                projectReferences: true,
                targets: {
                    dotnet: {
                        namespace: module.dotnetPackage,
                        packageId: module.dotnetPackage,
                        signAssembly: true,
                        assemblyOriginatorKeyFile: '../../key.snk',
                        iconUrl: 'https://raw.githubusercontent.com/aws/aws-cdk/master/logo/default-256-dark.png',
                    },
                    java: {
                        package: `${module.javaGroupId}.${module.javaPackage}`,
                        maven: {
                            groupId: module.javaGroupId,
                            artifactId: module.javaArtifactId,
                        },
                    },
                    python: {
                        classifiers: [
                            'Framework :: AWS CDK',
                            'Framework :: AWS CDK :: 1',
                        ],
                        distName: module.pythonDistName,
                        module: module.pythonModuleName,
                    },
                },
                metadata: {
                    jsii: {
                        rosetta: {
                            strict: true,
                        },
                    },
                },
            },
            repository: {
                type: 'git',
                url: 'https://github.com/aws/aws-cdk.git',
                directory: `packages/${module.packageName}`,
            },
            homepage: 'https://github.com/aws/aws-cdk',
            scripts: {
                build: 'cdk-build',
                watch: 'cdk-watch',
                lint: 'cdk-lint',
                test: 'cdk-test',
                integ: 'cdk-integ',
                pkglint: 'pkglint -f',
                package: 'cdk-package',
                awslint: 'cdk-awslint',
                cfn2ts: 'cfn2ts',
                'build+test': 'yarn build && yarn test',
                'build+test+package': 'yarn build+test && yarn package',
                compat: 'cdk-compat',
                gen: 'cfn2ts',
                'rosetta:extract': 'yarn --silent jsii-rosetta extract',
                'build+extract': 'yarn build && yarn rosetta:extract',
                'build+test+extract': 'yarn build+test && yarn rosetta:extract',
            },
            'cdk-build': {
                cloudformation: namespace,
                jest: true,
                env: {
                    AWSLINT_BASE_CONSTRUCT: 'true',
                },
            },
            keywords: [
                'aws',
                'cdk',
                'constructs',
                namespace,
                module.moduleName,
            ],
            author: {
                name: 'Amazon Web Services',
                url: 'https://aws.amazon.com',
                organization: true,
            },
            license: 'Apache-2.0',
            devDependencies: {
                '@aws-cdk/assertions': version,
                '@aws-cdk/cdk-build-tools': version,
                '@aws-cdk/cfn2ts': version,
                '@aws-cdk/pkglint': version,
                '@types/jest': jestTypesVersion,
            },
            dependencies: {
                '@aws-cdk/core': version,
                'constructs': '^3.3.69',
            },
            peerDependencies: {
                '@aws-cdk/core': version,
                'constructs': '^3.3.69',
            },
            engines: {
                node: '>= 14.15.0',
            },
            stability: 'experimental',
            maturity: 'cfn-only',
            awscdkio: {
                announce: false,
            },
            publishConfig: {
                tag: 'latest',
            },
        });
        await write('.gitignore', [
            '*.js',
            '*.js.map',
            '*.d.ts',
            'tsconfig.json',
            'node_modules',
            '*.generated.ts',
            'dist',
            '.jsii',
            '',
            '.LAST_BUILD',
            '.nyc_output',
            'coverage',
            '.nycrc',
            '.LAST_PACKAGE',
            '*.snk',
            'nyc.config.js',
            '!.eslintrc.js',
            '!jest.config.js',
            'junit.xml',
        ]);
        await write('.npmignore', [
            '# Don\'t include original .ts files when doing `npm pack`',
            '*.ts',
            '!*.d.ts',
            'coverage',
            '.nyc_output',
            '*.tgz',
            '',
            'dist',
            '.LAST_PACKAGE',
            '.LAST_BUILD',
            '!*.js',
            '',
            '# Include .jsii',
            '!.jsii',
            '',
            '*.snk',
            '',
            '*.tsbuildinfo',
            '',
            'tsconfig.json',
            '',
            '.eslintrc.js',
            'jest.config.js',
            '',
            '# exclude cdk artifacts',
            '**/cdk.out',
            'junit.xml',
            'test/',
            '!*.lit.ts',
        ]);
        await write('lib/index.ts', [
            `// ${namespace} CloudFormation Resources:`,
            `export * from './${lowcaseModuleName}.generated';`,
        ]);
        await write(`test/${lowcaseModuleName}.test.ts`, [
            "import '@aws-cdk/assertions';",
            "import {} from '../lib';",
            '',
            "test('No tests are specified for this package', () => {",
            '  expect(true).toBe(true);',
            '});',
        ]);
        await pkglint.createLibraryReadme(namespace, path.join(packagePath, 'README.md'));
        await write('.eslintrc.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/eslintrc');",
            "baseConfig.parserOptions.project = __dirname + '/tsconfig.json';",
            'module.exports = baseConfig;',
        ]);
        await write('jest.config.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/jest.config');",
            'module.exports = baseConfig;',
        ]);
        await write('rosetta/default.ts-fixture', [
            "import { Construct } from 'constructs';",
            "import { Stack } from '@aws-cdk/core';",
            '',
            'class MyStack extends Stack {',
            '  constructor(scope: Construct, id: string) {',
            '    /// here',
            '  }',
            '}',
        ]);
        const templateDir = path.join(__dirname, 'template');
        for (const file of await fs.readdir(templateDir)) {
            await fs.copy(path.join(templateDir, file), path.join(packagePath, file));
        }
        await addDependencyToMegaPackage(path.join('@aws-cdk', 'cloudformation-include'), module.packageName, version, ['dependencies', 'peerDependencies']);
        await addDependencyToMegaPackage('aws-cdk-lib', module.packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('monocdk', module.packageName, version, ['devDependencies']);
    }
}
/**
 * A few of our packages (e.g., aws-cdk-lib) require a dependency on every service package.
 * This automates adding the dependency (and peer dependency) to the package.json.
 */
async function addDependencyToMegaPackage(megaPackageName, packageName, version, dependencyTypes) {
    const packageJsonPath = path.join(__dirname, '..', '..', '..', megaPackageName, 'package.json');
    const packageJson = JSON.parse(await fs.readFile(packageJsonPath, 'utf8'));
    dependencyTypes.forEach(dependencyType => {
        const unorderedDeps = {
            ...packageJson[dependencyType],
            [packageName]: version,
        };
        packageJson[dependencyType] = {};
        Object.keys(unorderedDeps).sort().forEach(k => {
            packageJson[dependencyType][k] = unorderedDeps[k];
        });
    });
    await fs.writeFile(packageJsonPath, JSON.stringify(packageJson, null, 2) + '\n');
}
main().catch(e => {
    console.error(e);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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