"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RegionInfo = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_entities_1 = require("./aws-entities");
const fact_1 = require("./fact");
/**
 * Information pertaining to an AWS region.
 */
class RegionInfo {
    constructor(name) {
        this.name = name;
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     *          may not be an exaustive list of all available AWS regions.
     */
    static get regions() {
        return fact_1.Fact.regions.map(RegionInfo.get);
    }
    /**
     * Retrieves a collection of all fact values for all regions that fact is defined in.
     *
     * @param factName the name of the fact to retrieve values for.
     *   For a list of common fact names, see the FactName class
     * @returns a mapping with AWS region codes as the keys,
     *   and the fact in the given region as the value for that key
     */
    static regionMap(factName) {
        const ret = {};
        for (const regionInfo of RegionInfo.regions) {
            const fact = fact_1.Fact.find(regionInfo.name, factName);
            if (fact) {
                ret[regionInfo.name] = fact;
            }
        }
        return ret;
    }
    /**
     * Retrieves a collection of all fact values for all regions, limited to some partitions
     *
     * @param factName the name of the fact to retrieve values for.
     *   For a list of common fact names, see the FactName class
     * @param partitions list of partitions to retrieve facts for. Defaults
     *   to `['aws', 'aws-cn']`.
     * @returns a mapping with AWS region codes as the keys,
     *   and the fact in the given region as the value for that key
     */
    static limitedRegionMap(factName, partitions) {
        const ret = {};
        for (const [region, value] of Object.entries(RegionInfo.regionMap(factName))) {
            if (partitions.includes(aws_entities_1.partitionInformation(region).partition)) {
                ret[region] = value;
            }
        }
        return ret;
    }
    /**
     * Obtain region info for a given region name.
     *
     * @param name the name of the region (e.g: us-east-1)
     */
    static get(name) {
        return new RegionInfo(name);
    }
    /**
     * Whether the `AWS::CDK::Metadata` CloudFormation Resource is available in this region or not.
     */
    get cdkMetadataResourceAvailable() {
        return fact_1.Fact.find(this.name, fact_1.FactName.CDK_METADATA_RESOURCE_AVAILABLE) === 'YES';
    }
    /**
     * The domain name suffix (e.g: amazonaws.com) for this region.
     */
    get domainSuffix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DOMAIN_SUFFIX);
    }
    /**
     * The name of the ARN partition for this region (e.g: aws).
     */
    get partition() {
        return fact_1.Fact.find(this.name, fact_1.FactName.PARTITION);
    }
    /**
     * The endpoint used by S3 static website hosting in this region (e.g: s3-static-website-us-east-1.amazonaws.com)
     */
    get s3StaticWebsiteEndpoint() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ENDPOINT);
    }
    /**
     * The hosted zone ID used by Route 53 to alias a S3 static website in this region (e.g: Z2O1EMRO9K5GLX)
     */
    get s3StaticWebsiteHostedZoneId() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID);
    }
    /**
    * The hosted zone ID used by Route 53 to alias a EBS environment endpoint in this region (e.g: Z2O1EMRO9K5GLX)
    */
    get ebsEnvEndpointHostedZoneId() {
        return fact_1.Fact.find(this.name, fact_1.FactName.EBS_ENV_ENDPOINT_HOSTED_ZONE_ID);
    }
    /**
     * The prefix for VPC Endpoint Service names,
     * cn.com.amazonaws.vpce for China regions,
     * com.amazonaws.vpce otherwise.
     */
    get vpcEndpointServiceNamePrefix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX);
    }
    /**
     * The name of the service principal for a given service in this region.
     * @param service the service name (e.g: s3.amazonaws.com)
     */
    servicePrincipal(service) {
        return fact_1.Fact.find(this.name, fact_1.FactName.servicePrincipal(service));
    }
    /**
     * The account ID for ELBv2 in this region
     *
     */
    get elbv2Account() {
        return fact_1.Fact.find(this.name, fact_1.FactName.ELBV2_ACCOUNT);
    }
    /**
     * The ID of the AWS account that owns the public ECR repository containing the
     * AWS Deep Learning Containers images in this region.
     */
    get dlcRepositoryAccount() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DLC_REPOSITORY_ACCOUNT);
    }
    /**
     * The ARN of the CloudWatch Lambda Insights extension, for the given version.
     * @param insightsVersion the version (e.g. 1.0.98.0)
     * @param architecture the Lambda Function architecture (e.g. 'x86_64' or 'arm64')
     */
    cloudwatchLambdaInsightsArn(insightsVersion, architecture) {
        return fact_1.Fact.find(this.name, fact_1.FactName.cloudwatchLambdaInsightsVersion(insightsVersion, architecture));
    }
    /**
     * The ID of the AWS account that owns the public ECR repository that contains the
     * AWS App Mesh Envoy Proxy images in a given region.
     */
    get appMeshRepositoryAccount() {
        return fact_1.Fact.find(this.name, fact_1.FactName.APPMESH_ECR_ACCOUNT);
    }
    /**
     * The CIDR block used by Kinesis Data Firehose servers.
     */
    get firehoseCidrBlock() {
        return fact_1.Fact.find(this.name, fact_1.FactName.FIREHOSE_CIDR_BLOCK);
    }
}
exports.RegionInfo = RegionInfo;
_a = JSII_RTTI_SYMBOL_1;
RegionInfo[_a] = { fqn: "@aws-cdk/region-info.RegionInfo", version: "1.171.0" };
//# sourceMappingURL=data:application/json;base64,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