"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NestedStack = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const assets_1 = require("./assets");
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cloudformation_generated_1 = require("./cloudformation.generated");
const lazy_1 = require("./lazy");
const names_1 = require("./names");
const removal_policy_1 = require("./removal-policy");
const stack_1 = require("./stack");
const stack_synthesizers_1 = require("./stack-synthesizers");
const token_1 = require("./token");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("./construct-compat");
const NESTED_STACK_SYMBOL = Symbol.for('@aws-cdk/core.NestedStack');
/**
 * A CloudFormation nested stack.
 *
 * When you apply template changes to update a top-level stack, CloudFormation
 * updates the top-level stack and initiates an update to its nested stacks.
 * CloudFormation updates the resources of modified nested stacks, but does not
 * update the resources of unmodified nested stacks.
 *
 * Furthermore, this stack will not be treated as an independent deployment
 * artifact (won't be listed in "cdk list" or deployable through "cdk deploy"),
 * but rather only synthesized as a template and uploaded as an asset to S3.
 *
 * Cross references of resource attributes between the parent stack and the
 * nested stack will automatically be translated to stack parameters and
 * outputs.
 *
 */
class NestedStack extends stack_1.Stack {
    constructor(scope, id, props = {}) {
        var _b;
        try {
            jsiiDeprecationWarnings._aws_cdk_core_NestedStackProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const parentStack = findParentStack(scope);
        super(scope, id, {
            env: { account: parentStack.account, region: parentStack.region },
            synthesizer: new stack_synthesizers_1.NestedStackSynthesizer(parentStack.synthesizer),
        });
        this._parentStack = parentStack;
        // @deprecate: remove this in v2.0 (redundent)
        const parentScope = new construct_compat_1.Construct(scope, id + '.NestedStack');
        Object.defineProperty(this, NESTED_STACK_SYMBOL, { value: true });
        // this is the file name of the synthesized template file within the cloud assembly
        this.templateFile = `${names_1.Names.uniqueId(this)}.nested.template.json`;
        this.parameters = props.parameters || {};
        this.resource = new cloudformation_generated_1.CfnStack(parentScope, `${id}.NestedStackResource`, {
            // This value cannot be cached since it changes during the synthesis phase
            templateUrl: lazy_1.Lazy.uncachedString({ produce: () => this._templateUrl || '<unresolved>' }),
            parameters: lazy_1.Lazy.any({ produce: () => Object.keys(this.parameters).length > 0 ? this.parameters : undefined }),
            notificationArns: props.notificationArns,
            timeoutInMinutes: props.timeout ? props.timeout.toMinutes() : undefined,
        });
        this.resource.applyRemovalPolicy((_b = props.removalPolicy) !== null && _b !== void 0 ? _b : removal_policy_1.RemovalPolicy.DESTROY);
        this.nestedStackResource = this.resource;
        this.node.defaultChild = this.resource;
        // context-aware stack name: if resolved from within this stack, return AWS::StackName
        // if resolved from the outer stack, use the { Ref } of the AWS::CloudFormation::Stack resource
        // which resolves the ARN of the stack. We need to extract the stack name, which is the second
        // component after splitting by "/"
        this._contextualStackName = this.contextualAttribute(cfn_pseudo_1.Aws.STACK_NAME, cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split('/', this.resource.ref)));
        this._contextualStackId = this.contextualAttribute(cfn_pseudo_1.Aws.STACK_ID, this.resource.ref);
    }
    /**
     * Checks if `x` is an object of type `NestedStack`.
     */
    static isNestedStack(x) {
        return x != null && typeof (x) === 'object' && NESTED_STACK_SYMBOL in x;
    }
    /**
     * An attribute that represents the name of the nested stack.
     *
     * This is a context aware attribute:
     * - If this is referenced from the parent stack, it will return a token that parses the name from the stack ID.
     * - If this is referenced from the context of the nested stack, it will return `{ "Ref": "AWS::StackName" }`
     *
     * Example value: `mystack-mynestedstack-sggfrhxhum7w`
     * @attribute
     */
    get stackName() {
        return this._contextualStackName;
    }
    /**
     * An attribute that represents the ID of the stack.
     *
     * This is a context aware attribute:
     * - If this is referenced from the parent stack, it will return `{ "Ref": "LogicalIdOfNestedStackResource" }`.
     * - If this is referenced from the context of the nested stack, it will return `{ "Ref": "AWS::StackId" }`
     *
     * Example value: `arn:aws:cloudformation:us-east-2:123456789012:stack/mystack-mynestedstack-sggfrhxhum7w/f449b250-b969-11e0-a185-5081d0136786`
     * @attribute
     */
    get stackId() {
        return this._contextualStackId;
    }
    /**
     * Assign a value to one of the nested stack parameters.
     * @param name The parameter name (ID)
     * @param value The value to assign
     */
    setParameter(name, value) {
        this.parameters[name] = value;
    }
    /**
     * Defines an asset at the parent stack which represents the template of this
     * nested stack.
     *
     * This private API is used by `App.prepare()` within a loop that rectifies
     * references every time an asset is added. This is because (at the moment)
     * assets are addressed using CloudFormation parameters.
     *
     * @returns `true` if a new asset was added or `false` if an asset was
     * previously added. When this returns `true`, App will do another reference
     * rectification cycle.
     *
     * @internal
     */
    _prepareTemplateAsset() {
        if (this._templateUrl) {
            return false;
        }
        // When adding tags to nested stack, the tags need to be added to all the resources in
        // in nested stack, which is handled by the `tags` property, But to tag the
        //  tags have to be added in the parent stack CfnStack resource. The CfnStack resource created
        // by this class dont share the same TagManager as that of the one exposed by the `tag` property of the
        //  class, all the tags need to be copied to the CfnStack resource before synthesizing the resource.
        // See https://github.com/aws/aws-cdk/pull/19128
        Object.entries(this.tags.tagValues()).forEach(([key, value]) => {
            this.resource.tags.setTag(key, value);
        });
        const cfn = JSON.stringify(this._toCloudFormation());
        const templateHash = crypto.createHash('sha256').update(cfn).digest('hex');
        const templateLocation = this._parentStack.synthesizer.addFileAsset({
            packaging: assets_1.FileAssetPackaging.FILE,
            sourceHash: templateHash,
            fileName: this.templateFile,
        });
        this.addResourceMetadata(this.resource, 'TemplateURL');
        // if bucketName/objectKey are cfn parameters from a stack other than the parent stack, they will
        // be resolved as cross-stack references like any other (see "multi" tests).
        this._templateUrl = `https://s3.${this._parentStack.region}.${this._parentStack.urlSuffix}/${templateLocation.bucketName}/${templateLocation.objectKey}`;
        return true;
    }
    contextualAttribute(innerValue, outerValue) {
        return token_1.Token.asString({
            resolve: (context) => {
                if (stack_1.Stack.of(context.scope) === this) {
                    return innerValue;
                }
                else {
                    return outerValue;
                }
            },
        });
    }
    addResourceMetadata(resource, resourceProperty) {
        if (!this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        // tell tools such as SAM CLI that the "TemplateURL" property of this resource
        // points to the nested stack template for local emulation
        resource.cfnOptions.metadata = resource.cfnOptions.metadata || {};
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PATH_KEY] = this.templateFile;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY] = resourceProperty;
    }
}
exports.NestedStack = NestedStack;
_a = JSII_RTTI_SYMBOL_1;
NestedStack[_a] = { fqn: "@aws-cdk/core.NestedStack", version: "1.159.0" };
/**
 * Validates the scope for a nested stack. Nested stacks must be defined within the scope of another `Stack`.
 */
function findParentStack(scope) {
    if (!scope) {
        throw new Error('Nested stacks cannot be defined as a root construct');
    }
    const parentStack = constructs_1.Node.of(scope).scopes.reverse().find(p => stack_1.Stack.isStack(p));
    if (!parentStack) {
        throw new Error('Nested stacks must be defined within scope of another non-nested stack');
    }
    return parentStack;
}
//# sourceMappingURL=data:application/json;base64,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