"use strict";
// --------------------------------------------------------------------------------
// This file defines context keys that enable certain features that are
// implemented behind a flag in order to preserve backwards compatibility for
// existing apps. When a new app is initialized through `cdk init`, the CLI will
// automatically add enable these features by adding them to the generated
// `cdk.json` file.
//
// Some of these flags only affect the behavior of the construct library --
// these will be removed in the next major release and the behavior they are
// gating will become the only behavior.
//
// Other flags also affect the generated CloudFormation templates, in a way
// that prevents seamless upgrading. In the next major version, their
// behavior will become the default, but the flag still exists so users can
// switch it *off* in order to revert to the old behavior. These flags
// are marked with with the [PERMANENT] tag below.
//
// See https://github.com/aws/aws-cdk-rfcs/blob/master/text/0055-feature-flags.md
// --------------------------------------------------------------------------------
Object.defineProperty(exports, "__esModule", { value: true });
exports.futureFlagDefault = exports.FUTURE_FLAGS_EXPIRED = exports.NEW_PROJECT_DEFAULT_CONTEXT = exports.FUTURE_FLAGS = exports.IAM_MINIMIZE_POLICIES = exports.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME = exports.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER = exports.TARGET_PARTITIONS = exports.CHECK_SECRET_USAGE = exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021 = exports.LAMBDA_RECOGNIZE_VERSION_PROPS = exports.EFS_DEFAULT_ENCRYPTION_AT_REST = exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID = exports.RDS_LOWERCASE_DB_IDENTIFIER = exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT = exports.S3_GRANT_WRITE_WITHOUT_ACL = exports.KMS_DEFAULT_KEY_POLICIES = exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = exports.DOCKER_IGNORE_SUPPORT = exports.STACK_RELATIVE_EXPORTS_CONTEXT = exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT = exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = void 0;
/**
 * If this is set, multiple stacks can use the same stack name (e.g. deployed to
 * different environments). This means that the name of the synthesized template
 * file will be based on the construct path and not on the defined `stackName`
 * of the stack.
 *
 * This is a "future flag": the feature is disabled by default for backwards
 * compatibility, but new projects created using `cdk init` will have this
 * enabled through the generated `cdk.json`.
 */
exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = '@aws-cdk/core:enableStackNameDuplicates';
/**
 * IF this is set, `cdk diff` will always exit with 0.
 *
 * Use `cdk diff --fail` to exit with 1 if there's a diff.
 */
exports.ENABLE_DIFF_NO_FAIL_CONTEXT = 'aws-cdk:enableDiffNoFail';
/** @deprecated use `ENABLE_DIFF_NO_FAIL_CONTEXT` */
exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT;
/**
 * Switch to new stack synthesis method which enable CI/CD
 *
 * [PERMANENT]
 */
exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = '@aws-cdk/core:newStyleStackSynthesis';
/**
 * Name exports based on the construct paths relative to the stack, rather than the global construct path
 *
 * Combined with the stack name this relative construct path is good enough to
 * ensure uniqueness, and makes the export names robust against refactoring
 * the location of the stack in the construct tree (specifically, moving the Stack
 * into a Stage).
 *
 * [PERMANENT]
 */
exports.STACK_RELATIVE_EXPORTS_CONTEXT = '@aws-cdk/core:stackRelativeExports';
/**
 * DockerImageAsset properly supports `.dockerignore` files by default
 *
 * If this flag is not set, the default behavior for `DockerImageAsset` is to use
 * glob semantics for `.dockerignore` files. If this flag is set, the default behavior
 * is standard Docker ignore semantics.
 *
 * This is a feature flag as the old behavior was technically incorrect but
 * users may have come to depend on it.
 */
exports.DOCKER_IGNORE_SUPPORT = '@aws-cdk/aws-ecr-assets:dockerIgnoreSupport';
/**
 * Secret.secretName for an "owned" secret will attempt to parse the secretName from the ARN,
 * rather than the default full resource name, which includes the SecretsManager suffix.
 *
 * If this flag is not set, Secret.secretName will include the SecretsManager suffix, which cannot be directly
 * used by SecretsManager.DescribeSecret, and must be parsed by the user first (e.g., Fn:Join, Fn:Select, Fn:Split).
 */
exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = '@aws-cdk/aws-secretsmanager:parseOwnedSecretName';
/**
 * KMS Keys start with a default key policy that grants the account access to administer the key,
 * mirroring the behavior of the KMS SDK/CLI/Console experience. Users may override the default key
 * policy by specifying their own.
 *
 * If this flag is not set, the default key policy depends on the setting of the `trustAccountIdentities`
 * flag. If false (the default, for backwards-compatibility reasons), the default key policy somewhat
 * resemebles the default admin key policy, but with the addition of 'GenerateDataKey' permissions. If
 * true, the policy matches what happens when this feature flag is set.
 *
 * Additionally, if this flag is not set and the user supplies a custom key policy, this will be appended
 * to the key's default policy (rather than replacing it).
 */
exports.KMS_DEFAULT_KEY_POLICIES = '@aws-cdk/aws-kms:defaultKeyPolicies';
/**
 * Change the old 's3:PutObject*' permission to 's3:PutObject' on Bucket,
 * as the former includes 's3:PutObjectAcl',
 * which could be used to grant read/write object access to IAM principals in other accounts.
 * Use a feature flag to make sure existing customers who might be relying
 * on the overly-broad permissions are not broken.
 */
exports.S3_GRANT_WRITE_WITHOUT_ACL = '@aws-cdk/aws-s3:grantWriteWithoutAcl';
/**
 * ApplicationLoadBalancedServiceBase, ApplicationMultipleTargetGroupServiceBase,
 * NetworkLoadBalancedServiceBase, NetworkMultipleTargetGroupServiceBase, and
 * QueueProcessingServiceBase currently determine a default value for the desired count of
 * a CfnService if a desiredCount is not provided.
 *
 * If this flag is not set, the default behaviour for CfnService.desiredCount is to set a
 * desiredCount of 1, if one is not provided. If true, a default will not be defined for
 * CfnService.desiredCount and as such desiredCount will be undefined, if one is not provided.
 *
 * This is a feature flag as the old behavior was technically incorrect, but
 * users may have come to depend on it.
 */
exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT = '@aws-cdk/aws-ecs-patterns:removeDefaultDesiredCount';
/**
 * ServerlessCluster.clusterIdentifier currently can has uppercase letters,
 * and ServerlessCluster pass it through to CfnDBCluster.dbClusterIdentifier.
 * The identifier is saved as lowercase string in AWS and is resolved as original string in CloudFormation.
 *
 * If this flag is not set, original value that one set to ServerlessCluster.clusterIdentifier
 * is passed to CfnDBCluster.dbClusterIdentifier.
 * If this flag is true, ServerlessCluster.clusterIdentifier is converted into a string containing
 * only lowercase characters by the `toLowerCase` function and passed to CfnDBCluster.dbClusterIdentifier.
 *
 * This feature flag make correct the ServerlessCluster.clusterArn when
 * clusterIdentifier contains a Upper case letters.
 *
 * [PERMANENT]
 */
exports.RDS_LOWERCASE_DB_IDENTIFIER = '@aws-cdk/aws-rds:lowercaseDbIdentifier';
/**
 * The UsagePlanKey resource connects an ApiKey with a UsagePlan. API Gateway does not allow more than one UsagePlanKey
 * for any given UsagePlan and ApiKey combination. For this reason, CloudFormation cannot replace this resource without
 * either the UsagePlan or ApiKey changing.
 *
 * The feature addition to support multiple UsagePlanKey resources - 142bd0e2 - recognized this and attempted to keep
 * existing UsagePlanKey logical ids unchanged.
 * However, this intentionally caused the logical id of the UsagePlanKey to be sensitive to order. That is, when
 * the 'first' UsagePlanKey resource is removed, the logical id of the 'second' assumes what was originally the 'first',
 * which again is disallowed.
 *
 * In effect, there is no way to get out of this mess in a backwards compatible way, while supporting existing stacks.
 * This flag changes the logical id layout of UsagePlanKey to not be sensitive to order.
 *
 * [PERMANENT]
 */
exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID = '@aws-cdk/aws-apigateway:usagePlanKeyOrderInsensitiveId';
/**
 * Enable this feature flag to have elastic file systems encrypted at rest by default.
 *
 * Encryption can also be configured explicitly using the `encrypted` property.
 */
exports.EFS_DEFAULT_ENCRYPTION_AT_REST = '@aws-cdk/aws-efs:defaultEncryptionAtRest';
/**
 * Enable this feature flag to opt in to the updated logical id calculation for Lambda Version created using the
 * `fn.currentVersion`.
 *
 * The previous calculation incorrectly considered properties of the `AWS::Lambda::Function` resource that did
 * not constitute creating a new Version.
 *
 * See 'currentVersion' section in the aws-lambda module's README for more details.
 *
 * [PERMANENT]
 */
exports.LAMBDA_RECOGNIZE_VERSION_PROPS = '@aws-cdk/aws-lambda:recognizeVersionProps';
/**
 * Enable this feature flag to have cloudfront distributions use the security policy TLSv1.2_2021 by default.
 *
 * The security policy can also be configured explicitly using the `minimumProtocolVersion` property.
 *
 * [PERMANENT]
 */
exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021 = '@aws-cdk/aws-cloudfront:defaultSecurityPolicyTLSv1.2_2021';
/**
 * Enable this flag to make it impossible to accidentally use SecretValues in unsafe locations
 *
 * With this flag enabled, `SecretValue` instances can only be passed to
 * constructs that accept `SecretValue`s; otherwise, `unsafeUnwrap()` must be
 * called to use it as a regular string.
 */
exports.CHECK_SECRET_USAGE = '@aws-cdk/core:checkSecretUsage';
/**
 * What regions to include in lookup tables of environment agnostic stacks
 *
 * Has no effect on stacks that have a defined region, but will limit the amount
 * of unnecessary regions included in stacks without a known region.
 *
 * The type of this value should be a list of strings.
 *
 * [PERMANENT]
 */
exports.TARGET_PARTITIONS = '@aws-cdk/core:target-partitions';
/**
 * Enable this feature flag to configure default logging behavior for the ECS Service Extensions. This will enable the
 * `awslogs` log driver for the application container of the service to send the container logs to CloudWatch Logs.
 *
 * This is a feature flag as the new behavior provides a better default experience for the users.
 *
 * [PERMANENT]
 */
exports.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER = '@aws-cdk-containers/ecs-service-extensions:enableDefaultLogDriver';
/**
 * Enable this feature flag to have Launch Templates generated by the `InstanceRequireImdsv2Aspect` use unique names.
 *
 * Previously, the generated Launch Template names were only unique within a stack because they were based only on the
 * `Instance` construct ID. If another stack that has an `Instance` with the same construct ID is deployed in the same
 * account and region, the deployments would always fail as the generated Launch Template names were the same.
 *
 * The new implementation addresses this issue by generating the Launch Template name with the `Names.uniqueId` method.
 */
exports.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME = '@aws-cdk/aws-ec2:uniqueImdsv2TemplateName';
/**
 * Minimize IAM policies by combining Principals, Actions and Resources of two
 * Statements in the policies, as long as it doesn't change the meaning of the
 * policy.
 *
 * [PERMANENT]
 */
exports.IAM_MINIMIZE_POLICIES = '@aws-cdk/aws-iam:minimizePolicies';
/**
 * Flag values that should apply for new projects
 *
 * Add a flag in here (typically with the value `true`), to enable
 * backwards-breaking behavior changes only for new projects.  New projects
 * generated through `cdk init` will include these flags in their generated
 *
 * Tests must cover the default (disabled) case and the future (enabled) case.
 */
exports.FUTURE_FLAGS = {
    [exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID]: true,
    [exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: true,
    [exports.ENABLE_DIFF_NO_FAIL_CONTEXT]: true,
    [exports.STACK_RELATIVE_EXPORTS_CONTEXT]: true,
    [exports.DOCKER_IGNORE_SUPPORT]: true,
    [exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME]: true,
    [exports.KMS_DEFAULT_KEY_POLICIES]: true,
    [exports.S3_GRANT_WRITE_WITHOUT_ACL]: true,
    [exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT]: true,
    [exports.RDS_LOWERCASE_DB_IDENTIFIER]: true,
    [exports.EFS_DEFAULT_ENCRYPTION_AT_REST]: true,
    [exports.LAMBDA_RECOGNIZE_VERSION_PROPS]: true,
    [exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021]: true,
    [exports.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER]: true,
    [exports.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME]: true,
    [exports.CHECK_SECRET_USAGE]: true,
    [exports.IAM_MINIMIZE_POLICIES]: true,
};
/**
 * Values that will be set by default in a new project, which are not necessarily booleans (and don't expire)
 */
exports.NEW_PROJECT_DEFAULT_CONTEXT = {
    [exports.TARGET_PARTITIONS]: ['aws', 'aws-cn'],
};
/**
 * The list of future flags that are now expired. This is going to be used to identify
 * and block usages of old feature flags in the new major version of CDK.
 */
exports.FUTURE_FLAGS_EXPIRED = [];
/**
 * The default values of each of these flags.
 *
 * This is the effective value of the flag, unless it's overriden via
 * context.
 *
 * Adding new flags here is only allowed during the pre-release period of a new
 * major version!
 */
const FUTURE_FLAGS_DEFAULTS = {};
function futureFlagDefault(flag) {
    var _a;
    return (_a = FUTURE_FLAGS_DEFAULTS[flag]) !== null && _a !== void 0 ? _a : false;
}
exports.futureFlagDefault = futureFlagDefault;
//# sourceMappingURL=data:application/json;base64,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