"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateSchema = exports.bump = exports.update = exports.SCHEMAS = void 0;
const fs = require("fs");
const path = require("path");
const semver = require("semver");
// eslint-disable-next-line import/no-extraneous-dependencies
const tjs = require("typescript-json-schema");
function log(message) {
    // eslint-disable-next-line no-console
    console.log(message);
}
/**
 * Where schemas are committed.
 */
const SCHEMA_DIR = path.resolve(__dirname, '../schema');
const SCHEMA_DEFINITIONS = {
    'assets': { rootTypeName: 'AssetManifest' },
    'cloud-assembly': { rootTypeName: 'AssemblyManifest' },
    'integ': { rootTypeName: 'IntegManifest' },
};
exports.SCHEMAS = Object.keys(SCHEMA_DEFINITIONS);
function update() {
    for (const s of exports.SCHEMAS) {
        generateSchema(s);
    }
    bump();
}
exports.update = update;
function bump() {
    const versionFile = path.join(SCHEMA_DIR, 'cloud-assembly.version.json');
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const metadata = require(versionFile);
    const oldVersion = metadata.version;
    const newVersion = semver.inc(oldVersion, 'major');
    log(`Updating schema version: ${oldVersion} -> ${newVersion}`);
    fs.writeFileSync(versionFile, JSON.stringify({ version: newVersion }));
}
exports.bump = bump;
/**
 * Generates a schema from typescript types.
 * @returns JSON schema
 * @param schemaName the schema to generate
 * @param shouldBump writes a new version of the schema and bumps the major version
 */
function generateSchema(schemaName, saveToFile = true) {
    const spec = SCHEMA_DEFINITIONS[schemaName];
    const out = saveToFile ? path.join(SCHEMA_DIR, `${schemaName}.schema.json`) : '';
    const settings = {
        required: true,
        ref: true,
        topRef: true,
        noExtraProps: false,
        out,
    };
    const compilerOptions = {
        strictNullChecks: true,
    };
    const program = tjs.getProgramFromFiles([path.join(__dirname, '../lib/index.d.ts')], compilerOptions);
    const schema = tjs.generateSchema(program, spec.rootTypeName, settings);
    augmentDescription(schema);
    addAnyMetadataEntry(schema);
    if (out) {
        log(`Generating schema to ${out}`);
        fs.writeFileSync(out, JSON.stringify(schema, null, 4));
    }
    return schema;
}
exports.generateSchema = generateSchema;
/**
 * Remove 'default' from the schema since its generated
 * from the tsdocs, which are not necessarily actual values,
 * but rather descriptive behavior.
 *
 * To keep this inforamtion in the schema, we append it to the
 * 'description' of the property.
 */
function augmentDescription(schema) {
    function _recurse(o) {
        for (const prop in o) {
            if (prop === 'description' && typeof o[prop] === 'string') {
                const description = o[prop];
                const defaultValue = o.default;
                if (!defaultValue) {
                    // property doesn't have a default value
                    // skip
                    continue;
                }
                const descriptionWithDefault = `${description} (Default ${defaultValue})`;
                delete o.default;
                o[prop] = descriptionWithDefault;
            }
            else if (typeof o[prop] === 'object') {
                _recurse(o[prop]);
            }
        }
    }
    _recurse(schema);
}
/**
 * Patch the properties of MetadataEntry to allow
 * specifying any free form data. This is needed since source
 * code doesn't allow this in order to enforce stricter jsii
 * compatibility checks.
 */
function addAnyMetadataEntry(schema) {
    var _a;
    (_a = schema.definitions.MetadataEntry) === null || _a === void 0 ? void 0 : _a.properties.data.anyOf.push({ description: 'Free form data.' });
}
//# sourceMappingURL=data:application/json;base64,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