"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FactName = exports.Fact = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_entities_1 = require("./aws-entities");
/**
 * A database of regional information.
 */
class Fact {
    constructor() {
        throw new Error('Use the static methods of Fact instead!');
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     *          may not be an exhaustive list of all available AWS regions.
     */
    static get regions() {
        // Return by copy to ensure no modifications can be made to the undelying constant.
        return Array.from(aws_entities_1.AWS_REGIONS);
    }
    /**
     * Retrieves a fact from this Fact database.
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name   the name of the fact being looked up (see the `FactName` class for details)
     *
     * @returns the fact value if it is known, and `undefined` otherwise.
     */
    static find(region, name) {
        const regionFacts = this.database[region];
        return regionFacts && regionFacts[name];
    }
    /**
     * Retrieve a fact from the Fact database. (retrieval will fail if the specified region or
     * fact name does not exist.)
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name the name of the fact being looked up (see the `FactName` class for details)
     */
    static requireFact(region, name) {
        const foundFact = this.find(region, name);
        if (!foundFact) {
            throw new Error(`No fact ${name} could be found for region: ${region} and name: ${name}`);
        }
        return foundFact;
    }
    /**
     * Registers a new fact in this Fact database.
     *
     * @param fact           the new fact to be registered.
     * @param allowReplacing whether new facts can replace existing facts or not.
     */
    static register(fact, allowReplacing = false) {
        try {
            jsiiDeprecationWarnings._aws_cdk_region_info_IFact(fact);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.register);
            }
            throw error;
        }
        const regionFacts = this.database[fact.region] || (this.database[fact.region] = {});
        if (fact.name in regionFacts && regionFacts[fact.name] !== fact.value && !allowReplacing) {
            throw new Error(`Region ${fact.region} already has a fact ${fact.name}, with value ${regionFacts[fact.name]}`);
        }
        if (fact.value !== undefined) {
            regionFacts[fact.name] = fact.value;
        }
    }
    /**
     * Removes a fact from the database.
     * @param region the region for which the fact is to be removed.
     * @param name   the name of the fact to remove.
     * @param value  the value that should be removed (removal will fail if the value is specified, but does not match the
     *               current stored value).
     */
    static unregister(region, name, value) {
        const regionFacts = this.database[region] || {};
        if (name in regionFacts && value && regionFacts[name] !== value) {
            throw new Error(`Attempted to remove ${name} from ${region} with value ${value}, but the fact's value is ${regionFacts[name]}`);
        }
        delete regionFacts[name];
    }
}
exports.Fact = Fact;
_a = JSII_RTTI_SYMBOL_1;
Fact[_a] = { fqn: "@aws-cdk/region-info.Fact", version: "1.154.0" };
Fact.database = {};
/**
 * All standardized fact names.
 */
class FactName {
    /**
     * The ARN of CloudWatch Lambda Insights for a version (e.g. 1.0.98.0)
     */
    static cloudwatchLambdaInsightsVersion(version, arch) {
        // if we are provided an architecture use that, otherwise
        // default to x86_64 for backwards compatibility
        const suffix = version.split('.').join('_') + `_${arch !== null && arch !== void 0 ? arch : 'x86_64'}`;
        return `cloudwatch-lambda-insights-version:${suffix}`;
    }
    /**
     * The name of the regional service principal for a given service.
     *
     * @param service the service name, either simple (e.g: `s3`, `codedeploy`) or qualified (e.g: `s3.amazonaws.com`).
     *                The `.amazonaws.com` and `.amazonaws.com.cn` domains are stripped from service names, so they are
     *                canonicalized in that respect.
     */
    static servicePrincipal(service) {
        return `service-principal:${service.replace(/\.amazonaws\.com(\.cn)?$/, '')}`;
    }
}
exports.FactName = FactName;
_b = JSII_RTTI_SYMBOL_1;
FactName[_b] = { fqn: "@aws-cdk/region-info.FactName", version: "1.154.0" };
/**
 * The name of the partition for a region (e.g: 'aws', 'aws-cn', ...)
 */
FactName.PARTITION = 'partition';
/**
 * The domain suffix for a region (e.g: 'amazonaws.com`)
 */
FactName.DOMAIN_SUFFIX = 'domainSuffix';
/**
 * Whether the AWS::CDK::Metadata CloudFormation Resource is available in-region or not. The value is a boolean
 * modelled as `YES` or `NO`.
 */
FactName.CDK_METADATA_RESOURCE_AVAILABLE = 'cdk:metadata-resource:available';
/**
 * The endpoint used for hosting S3 static websites
 */
FactName.S3_STATIC_WEBSITE_ENDPOINT = 's3-static-website:endpoint';
/**
 * The endpoint used for aliasing S3 static websites in Route 53
 */
FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID = 's3-static-website:route-53-hosted-zone-id';
/**
 * The hosted zone ID used by Route 53 to alias a EBS environment endpoint in this region (e.g: Z2O1EMRO9K5GLX)
 */
FactName.EBS_ENV_ENDPOINT_HOSTED_ZONE_ID = 'ebs-environment:route-53-hosted-zone-id';
/**
 * The prefix for VPC Endpoint Service names,
 * cn.com.amazonaws.vpce for China regions,
 * com.amazonaws.vpce otherwise.
 */
FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'vpcEndpointServiceNamePrefix';
/**
 * The account for ELBv2 in this region
 */
FactName.ELBV2_ACCOUNT = 'elbv2Account';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the
 * AWS Deep Learning Containers images in a given region.
 */
FactName.DLC_REPOSITORY_ACCOUNT = 'dlcRepositoryAccount';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the
 * AWS App Mesh Envoy Proxy images in a given region.
 */
FactName.APPMESH_ECR_ACCOUNT = 'appMeshRepositoryAccount';
/**
 * The CIDR block used by Kinesis Data Firehose servers.
 */
FactName.FIREHOSE_CIDR_BLOCK = 'firehoseCidrBlock';
//# sourceMappingURL=data:application/json;base64,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