"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatTable = void 0;
const chalk = require("chalk");
const string_width_1 = require("string-width");
const table = require("table");
/**
 * Render a two-dimensional array to a visually attractive table
 *
 * First row is considered the table header.
 */
function formatTable(cells, columns) {
    return table.table(cells, {
        border: TABLE_BORDER_CHARACTERS,
        columns: buildColumnConfig(columns !== undefined ? calculcateColumnWidths(cells, columns) : undefined),
        drawHorizontalLine: (line) => {
            // Numbering like this: [line 0] [header = row[0]] [line 1] [row 1] [line 2] [content 2] [line 3]
            return (line < 2 || line === cells.length) || lineBetween(cells[line - 1], cells[line]);
        },
    }).trimRight();
}
exports.formatTable = formatTable;
/**
 * Whether we should draw a line between two rows
 *
 * Draw horizontal line if 2nd column values are different.
 */
function lineBetween(rowA, rowB) {
    return rowA[1] !== rowB[1];
}
function buildColumnConfig(widths) {
    if (widths === undefined) {
        return undefined;
    }
    const ret = {};
    widths.forEach((width, i) => {
        if (width === undefined) {
            return;
        }
        ret[i] = { width };
    });
    return ret;
}
/**
 * Calculate column widths given a terminal width
 *
 * We do this by calculating a fair share for every column. Extra width smaller
 * than the fair share is evenly distributed over all columns that exceed their
 * fair share.
 */
function calculcateColumnWidths(rows, terminalWidth) {
    // The terminal is sometimes reported to be 0. Also if the terminal is VERY narrow,
    // just assume a reasonable minimum size.
    terminalWidth = Math.max(terminalWidth, 40);
    // use 'string-width' to not count ANSI chars as actual character width
    const columns = rows[0].map((_, i) => Math.max(...rows.map(row => string_width_1.default(String(row[i])))));
    // If we have no terminal width, do nothing
    const contentWidth = terminalWidth - 2 - columns.length * 3;
    // If we don't exceed the terminal width, do nothing
    if (sum(columns) <= contentWidth) {
        return columns;
    }
    const fairShare = Math.min(contentWidth / columns.length);
    const smallColumns = columns.filter(w => w < fairShare);
    let distributableWidth = contentWidth - sum(smallColumns);
    const fairDistributable = Math.floor(distributableWidth / (columns.length - smallColumns.length));
    const ret = new Array();
    for (const requestedWidth of columns) {
        if (requestedWidth < fairShare) {
            // Small column gets what they want
            ret.push(requestedWidth);
        }
        else {
            // Last column gets all remaining, otherwise get fair redist share
            const width = distributableWidth < 2 * fairDistributable ? distributableWidth : fairDistributable;
            ret.push(width);
            distributableWidth -= width;
        }
    }
    return ret;
}
function sum(xs) {
    let total = 0;
    for (const x of xs) {
        total += x;
    }
    return total;
}
// What color the table is going to be
const tableColor = chalk.gray;
// Unicode table characters with a color
const TABLE_BORDER_CHARACTERS = {
    topBody: tableColor('─'),
    topJoin: tableColor('┬'),
    topLeft: tableColor('┌'),
    topRight: tableColor('┐'),
    bottomBody: tableColor('─'),
    bottomJoin: tableColor('┴'),
    bottomLeft: tableColor('└'),
    bottomRight: tableColor('┘'),
    bodyLeft: tableColor('│'),
    bodyRight: tableColor('│'),
    bodyJoin: tableColor('│'),
    joinBody: tableColor('─'),
    joinLeft: tableColor('├'),
    joinRight: tableColor('┤'),
    joinJoin: tableColor('┼'),
};
//# sourceMappingURL=data:application/json;base64,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