"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Default = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_entities_1 = require("./aws-entities");
/**
 * Provides default values for certain regional information points.
 */
class Default {
    constructor() { }
    /**
     * Computes a "standard" AWS Service principal for a given service, region and suffix. This is useful for example when
     * you need to compute a service principal name, but you do not have a synthesize-time region literal available (so
     * all you have is `{ "Ref": "AWS::Region" }`). This way you get the same defaulting behavior that is normally used
     * for built-in data.
     *
     * @param serviceFqn the name of the service (s3, s3.amazonaws.com, ...)
     * @param region    the region in which the service principal is needed.
     * @param urlSuffix deprecated and ignored.
     */
    static servicePrincipal(serviceFqn, region, urlSuffix) {
        const serviceName = extractSimpleName(serviceFqn);
        if (!serviceName) {
            // Return "service" if it does not look like any of the following:
            // - s3
            // - s3.amazonaws.com
            // - s3.amazonaws.com.cn
            // - s3.c2s.ic.gov
            // - s3.sc2s.sgov.gov
            return serviceFqn;
        }
        function determineConfiguration(service) {
            function universal(s) { return `${s}.amazonaws.com`; }
            function partitional(s, _, u) { return `${s}.${u}`; }
            function regional(s, r) { return `${s}.${r}.amazonaws.com`; }
            function regionalPartitional(s, r, u) { return `${s}.${r}.${u}`; }
            // Exceptions for Service Principals in us-iso-*
            const US_ISO_EXCEPTIONS = new Set([
                'cloudhsm',
                'config',
                'states',
                'workspaces',
            ]);
            // Account for idiosyncratic Service Principals in `us-iso-*` regions
            if (region.startsWith('us-iso-') && US_ISO_EXCEPTIONS.has(service)) {
                switch (service) {
                    // Services with universal principal
                    case ('states'):
                        return universal;
                    // Services with a partitional principal
                    default:
                        return partitional;
                }
            }
            // Exceptions for Service Principals in us-isob-*
            const US_ISOB_EXCEPTIONS = new Set([
                'dms',
                'states',
            ]);
            // Account for idiosyncratic Service Principals in `us-isob-*` regions
            if (region.startsWith('us-isob-') && US_ISOB_EXCEPTIONS.has(service)) {
                switch (service) {
                    // Services with universal principal
                    case ('states'):
                        return universal;
                    // Services with a partitional principal
                    default:
                        return partitional;
                }
            }
            switch (service) {
                // SSM turned from global to regional at some point
                case 'ssm':
                    return aws_entities_1.before(region, aws_entities_1.RULE_SSM_PRINCIPALS_ARE_REGIONAL)
                        ? universal
                        : regional;
                // CodeDeploy is regional+partitional in CN, only regional everywhere else
                case 'codedeploy':
                    return region.startsWith('cn-')
                        ? regionalPartitional
                        // ...except in the isolated regions, where it's universal
                        : (region.startsWith('us-iso') ? universal : regional);
                // Services with a regional AND partitional principal
                case 'logs':
                    return regionalPartitional;
                // Services with a regional principal
                case 'states':
                    return regional;
                // Services with a partitional principal
                case 'ec2':
                    return partitional;
                // Services with a universal principal across all regions/partitions (the default case)
                default:
                    return universal;
            }
        }
        ;
        const configuration = determineConfiguration(serviceName);
        return configuration(serviceName, region, urlSuffix);
    }
}
exports.Default = Default;
_a = JSII_RTTI_SYMBOL_1;
Default[_a] = { fqn: "@aws-cdk/region-info.Default", version: "1.153.1" };
/**
 * The default value for a VPC Endpoint Service name prefix, useful if you do
 * not have a synthesize-time region literal available (all you have is
 * `{ "Ref": "AWS::Region" }`)
 */
Default.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'com.amazonaws.vpce';
function extractSimpleName(serviceFqn) {
    const matches = serviceFqn.match(/^([^.]+)(?:(?:\.amazonaws\.com(?:\.cn)?)|(?:\.c2s\.ic\.gov)|(?:\.sc2s\.sgov\.gov))?$/);
    return matches ? matches[1] : undefined;
}
//# sourceMappingURL=data:application/json;base64,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