"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretValue = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cfn_dynamic_reference_1 = require("./cfn-dynamic-reference");
const intrinsic_1 = require("./private/intrinsic");
const token_1 = require("./token");
/**
 * Work with secret values in the CDK
 *
 * Secret values in the CDK (such as those retrieved from SecretsManager) are
 * represented as regular strings, just like other values that are only
 * available at deployment time.
 *
 * To help you avoid accidental mistakes which would lead to you putting your
 * secret values directly into a CloudFormation template, constructs that take
 * secret values will not allow you to pass in a literal secret value. They do
 * so by calling `Secret.assertSafeSecret()`.
 *
 * You can escape the check by calling `Secret.plainText()`, but doing
 * so is highly discouraged.
 */
class SecretValue extends intrinsic_1.Intrinsic {
    /**
     * Construct a literal secret value for use with secret-aware constructs
     *
     * *Do not use this method for any secrets that you care about.*
     *
     * The only reasonable use case for using this method is when you are testing.
     */
    static plainText(secret) {
        return new SecretValue(secret);
    }
    /**
     * Creates a `SecretValue` with a value which is dynamically loaded from AWS Secrets Manager.
     * @param secretId The ID or ARN of the secret
     * @param options Options
     */
    static secretsManager(secretId, options = {}) {
        jsiiDeprecationWarnings._aws_cdk_core_SecretsManagerSecretOptions(options);
        if (!secretId) {
            throw new Error('secretId cannot be empty');
        }
        if (!token_1.Token.isUnresolved(secretId) && !secretId.startsWith('arn:') && secretId.includes(':')) {
            throw new Error(`secret id "${secretId}" is not an ARN but contains ":"`);
        }
        if (options.versionStage && options.versionId) {
            throw new Error(`verionStage: '${options.versionStage}' and versionId: '${options.versionId}' were both provided but only one is allowed`);
        }
        const parts = [
            secretId,
            'SecretString',
            options.jsonField || '',
            options.versionStage || '',
            options.versionId || '',
        ];
        const dyref = new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SECRETS_MANAGER, parts.join(':'));
        return this.cfnDynamicReference(dyref);
    }
    /**
     * Use a secret value stored from a Systems Manager (SSM) parameter.
     *
     * @param parameterName The name of the parameter in the Systems Manager
     * Parameter Store. The parameter name is case-sensitive.
     *
     * @param version An integer that specifies the version of the parameter to
     * use. If you don't specify the exact version, AWS CloudFormation uses the
     * latest version of the parameter.
     */
    static ssmSecure(parameterName, version) {
        return this.cfnDynamicReference(new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SSM_SECURE, version ? `${parameterName}:${version}` : parameterName));
    }
    /**
     * Obtain the secret value through a CloudFormation dynamic reference.
     *
     * If possible, use `SecretValue.ssmSecure` or `SecretValue.secretsManager` directly.
     *
     * @param ref The dynamic reference to use.
     */
    static cfnDynamicReference(ref) {
        jsiiDeprecationWarnings._aws_cdk_core_CfnDynamicReference(ref);
        return new SecretValue(ref);
    }
    /**
     * Obtain the secret value through a CloudFormation parameter.
     *
     * Generally, this is not a recommended approach. AWS Secrets Manager is the
     * recommended way to reference secrets.
     *
     * @param param The CloudFormation parameter to use.
     */
    static cfnParameter(param) {
        jsiiDeprecationWarnings._aws_cdk_core_CfnParameter(param);
        if (!param.noEcho) {
            throw new Error('CloudFormation parameter must be configured with "NoEcho"');
        }
        return new SecretValue(param.value);
    }
}
exports.SecretValue = SecretValue;
_a = JSII_RTTI_SYMBOL_1;
SecretValue[_a] = { fqn: "@aws-cdk/core.SecretValue", version: "1.152.0" };
//# sourceMappingURL=data:application/json;base64,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