"use strict";
// --------------------------------------------------------------------------------
// This file defines context keys that enable certain features that are
// implemented behind a flag in order to preserve backwards compatibility for
// existing apps. When a new app is initialized through `cdk init`, the CLI will
// automatically add enable these features by adding them to the generated
// `cdk.json` file.
//
// Some of these flags only affect the behavior of the construct library --
// these will be removed in the next major release and the behavior they are
// gating will become the only behavior.
//
// Other flags also affect the generated CloudFormation templates, in a way
// that prevents seamless upgrading. In the next major version, their
// behavior will become the default, but the flag still exists so users can
// switch it *off* in order to revert to the old behavior. These flags
// are marked with with the [PERMANENT] tag below.
//
// See https://github.com/aws/aws-cdk-rfcs/blob/master/text/0055-feature-flags.md
// --------------------------------------------------------------------------------
Object.defineProperty(exports, "__esModule", { value: true });
exports.futureFlagDefault = exports.FUTURE_FLAGS_EXPIRED = exports.NEW_PROJECT_DEFAULT_CONTEXT = exports.FUTURE_FLAGS = exports.IAM_MINIMIZE_POLICIES = exports.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME = exports.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER = exports.TARGET_PARTITIONS = exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021 = exports.LAMBDA_RECOGNIZE_VERSION_PROPS = exports.EFS_DEFAULT_ENCRYPTION_AT_REST = exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID = exports.RDS_LOWERCASE_DB_IDENTIFIER = exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT = exports.S3_GRANT_WRITE_WITHOUT_ACL = exports.KMS_DEFAULT_KEY_POLICIES = exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = exports.DOCKER_IGNORE_SUPPORT = exports.STACK_RELATIVE_EXPORTS_CONTEXT = exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT = exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = void 0;
/**
 * If this is set, multiple stacks can use the same stack name (e.g. deployed to
 * different environments). This means that the name of the synthesized template
 * file will be based on the construct path and not on the defined `stackName`
 * of the stack.
 *
 * This is a "future flag": the feature is disabled by default for backwards
 * compatibility, but new projects created using `cdk init` will have this
 * enabled through the generated `cdk.json`.
 */
exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = '@aws-cdk/core:enableStackNameDuplicates';
/**
 * IF this is set, `cdk diff` will always exit with 0.
 *
 * Use `cdk diff --fail` to exit with 1 if there's a diff.
 */
exports.ENABLE_DIFF_NO_FAIL_CONTEXT = 'aws-cdk:enableDiffNoFail';
/** @deprecated use `ENABLE_DIFF_NO_FAIL_CONTEXT` */
exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT;
/**
 * Switch to new stack synthesis method which enable CI/CD
 *
 * [PERMANENT]
 */
exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = '@aws-cdk/core:newStyleStackSynthesis';
/**
 * Name exports based on the construct paths relative to the stack, rather than the global construct path
 *
 * Combined with the stack name this relative construct path is good enough to
 * ensure uniqueness, and makes the export names robust against refactoring
 * the location of the stack in the construct tree (specifically, moving the Stack
 * into a Stage).
 *
 * [PERMANENT]
 */
exports.STACK_RELATIVE_EXPORTS_CONTEXT = '@aws-cdk/core:stackRelativeExports';
/**
 * DockerImageAsset properly supports `.dockerignore` files by default
 *
 * If this flag is not set, the default behavior for `DockerImageAsset` is to use
 * glob semantics for `.dockerignore` files. If this flag is set, the default behavior
 * is standard Docker ignore semantics.
 *
 * This is a feature flag as the old behavior was technically incorrect but
 * users may have come to depend on it.
 */
exports.DOCKER_IGNORE_SUPPORT = '@aws-cdk/aws-ecr-assets:dockerIgnoreSupport';
/**
 * Secret.secretName for an "owned" secret will attempt to parse the secretName from the ARN,
 * rather than the default full resource name, which includes the SecretsManager suffix.
 *
 * If this flag is not set, Secret.secretName will include the SecretsManager suffix, which cannot be directly
 * used by SecretsManager.DescribeSecret, and must be parsed by the user first (e.g., Fn:Join, Fn:Select, Fn:Split).
 */
exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = '@aws-cdk/aws-secretsmanager:parseOwnedSecretName';
/**
 * KMS Keys start with a default key policy that grants the account access to administer the key,
 * mirroring the behavior of the KMS SDK/CLI/Console experience. Users may override the default key
 * policy by specifying their own.
 *
 * If this flag is not set, the default key policy depends on the setting of the `trustAccountIdentities`
 * flag. If false (the default, for backwards-compatibility reasons), the default key policy somewhat
 * resemebles the default admin key policy, but with the addition of 'GenerateDataKey' permissions. If
 * true, the policy matches what happens when this feature flag is set.
 *
 * Additionally, if this flag is not set and the user supplies a custom key policy, this will be appended
 * to the key's default policy (rather than replacing it).
 */
exports.KMS_DEFAULT_KEY_POLICIES = '@aws-cdk/aws-kms:defaultKeyPolicies';
/**
 * Change the old 's3:PutObject*' permission to 's3:PutObject' on Bucket,
 * as the former includes 's3:PutObjectAcl',
 * which could be used to grant read/write object access to IAM principals in other accounts.
 * Use a feature flag to make sure existing customers who might be relying
 * on the overly-broad permissions are not broken.
 */
exports.S3_GRANT_WRITE_WITHOUT_ACL = '@aws-cdk/aws-s3:grantWriteWithoutAcl';
/**
 * ApplicationLoadBalancedServiceBase, ApplicationMultipleTargetGroupServiceBase,
 * NetworkLoadBalancedServiceBase, NetworkMultipleTargetGroupServiceBase, and
 * QueueProcessingServiceBase currently determine a default value for the desired count of
 * a CfnService if a desiredCount is not provided.
 *
 * If this flag is not set, the default behaviour for CfnService.desiredCount is to set a
 * desiredCount of 1, if one is not provided. If true, a default will not be defined for
 * CfnService.desiredCount and as such desiredCount will be undefined, if one is not provided.
 *
 * This is a feature flag as the old behavior was technically incorrect, but
 * users may have come to depend on it.
 */
exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT = '@aws-cdk/aws-ecs-patterns:removeDefaultDesiredCount';
/**
 * ServerlessCluster.clusterIdentifier currently can has uppercase letters,
 * and ServerlessCluster pass it through to CfnDBCluster.dbClusterIdentifier.
 * The identifier is saved as lowercase string in AWS and is resolved as original string in CloudFormation.
 *
 * If this flag is not set, original value that one set to ServerlessCluster.clusterIdentifier
 * is passed to CfnDBCluster.dbClusterIdentifier.
 * If this flag is true, ServerlessCluster.clusterIdentifier is converted into a string containing
 * only lowercase characters by the `toLowerCase` function and passed to CfnDBCluster.dbClusterIdentifier.
 *
 * This feature flag make correct the ServerlessCluster.clusterArn when
 * clusterIdentifier contains a Upper case letters.
 *
 * [PERMANENT]
 */
exports.RDS_LOWERCASE_DB_IDENTIFIER = '@aws-cdk/aws-rds:lowercaseDbIdentifier';
/**
 * The UsagePlanKey resource connects an ApiKey with a UsagePlan. API Gateway does not allow more than one UsagePlanKey
 * for any given UsagePlan and ApiKey combination. For this reason, CloudFormation cannot replace this resource without
 * either the UsagePlan or ApiKey changing.
 *
 * The feature addition to support multiple UsagePlanKey resources - 142bd0e2 - recognized this and attempted to keep
 * existing UsagePlanKey logical ids unchanged.
 * However, this intentionally caused the logical id of the UsagePlanKey to be sensitive to order. That is, when
 * the 'first' UsagePlanKey resource is removed, the logical id of the 'second' assumes what was originally the 'first',
 * which again is disallowed.
 *
 * In effect, there is no way to get out of this mess in a backwards compatible way, while supporting existing stacks.
 * This flag changes the logical id layout of UsagePlanKey to not be sensitive to order.
 *
 * [PERMANENT]
 */
exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID = '@aws-cdk/aws-apigateway:usagePlanKeyOrderInsensitiveId';
/**
 * Enable this feature flag to have elastic file systems encrypted at rest by default.
 *
 * Encryption can also be configured explicitly using the `encrypted` property.
 */
exports.EFS_DEFAULT_ENCRYPTION_AT_REST = '@aws-cdk/aws-efs:defaultEncryptionAtRest';
/**
 * Enable this feature flag to opt in to the updated logical id calculation for Lambda Version created using the
 * `fn.currentVersion`.
 *
 * The previous calculation incorrectly considered properties of the `AWS::Lambda::Function` resource that did
 * not constitute creating a new Version.
 *
 * See 'currentVersion' section in the aws-lambda module's README for more details.
 *
 * [PERMANENT]
 */
exports.LAMBDA_RECOGNIZE_VERSION_PROPS = '@aws-cdk/aws-lambda:recognizeVersionProps';
/**
 * Enable this feature flag to have cloudfront distributions use the security policy TLSv1.2_2021 by default.
 *
 * The security policy can also be configured explicitly using the `minimumProtocolVersion` property.
 *
 * [PERMANENT]
 */
exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021 = '@aws-cdk/aws-cloudfront:defaultSecurityPolicyTLSv1.2_2021';
/**
 * What regions to include in lookup tables of environment agnostic stacks
 *
 * Has no effect on stacks that have a defined region, but will limit the amount
 * of unnecessary regions included in stacks without a known region.
 *
 * The type of this value should be a list of strings.
 *
 * [PERMANENT]
 */
exports.TARGET_PARTITIONS = '@aws-cdk/core:target-partitions';
/**
 * Enable this feature flag to configure default logging behavior for the ECS Service Extensions. This will enable the
 * `awslogs` log driver for the application container of the service to send the container logs to CloudWatch Logs.
 *
 * This is a feature flag as the new behavior provides a better default experience for the users.
 *
 * [PERMANENT]
 */
exports.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER = '@aws-cdk-containers/ecs-service-extensions:enableDefaultLogDriver';
/**
 * Enable this feature flag to have Launch Templates generated by the `InstanceRequireImdsv2Aspect` use unique names.
 *
 * Previously, the generated Launch Template names were only unique within a stack because they were based only on the
 * `Instance` construct ID. If another stack that has an `Instance` with the same construct ID is deployed in the same
 * account and region, the deployments would always fail as the generated Launch Template names were the same.
 *
 * The new implementation addresses this issue by generating the Launch Template name with the `Names.uniqueId` method.
 */
exports.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME = '@aws-cdk/aws-ec2:uniqueImdsv2TemplateName';
/**
 * Minimize IAM policies by combining Principals, Actions and Resources of two
 * Statements in the policies, as long as it doesn't change the meaning of the
 * policy.
 *
 * [PERMANENT]
 */
exports.IAM_MINIMIZE_POLICIES = '@aws-cdk/aws-iam:minimizePolicies';
/**
 * Flag values that should apply for new projects
 *
 * Add a flag in here (typically with the value `true`), to enable
 * backwards-breaking behavior changes only for new projects.  New projects
 * generated through `cdk init` will include these flags in their generated
 *
 * Tests must cover the default (disabled) case and the future (enabled) case.
 */
exports.FUTURE_FLAGS = {
    [exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID]: true,
    [exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: true,
    [exports.ENABLE_DIFF_NO_FAIL_CONTEXT]: true,
    [exports.STACK_RELATIVE_EXPORTS_CONTEXT]: true,
    [exports.DOCKER_IGNORE_SUPPORT]: true,
    [exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME]: true,
    [exports.KMS_DEFAULT_KEY_POLICIES]: true,
    [exports.S3_GRANT_WRITE_WITHOUT_ACL]: true,
    [exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT]: true,
    [exports.RDS_LOWERCASE_DB_IDENTIFIER]: true,
    [exports.EFS_DEFAULT_ENCRYPTION_AT_REST]: true,
    [exports.LAMBDA_RECOGNIZE_VERSION_PROPS]: true,
    [exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021]: true,
    [exports.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER]: true,
    [exports.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME]: true,
    [exports.IAM_MINIMIZE_POLICIES]: true,
};
/**
 * Values that will be set by default in a new project, which are not necessarily booleans (and don't expire)
 */
exports.NEW_PROJECT_DEFAULT_CONTEXT = {
    [exports.TARGET_PARTITIONS]: ['aws', 'aws-cn'],
};
/**
 * The list of future flags that are now expired. This is going to be used to identify
 * and block usages of old feature flags in the new major version of CDK.
 */
exports.FUTURE_FLAGS_EXPIRED = [];
/**
 * The default values of each of these flags.
 *
 * This is the effective value of the flag, unless it's overriden via
 * context.
 *
 * Adding new flags here is only allowed during the pre-release period of a new
 * major version!
 */
const FUTURE_FLAGS_DEFAULTS = {};
function futureFlagDefault(flag) {
    var _a;
    return (_a = FUTURE_FLAGS_DEFAULTS[flag]) !== null && _a !== void 0 ? _a : false;
}
exports.futureFlagDefault = futureFlagDefault;
//# sourceMappingURL=data:application/json;base64,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