"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Default = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_entities_1 = require("./aws-entities");
/**
 * Provides default values for certain regional information points.
 */
class Default {
    constructor() { }
    /**
     * Computes a "standard" AWS Service principal for a given service, region and suffix. This is useful for example when
     * you need to compute a service principal name, but you do not have a synthesize-time region literal available (so
     * all you have is `{ "Ref": "AWS::Region" }`). This way you get the same defaulting behavior that is normally used
     * for built-in data.
     *
     * @param serviceFqn the name of the service (s3, s3.amazonaws.com, ...)
     * @param region    the region in which the service principal is needed.
     * @param urlSuffix deprecated and ignored.
     */
    static servicePrincipal(serviceFqn, region, urlSuffix) {
        const serviceName = extractSimpleName(serviceFqn);
        if (!serviceName) {
            // Return "service" if it does not look like any of the following:
            // - s3
            // - s3.amazonaws.com
            // - s3.amazonaws.com.cn
            // - s3.c2s.ic.gov
            // - s3.sc2s.sgov.gov
            return serviceFqn;
        }
        function determineConfiguration(service) {
            function universal(s) { return `${s}.amazonaws.com`; }
            ;
            function partitional(s, _, u) { return `${s}.${u}`; }
            ;
            function regional(s, r) { return `${s}.${r}.amazonaws.com`; }
            ;
            function regionalPartitional(s, r, u) { return `${s}.${r}.${u}`; }
            ;
            // Exceptions for Service Principals in us-iso-*
            const US_ISO_EXCEPTIONS = new Set([
                'cloudhsm',
                'config',
                'states',
                'workspaces',
            ]);
            // Account for idiosyncratic Service Principals in `us-iso-*` regions
            if (region.startsWith('us-iso-') && US_ISO_EXCEPTIONS.has(service)) {
                switch (service) {
                    // Services with universal principal
                    case ('states'):
                        return universal;
                    // Services with a partitional principal
                    default:
                        return partitional;
                }
            }
            // Exceptions for Service Principals in us-isob-*
            const US_ISOB_EXCEPTIONS = new Set([
                'dms',
                'states',
            ]);
            // Account for idiosyncratic Service Principals in `us-isob-*` regions
            if (region.startsWith('us-isob-') && US_ISOB_EXCEPTIONS.has(service)) {
                switch (service) {
                    // Services with universal principal
                    case ('states'):
                        return universal;
                    // Services with a partitional principal
                    default:
                        return partitional;
                }
            }
            switch (service) {
                // SSM turned from global to regional at some point
                case 'ssm':
                    return aws_entities_1.before(region, aws_entities_1.RULE_SSM_PRINCIPALS_ARE_REGIONAL)
                        ? universal
                        : regional;
                // CodeDeploy is regional+partitional in CN, only regional everywhere else
                case 'codedeploy':
                    return region.startsWith('cn-')
                        ? regionalPartitional
                        : regional;
                // Services with a regional AND partitional principal
                case 'logs':
                    return regionalPartitional;
                // Services with a regional principal
                case 'states':
                    return regional;
                // Services with a partitional principal
                case 'ec2':
                    return partitional;
                // Services with a universal principal across all regions/partitions (the default case)
                default:
                    return universal;
            }
        }
        ;
        const configuration = determineConfiguration(serviceName);
        return configuration(serviceName, region, urlSuffix);
    }
}
exports.Default = Default;
_a = JSII_RTTI_SYMBOL_1;
Default[_a] = { fqn: "@aws-cdk/region-info.Default", version: "1.149.0" };
/**
 * The default value for a VPC Endpoint Service name prefix, useful if you do
 * not have a synthesize-time region literal available (all you have is
 * `{ "Ref": "AWS::Region" }`)
 */
Default.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'com.amazonaws.vpce';
function extractSimpleName(serviceFqn) {
    const matches = serviceFqn.match(/^([^.]+)(?:(?:\.amazonaws\.com(?:\.cn)?)|(?:\.c2s\.ic\.gov)|(?:\.sc2s\.sgov\.gov))?$/);
    return matches ? matches[1] : undefined;
}
//# sourceMappingURL=data:application/json;base64,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