"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const common_1 = require("../common");
const component_1 = require("../component");
const deps_1 = require("../deps");
const inventory_1 = require("../inventory");
const project_1 = require("../project");
/**
 * (experimental) Allows writing projenrc files in java.
 *
 * This will install `org.projen/projen` as a Maven dependency and will add a
 * `synth` task which will compile & execute `main()` from
 * `src/main/java/projenrc.java`.
 *
 * @experimental
 */
class Projenrc extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, pom, options = {}) {
        var _b, _c, _d;
        super(project);
        const projenVersion = (_b = options.projenVersion) !== null && _b !== void 0 ? _b : common_1.PROJEN_VERSION;
        this.className = (_c = options.className) !== null && _c !== void 0 ? _c : 'projenrc';
        this.testScope = (_d = options.testScope) !== null && _d !== void 0 ? _d : true;
        const depType = this.testScope ? deps_1.DependencyType.TEST : deps_1.DependencyType.RUNTIME;
        const execOpts = this.testScope ? ' -Dexec.classpathScope="test"' : '';
        const compileGoal = this.testScope ? 'compiler:testCompile' : 'compiler:compile';
        project.deps.addDependency(`com.github.eladb/projen@${projenVersion}`, depType);
        pom.addPlugin('org.codehaus.mojo/exec-maven-plugin@3.0.0');
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        const defaultTask = project.addTask(project_1.Project.DEFAULT_TASK, { description: 'Synthesize the project' });
        defaultTask.exec(`mvn ${compileGoal} --quiet`);
        defaultTask.exec(`mvn exec:java --quiet -Dexec.mainClass=${this.className}${execOpts}`);
        // if this is a new project, generate a skeleton for projenrc.java
        this.generateProjenrc();
    }
    generateProjenrc() {
        var _b, _c, _d;
        const bootstrap = this.project.newProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = inventory_1.readJsiiManifest(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const javaTarget = jsiiManifest.targets.java;
        const optionsTypeFqn = (_d = (_c = (_b = jsiiType.initializer) === null || _b === void 0 ? void 0 : _b.parameters) === null || _c === void 0 ? void 0 : _c[0].type) === null || _d === void 0 ? void 0 : _d.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn('cannot determine jsii type for project options');
            return;
        }
        const jsiiOptionsType = jsiiManifest.types[optionsTypeFqn];
        if (!jsiiOptionsType) {
            this.project.logger.warn(`cannot find jsii type for project options: ${optionsTypeFqn}`);
            return;
        }
        const dir = this.testScope ? 'src/test/java' : 'src/main/java';
        const split = this.className.split('.');
        let javaClass, javaPackage;
        if (split.length === 1) {
            javaClass = split[0];
            javaPackage = [];
        }
        else {
            javaPackage = split.slice(0, split.length - 2);
            javaClass = split[split.length - 1];
        }
        const javaFile = path_1.join(this.project.outdir, dir, ...javaPackage, javaClass + '.java');
        // skip if file exists
        if (fs_extra_1.existsSync(javaFile)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = '') => lines.push(' '.repeat(indent * 4) + line);
        const openBlock = (line = '') => { emit(line + ' {'); indent++; };
        const closeBlock = () => { indent--; emit('}'); };
        if (javaPackage.length > 0) {
            emit(`package ${javaPackage.join('.')};`);
            emit();
        }
        emit(`import ${javaTarget.package}.${toJavaFullTypeName(jsiiType)};`);
        emit(`import ${javaTarget.package}.${toJavaFullTypeName(jsiiOptionsType)};`);
        emit();
        openBlock(`public class ${javaClass}`);
        openBlock('public static void main(String[] args)');
        emit(`${jsiiType.name} project = new ${jsiiType.name}(${renderJavaOptions(indent, jsiiOptionsType.name, bootstrap.args)});`);
        emit('project.synth();');
        closeBlock();
        closeBlock();
        fs_extra_1.mkdirpSync(path_1.dirname(javaFile));
        fs_extra_1.writeFileSync(javaFile, lines.join('\n'));
        this.project.logger.info(`Project definition file was created at ${javaFile}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.java.Projenrc", version: "0.17.71" };
function renderJavaOptions(indent, optionsTypeName, initOptions) {
    if (!initOptions || Object.keys(initOptions).length === 0) {
        return ''; // no options
    }
    const lines = [`${optionsTypeName}.builder()`];
    for (const [name, value] of Object.entries(initOptions)) {
        lines.push(`.${toJavaProperty(name)}(${toJavaValue(value)})`);
    }
    lines.push('.build()');
    return lines.join(`\n${' '.repeat((indent + 1) * 4)}`);
}
function toJavaProperty(prop) {
    return prop;
}
function toJavaValue(value) {
    return JSON.stringify(value);
}
function toJavaFullTypeName(jsiiType) {
    return [jsiiType.namespace, jsiiType.name].filter(x => x).join('.');
}
//# sourceMappingURL=data:application/json;base64,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