"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readJsiiManifest = exports.resolveProjectType = exports.discover = void 0;
const path = require("path");
const fs = require("fs-extra");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const decamelize = require('decamelize');
const PROJEN_MODULE_ROOT = path.join(__dirname, '..');
const PROJECT_BASE_FQN = 'projen.Project';
/**
 * Returns a list of project types exported the modules defined in `moduleDirs`.
 * This list will always also include the built-in projen project types.
 * Modules without a .jsii manifest are skipped.
 *
 * @param moduleDirs A list of npm module directories
 */
function discover(...moduleDirs) {
    const jsii = {};
    const discoverJsii = (dir) => {
        const jsiiFile = path.join(dir, '.jsii');
        if (!fs.existsSync(jsiiFile)) {
            return;
        } // no jsii manifest
        const manifest = fs.readJsonSync(jsiiFile);
        for (const [fqn, type] of Object.entries(manifest.types)) {
            jsii[fqn] = type;
        }
    };
    // read all .jsii manifests from all modules (incl. projen itself) and merge
    // them all into a single map of fqn->type.
    for (const dir of [...moduleDirs, PROJEN_MODULE_ROOT]) {
        discoverJsii(dir);
        if (dir.includes('@') && fs.lstatSync(dir).isDirectory()) {
            const childDirs = fs.readdirSync(dir).map(file => path.join(dir, file));
            for (const child of childDirs) {
                discoverJsii(child);
            }
        }
    }
    const result = new Array();
    for (const fqn of Object.keys(jsii)) {
        const p = toProjectType(jsii, fqn);
        if (!p) {
            continue;
        }
        result.push(p);
    }
    return result.sort((r1, r2) => r1.pjid.localeCompare(r2.pjid));
}
exports.discover = discover;
function resolveProjectType(projectFqn) {
    const manifest = readJsiiManifest(projectFqn);
    const jsii = {};
    for (const [fqn, type] of Object.entries(manifest.types)) {
        jsii[fqn] = type;
    }
    return toProjectType(jsii, projectFqn);
}
exports.resolveProjectType = resolveProjectType;
function toProjectType(jsii, fqn) {
    var _a, _b, _c, _d;
    if (!isProjectType(jsii, fqn)) {
        return undefined;
    }
    const typeinfo = jsii[fqn];
    // projen.web.ReactProject -> web.ReactProject
    const typename = fqn.substring(fqn.indexOf('.') + 1);
    const docsurl = `https://github.com/projen/projen/blob/master/API.md#projen-${typename.toLocaleLowerCase()}`;
    let pjid = (_c = (_b = (_a = typeinfo.docs) === null || _a === void 0 ? void 0 : _a.custom) === null || _b === void 0 ? void 0 : _b.pjid) !== null && _c !== void 0 ? _c : decamelize(typename).replace(/_project$/, '');
    return {
        moduleName: typeinfo.assembly,
        typename,
        pjid,
        fqn,
        options: discoverOptions(jsii, fqn).sort((o1, o2) => o1.name.localeCompare(o2.name)),
        docs: (_d = typeinfo.docs) === null || _d === void 0 ? void 0 : _d.summary,
        docsurl,
    };
}
function readJsiiManifest(jsiiFqn) {
    let [moduleName] = jsiiFqn.split('.');
    if (moduleName === 'projen') {
        moduleName = PROJEN_MODULE_ROOT;
    }
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`);
    return fs.readJsonSync(jsiiManifestFile);
}
exports.readJsiiManifest = readJsiiManifest;
function discoverOptions(jsii, fqn) {
    var _a, _b, _c, _d;
    const options = {};
    const params = (_c = (_b = (_a = jsii[fqn]) === null || _a === void 0 ? void 0 : _a.initializer) === null || _b === void 0 ? void 0 : _b.parameters) !== null && _c !== void 0 ? _c : [];
    const optionsParam = params[0];
    const optionsTypeFqn = (_d = optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.type) === null || _d === void 0 ? void 0 : _d.fqn;
    if (params.length > 1 || (params.length === 1 && (optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.name) !== 'options')) {
        throw new Error(`constructor for project ${fqn} must have a single "options" argument of a struct type. got ${JSON.stringify(params)}`);
    }
    addOptions(optionsTypeFqn);
    const opts = Object.values(options);
    return opts.sort((a, b) => a.switch.localeCompare(b.switch));
    function addOptions(ofqn, basePath = [], optional = false) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        if (!ofqn) {
            return;
        }
        const struct = jsii[ofqn];
        if (!struct) {
            throw new Error(`unable to find options type ${ofqn} for project ${fqn}`);
        }
        for (const prop of (_a = struct.properties) !== null && _a !== void 0 ? _a : []) {
            const propPath = [...basePath, prop.name];
            // protect against double-booking
            if (prop.name in options) {
                throw new Error(`duplicate option "${prop.name}" in ${fqn} (already declared in ${options[prop.name].parent})`);
            }
            let typeName;
            if ((_b = prop.type) === null || _b === void 0 ? void 0 : _b.primitive) {
                typeName = (_c = prop.type) === null || _c === void 0 ? void 0 : _c.primitive; // e.g. 'string', 'boolean', 'number'
            }
            else if ((_d = prop.type) === null || _d === void 0 ? void 0 : _d.fqn) {
                typeName = (_e = prop.type) === null || _e === void 0 ? void 0 : _e.fqn.split('.').pop(); // projen.NodeProjectOptions -> NodeProjectOptions
            }
            else { // any other types such as collection types
                typeName = 'unknown';
            }
            const isOptional = optional || prop.optional;
            let defaultValue = (_f = prop.docs) === null || _f === void 0 ? void 0 : _f.default;
            if (defaultValue === 'undefined') {
                defaultValue = undefined;
            }
            // if this is a mandatory option and we have a default value, it has to be JSON-parsable to the correct type
            if (!isOptional && defaultValue) {
                if (!((_g = prop.type) === null || _g === void 0 ? void 0 : _g.primitive)) {
                    throw new Error(`required option "${prop.name}" with a @default must use primitive types (string, number or boolean). type found is: ${typeName}`);
                }
                checkDefaultIsParsable(prop.name, defaultValue, (_h = prop.type) === null || _h === void 0 ? void 0 : _h.primitive);
            }
            options[prop.name] = filterUndefined({
                path: propPath,
                parent: struct.name,
                name: prop.name,
                docs: prop.docs.summary,
                type: typeName,
                switch: propPath.map(p => decamelize(p).replace(/_/g, '-')).join('-'),
                default: defaultValue,
                optional: isOptional,
                deprecated: prop.docs.stability === 'deprecated' ? true : undefined,
            });
        }
        for (const ifc of (_j = struct.interfaces) !== null && _j !== void 0 ? _j : []) {
            addOptions(ifc);
        }
    }
}
function filterUndefined(obj) {
    const ret = {};
    for (const [k, v] of Object.entries(obj)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
function isProjectType(jsii, fqn) {
    var _a;
    const type = jsii[fqn];
    if (type.kind !== 'class') {
        return false;
    }
    if (type.abstract) {
        return false;
    }
    if ((_a = type.docs) === null || _a === void 0 ? void 0 : _a.deprecated) {
        return false;
    }
    let curr = type;
    while (true) {
        if (curr.fqn === PROJECT_BASE_FQN) {
            return true;
        }
        if (!curr.base) {
            return false;
        }
        curr = jsii[curr.base];
        if (!curr) {
            return false;
        }
    }
}
function checkDefaultIsParsable(prop, value, type) {
    // macros are pass-through
    if (value.startsWith('$')) {
        return;
    }
    try {
        const parsed = JSON.parse(value);
        if (typeof (parsed) !== type) {
            throw new Error(`cannot parse @default value for mandatory option ${prop} as a ${type}: ${parsed}`);
        }
    }
    catch (e) {
        throw new Error(`unable to JSON.parse() value "${value}" specified as @default for mandatory option "${prop}": ${e.message}`);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW52ZW50b3J5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL2ludmVudG9yeS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFDN0IsK0JBQStCO0FBRS9CLGlFQUFpRTtBQUNqRSxNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUM7QUFDekMsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLENBQUMsQ0FBQztBQUN0RCxNQUFNLGdCQUFnQixHQUFHLGdCQUFnQixDQUFDO0FBK0QxQzs7Ozs7O0dBTUc7QUFDSCxTQUFnQixRQUFRLENBQUMsR0FBRyxVQUFvQjtJQUM5QyxNQUFNLElBQUksR0FBYyxFQUFFLENBQUM7SUFFM0IsTUFBTSxZQUFZLEdBQUcsQ0FBQyxHQUFXLEVBQUUsRUFBRTtRQUNuQyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN6QyxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUFFLE9BQU87U0FBRSxDQUFDLG1CQUFtQjtRQUU3RCxNQUFNLFFBQVEsR0FBRyxFQUFFLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzNDLEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxJQUFJLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxLQUFrQixDQUFDLEVBQUU7WUFDckUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLElBQUksQ0FBQztTQUNsQjtJQUNILENBQUMsQ0FBQztJQUVGLDRFQUE0RTtJQUM1RSwyQ0FBMkM7SUFDM0MsS0FBSyxNQUFNLEdBQUcsSUFBSSxDQUFDLEdBQUcsVUFBVSxFQUFFLGtCQUFrQixDQUFDLEVBQUU7UUFDckQsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRWxCLElBQUksR0FBRyxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFO1lBQ3hELE1BQU0sU0FBUyxHQUFHLEVBQUUsQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztZQUN4RSxLQUFLLE1BQU0sS0FBSyxJQUFJLFNBQVMsRUFBRTtnQkFDN0IsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDO2FBQ3JCO1NBQ0Y7S0FDRjtJQUVELE1BQU0sTUFBTSxHQUFHLElBQUksS0FBSyxFQUFlLENBQUM7SUFFeEMsS0FBSyxNQUFNLEdBQUcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFO1FBQ25DLE1BQU0sQ0FBQyxHQUFHLGFBQWEsQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDbkMsSUFBSSxDQUFDLENBQUMsRUFBRTtZQUNOLFNBQVM7U0FDVjtRQUVELE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7S0FDaEI7SUFFRCxPQUFPLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztBQUNqRSxDQUFDO0FBdENELDRCQXNDQztBQUVELFNBQWdCLGtCQUFrQixDQUFDLFVBQWtCO0lBQ25ELE1BQU0sUUFBUSxHQUFHLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQzlDLE1BQU0sSUFBSSxHQUFjLEVBQUUsQ0FBQztJQUMzQixLQUFLLE1BQU0sQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsS0FBa0IsQ0FBQyxFQUFFO1FBQ3JFLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxJQUFJLENBQUM7S0FDbEI7SUFFRCxPQUFPLGFBQWEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxDQUFDLENBQUM7QUFDekMsQ0FBQztBQVJELGdEQVFDO0FBRUQsU0FBUyxhQUFhLENBQUMsSUFBZSxFQUFFLEdBQVc7O0lBQ2pELElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxFQUFFO1FBQzdCLE9BQU8sU0FBUyxDQUFDO0tBQ2xCO0lBRUQsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBRTNCLDhDQUE4QztJQUM5QyxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7SUFFckQsTUFBTSxPQUFPLEdBQUcsOERBQThELFFBQVEsQ0FBQyxpQkFBaUIsRUFBRSxFQUFFLENBQUM7SUFDN0csSUFBSSxJQUFJLHFCQUFHLFFBQVEsQ0FBQyxJQUFJLDBDQUFFLE1BQU0sMENBQUUsSUFBSSxtQ0FBSSxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUN4RixPQUFPO1FBQ0wsVUFBVSxFQUFFLFFBQVEsQ0FBQyxRQUFRO1FBQzdCLFFBQVE7UUFDUixJQUFJO1FBQ0osR0FBRztRQUNILE9BQU8sRUFBRSxlQUFlLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNwRixJQUFJLFFBQUUsUUFBUSxDQUFDLElBQUksMENBQUUsT0FBTztRQUM1QixPQUFPO0tBQ08sQ0FBQztBQUNuQixDQUFDO0FBRUQsU0FBZ0IsZ0JBQWdCLENBQUMsT0FBZTtJQUM5QyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsT0FBTyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QyxJQUFJLFVBQVUsS0FBSyxRQUFRLEVBQUU7UUFDM0IsVUFBVSxHQUFHLGtCQUFrQixDQUFDO0tBQ2pDO0lBRUQsTUFBTSxnQkFBZ0IsR0FBRyxPQUFPLENBQUMsT0FBTyxDQUFDLEdBQUcsVUFBVSxRQUFRLENBQUMsQ0FBQztJQUNoRSxPQUFPLEVBQUUsQ0FBQyxZQUFZLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztBQUMzQyxDQUFDO0FBUkQsNENBUUM7QUFFRCxTQUFTLGVBQWUsQ0FBQyxJQUFlLEVBQUUsR0FBVzs7SUFDbkQsTUFBTSxPQUFPLEdBQXNDLEVBQUUsQ0FBQztJQUN0RCxNQUFNLE1BQU0scUJBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQywwQ0FBRSxXQUFXLDBDQUFFLFVBQVUsbUNBQUksRUFBRSxDQUFDO0lBQ3hELE1BQU0sWUFBWSxHQUFHLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUMvQixNQUFNLGNBQWMsU0FBRyxZQUFZLGFBQVosWUFBWSx1QkFBWixZQUFZLENBQUUsSUFBSSwwQ0FBRSxHQUFHLENBQUM7SUFFL0MsSUFBSSxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLENBQUEsWUFBWSxhQUFaLFlBQVksdUJBQVosWUFBWSxDQUFFLElBQUksTUFBSyxTQUFTLENBQUMsRUFBRTtRQUNsRixNQUFNLElBQUksS0FBSyxDQUFDLDJCQUEyQixHQUFHLGdFQUFnRSxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQztLQUN6STtJQUVELFVBQVUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUUzQixNQUFNLElBQUksR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBRXBDLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO0lBRTdELFNBQVMsVUFBVSxDQUFDLElBQWEsRUFBRSxXQUFxQixFQUFFLEVBQUUsUUFBUSxHQUFHLEtBQUs7O1FBQzFFLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFDVCxPQUFPO1NBQ1I7UUFFRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDMUIsSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNYLE1BQU0sSUFBSSxLQUFLLENBQUMsK0JBQStCLElBQUksZ0JBQWdCLEdBQUcsRUFBRSxDQUFDLENBQUM7U0FDM0U7UUFFRCxLQUFLLE1BQU0sSUFBSSxVQUFJLE1BQU0sQ0FBQyxVQUFVLG1DQUFJLEVBQUUsRUFBRTtZQUMxQyxNQUFNLFFBQVEsR0FBRyxDQUFDLEdBQUcsUUFBUSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUUxQyxpQ0FBaUM7WUFDakMsSUFBSSxJQUFJLENBQUMsSUFBSSxJQUFJLE9BQU8sRUFBRTtnQkFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsSUFBSSxDQUFDLElBQUksUUFBUSxHQUFHLHlCQUF5QixPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7YUFDakg7WUFFRCxJQUFJLFFBQVEsQ0FBQztZQUNiLFVBQUksSUFBSSxDQUFDLElBQUksMENBQUUsU0FBUyxFQUFFO2dCQUN4QixRQUFRLFNBQUcsSUFBSSxDQUFDLElBQUksMENBQUUsU0FBUyxDQUFDLENBQUMscUNBQXFDO2FBQ3ZFO2lCQUFNLFVBQUksSUFBSSxDQUFDLElBQUksMENBQUUsR0FBRyxFQUFFO2dCQUN6QixRQUFRLFNBQUcsSUFBSSxDQUFDLElBQUksMENBQUUsR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxrREFBa0Q7YUFDL0Y7aUJBQU0sRUFBRSwyQ0FBMkM7Z0JBQ2xELFFBQVEsR0FBRyxTQUFTLENBQUM7YUFDdEI7WUFFRCxNQUFNLFVBQVUsR0FBRyxRQUFRLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUM3QyxJQUFJLFlBQVksU0FBRyxJQUFJLENBQUMsSUFBSSwwQ0FBRSxPQUFPLENBQUM7WUFFdEMsSUFBSSxZQUFZLEtBQUssV0FBVyxFQUFFO2dCQUNoQyxZQUFZLEdBQUcsU0FBUyxDQUFDO2FBQzFCO1lBRUQsNEdBQTRHO1lBQzVHLElBQUksQ0FBQyxVQUFVLElBQUksWUFBWSxFQUFFO2dCQUMvQixJQUFJLFFBQUMsSUFBSSxDQUFDLElBQUksMENBQUUsU0FBUyxDQUFBLEVBQUU7b0JBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQUMsb0JBQW9CLElBQUksQ0FBQyxJQUFJLDBGQUEwRixRQUFRLEVBQUUsQ0FBQyxDQUFDO2lCQUNwSjtnQkFFRCxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFlBQVksUUFBRSxJQUFJLENBQUMsSUFBSSwwQ0FBRSxTQUFTLENBQUMsQ0FBQzthQUN2RTtZQUVELE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsZUFBZSxDQUFDO2dCQUNuQyxJQUFJLEVBQUUsUUFBUTtnQkFDZCxNQUFNLEVBQUUsTUFBTSxDQUFDLElBQUk7Z0JBQ25CLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtnQkFDZixJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPO2dCQUN2QixJQUFJLEVBQUUsUUFBUTtnQkFDZCxNQUFNLEVBQUUsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQztnQkFDckUsT0FBTyxFQUFFLFlBQVk7Z0JBQ3JCLFFBQVEsRUFBRSxVQUFVO2dCQUNwQixVQUFVLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEtBQUssWUFBWSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVM7YUFDcEUsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxLQUFLLE1BQU0sR0FBRyxVQUFJLE1BQU0sQ0FBQyxVQUFVLG1DQUFJLEVBQUUsRUFBRTtZQUN6QyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDakI7SUFDSCxDQUFDO0FBQ0gsQ0FBQztBQUVELFNBQVMsZUFBZSxDQUFDLEdBQVE7SUFDL0IsTUFBTSxHQUFHLEdBQVEsRUFBRSxDQUFDO0lBQ3BCLEtBQUssTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQ3hDLElBQUksQ0FBQyxLQUFLLFNBQVMsRUFBRTtZQUNuQixHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ1o7S0FDRjtJQUNELE9BQU8sR0FBRyxDQUFDO0FBQ2IsQ0FBQztBQUVELFNBQVMsYUFBYSxDQUFDLElBQWUsRUFBRSxHQUFXOztJQUNqRCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7SUFFdkIsSUFBSSxJQUFJLENBQUMsSUFBSSxLQUFLLE9BQU8sRUFBRTtRQUN6QixPQUFPLEtBQUssQ0FBQztLQUNkO0lBQ0QsSUFBSSxJQUFJLENBQUMsUUFBUSxFQUFFO1FBQ2pCLE9BQU8sS0FBSyxDQUFDO0tBQ2Q7SUFFRCxVQUFJLElBQUksQ0FBQyxJQUFJLDBDQUFFLFVBQVUsRUFBRTtRQUN6QixPQUFPLEtBQUssQ0FBQztLQUNkO0lBRUQsSUFBSSxJQUFJLEdBQUcsSUFBSSxDQUFDO0lBQ2hCLE9BQU8sSUFBSSxFQUFFO1FBQ1gsSUFBSSxJQUFJLENBQUMsR0FBRyxLQUFLLGdCQUFnQixFQUFFO1lBQ2pDLE9BQU8sSUFBSSxDQUFDO1NBQ2I7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRTtZQUNkLE9BQU8sS0FBSyxDQUFDO1NBQ2Q7UUFFRCxJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2QixJQUFJLENBQUMsSUFBSSxFQUFFO1lBQ1QsT0FBTyxLQUFLLENBQUM7U0FDZDtLQUNGO0FBQ0gsQ0FBQztBQUVELFNBQVMsc0JBQXNCLENBQUMsSUFBWSxFQUFFLEtBQWEsRUFBRSxJQUFZO0lBQ3ZFLDBCQUEwQjtJQUMxQixJQUFJLEtBQUssQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLEVBQUU7UUFDekIsT0FBTztLQUNSO0lBQ0QsSUFBSTtRQUNGLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDakMsSUFBSSxPQUFNLENBQUMsTUFBTSxDQUFDLEtBQUssSUFBSSxFQUFFO1lBQzNCLE1BQU0sSUFBSSxLQUFLLENBQUMsb0RBQW9ELElBQUksU0FBUyxJQUFJLEtBQUssTUFBTSxFQUFFLENBQUMsQ0FBQztTQUNyRztLQUVGO0lBQUMsT0FBTyxDQUFDLEVBQUU7UUFDVixNQUFNLElBQUksS0FBSyxDQUFDLGlDQUFpQyxLQUFLLGlEQUFpRCxJQUFJLE1BQU0sQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7S0FDL0g7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzLWV4dHJhJztcblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIEB0eXBlc2NyaXB0LWVzbGludC9uby1yZXF1aXJlLWltcG9ydHNcbmNvbnN0IGRlY2FtZWxpemUgPSByZXF1aXJlKCdkZWNhbWVsaXplJyk7XG5jb25zdCBQUk9KRU5fTU9EVUxFX1JPT1QgPSBwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4nKTtcbmNvbnN0IFBST0pFQ1RfQkFTRV9GUU4gPSAncHJvamVuLlByb2plY3QnO1xuXG50eXBlIEpzaWlUeXBlcyA9IHsgW25hbWU6IHN0cmluZ106IEpzaWlUeXBlIH07XG5cbmV4cG9ydCBpbnRlcmZhY2UgUHJvamVjdE9wdGlvbiB7XG4gIHBhdGg6IHN0cmluZ1tdO1xuICBuYW1lOiBzdHJpbmc7XG4gIHN3aXRjaDogc3RyaW5nO1xuICB0eXBlOiBzdHJpbmc7XG4gIHBhcmVudDogc3RyaW5nO1xuICBkb2NzPzogc3RyaW5nO1xuICBkZWZhdWx0Pzogc3RyaW5nO1xuICBvcHRpb25hbD86IGJvb2xlYW47XG4gIGRlcHJlY2F0ZWQ/OiBib29sZWFuO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFByb2plY3RUeXBlIHtcbiAgbW9kdWxlTmFtZTogc3RyaW5nO1xuICBwamlkOiBzdHJpbmc7XG4gIGZxbjogc3RyaW5nO1xuICB0eXBlbmFtZTogc3RyaW5nO1xuICBvcHRpb25zOiBQcm9qZWN0T3B0aW9uW107XG4gIGRvY3M/OiBzdHJpbmc7XG4gIGRvY3N1cmw6IHN0cmluZztcbn1cblxuaW50ZXJmYWNlIEpzaWlUeXBlIHtcbiAgbmFtZTogc3RyaW5nO1xuICBhc3NlbWJseTogc3RyaW5nO1xuICBraW5kOiBzdHJpbmc7XG4gIGFic3RyYWN0PzogYm9vbGVhbjtcbiAgYmFzZT86IHN0cmluZztcbiAgZnFuOiBzdHJpbmc7XG4gIGludGVyZmFjZXM/OiBzdHJpbmdbXTtcbiAgaW5pdGlhbGl6ZXI/OiB7XG4gICAgcGFyYW1ldGVycz86IEFycmF5PHtcbiAgICAgIG5hbWU6IHN0cmluZztcbiAgICAgIHR5cGU/OiB7IGZxbj86IHN0cmluZyB9O1xuICAgIH0+O1xuICB9O1xuICBwcm9wZXJ0aWVzPzogQXJyYXk8e1xuICAgIG5hbWU6IHN0cmluZztcbiAgICBkb2NzOiB7XG4gICAgICBzdW1tYXJ5Pzogc3RyaW5nO1xuICAgICAgZGVmYXVsdD86IHN0cmluZztcbiAgICAgIGRlcHJlY2F0ZWQ/OiBzdHJpbmc7XG4gICAgICBzdGFiaWxpdHk/OiBzdHJpbmc7XG4gICAgfTtcbiAgICBvcHRpb25hbD86IGJvb2xlYW47XG4gICAgdHlwZT86IHtcbiAgICAgIHByaW1pdGl2ZT86IHN0cmluZztcbiAgICAgIGZxbj86IHN0cmluZztcbiAgICB9O1xuICB9PjtcbiAgZG9jcz86IHtcbiAgICBzdW1tYXJ5Pzogc3RyaW5nO1xuICAgIGRlcHJlY2F0ZWQ/OiBzdHJpbmc7XG4gICAgY3VzdG9tPzoge1xuICAgICAgcGppZD86IHN0cmluZztcbiAgICB9O1xuICB9O1xufVxuXG4vKipcbiAqIFJldHVybnMgYSBsaXN0IG9mIHByb2plY3QgdHlwZXMgZXhwb3J0ZWQgdGhlIG1vZHVsZXMgZGVmaW5lZCBpbiBgbW9kdWxlRGlyc2AuXG4gKiBUaGlzIGxpc3Qgd2lsbCBhbHdheXMgYWxzbyBpbmNsdWRlIHRoZSBidWlsdC1pbiBwcm9qZW4gcHJvamVjdCB0eXBlcy5cbiAqIE1vZHVsZXMgd2l0aG91dCBhIC5qc2lpIG1hbmlmZXN0IGFyZSBza2lwcGVkLlxuICpcbiAqIEBwYXJhbSBtb2R1bGVEaXJzIEEgbGlzdCBvZiBucG0gbW9kdWxlIGRpcmVjdG9yaWVzXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBkaXNjb3ZlciguLi5tb2R1bGVEaXJzOiBzdHJpbmdbXSkge1xuICBjb25zdCBqc2lpOiBKc2lpVHlwZXMgPSB7fTtcblxuICBjb25zdCBkaXNjb3ZlckpzaWkgPSAoZGlyOiBzdHJpbmcpID0+IHtcbiAgICBjb25zdCBqc2lpRmlsZSA9IHBhdGguam9pbihkaXIsICcuanNpaScpO1xuICAgIGlmICghZnMuZXhpc3RzU3luYyhqc2lpRmlsZSkpIHsgcmV0dXJuOyB9IC8vIG5vIGpzaWkgbWFuaWZlc3RcblxuICAgIGNvbnN0IG1hbmlmZXN0ID0gZnMucmVhZEpzb25TeW5jKGpzaWlGaWxlKTtcbiAgICBmb3IgKGNvbnN0IFtmcW4sIHR5cGVdIG9mIE9iamVjdC5lbnRyaWVzKG1hbmlmZXN0LnR5cGVzIGFzIEpzaWlUeXBlcykpIHtcbiAgICAgIGpzaWlbZnFuXSA9IHR5cGU7XG4gICAgfVxuICB9O1xuXG4gIC8vIHJlYWQgYWxsIC5qc2lpIG1hbmlmZXN0cyBmcm9tIGFsbCBtb2R1bGVzIChpbmNsLiBwcm9qZW4gaXRzZWxmKSBhbmQgbWVyZ2VcbiAgLy8gdGhlbSBhbGwgaW50byBhIHNpbmdsZSBtYXAgb2YgZnFuLT50eXBlLlxuICBmb3IgKGNvbnN0IGRpciBvZiBbLi4ubW9kdWxlRGlycywgUFJPSkVOX01PRFVMRV9ST09UXSkge1xuICAgIGRpc2NvdmVySnNpaShkaXIpO1xuXG4gICAgaWYgKGRpci5pbmNsdWRlcygnQCcpICYmIGZzLmxzdGF0U3luYyhkaXIpLmlzRGlyZWN0b3J5KCkpIHtcbiAgICAgIGNvbnN0IGNoaWxkRGlycyA9IGZzLnJlYWRkaXJTeW5jKGRpcikubWFwKGZpbGUgPT4gcGF0aC5qb2luKGRpciwgZmlsZSkpO1xuICAgICAgZm9yIChjb25zdCBjaGlsZCBvZiBjaGlsZERpcnMpIHtcbiAgICAgICAgZGlzY292ZXJKc2lpKGNoaWxkKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBjb25zdCByZXN1bHQgPSBuZXcgQXJyYXk8UHJvamVjdFR5cGU+KCk7XG5cbiAgZm9yIChjb25zdCBmcW4gb2YgT2JqZWN0LmtleXMoanNpaSkpIHtcbiAgICBjb25zdCBwID0gdG9Qcm9qZWN0VHlwZShqc2lpLCBmcW4pO1xuICAgIGlmICghcCkge1xuICAgICAgY29udGludWU7XG4gICAgfVxuXG4gICAgcmVzdWx0LnB1c2gocCk7XG4gIH1cblxuICByZXR1cm4gcmVzdWx0LnNvcnQoKHIxLCByMikgPT4gcjEucGppZC5sb2NhbGVDb21wYXJlKHIyLnBqaWQpKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHJlc29sdmVQcm9qZWN0VHlwZShwcm9qZWN0RnFuOiBzdHJpbmcpIHtcbiAgY29uc3QgbWFuaWZlc3QgPSByZWFkSnNpaU1hbmlmZXN0KHByb2plY3RGcW4pO1xuICBjb25zdCBqc2lpOiBKc2lpVHlwZXMgPSB7fTtcbiAgZm9yIChjb25zdCBbZnFuLCB0eXBlXSBvZiBPYmplY3QuZW50cmllcyhtYW5pZmVzdC50eXBlcyBhcyBKc2lpVHlwZXMpKSB7XG4gICAganNpaVtmcW5dID0gdHlwZTtcbiAgfVxuXG4gIHJldHVybiB0b1Byb2plY3RUeXBlKGpzaWksIHByb2plY3RGcW4pO1xufVxuXG5mdW5jdGlvbiB0b1Byb2plY3RUeXBlKGpzaWk6IEpzaWlUeXBlcywgZnFuOiBzdHJpbmcpIHtcbiAgaWYgKCFpc1Byb2plY3RUeXBlKGpzaWksIGZxbikpIHtcbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG5cbiAgY29uc3QgdHlwZWluZm8gPSBqc2lpW2Zxbl07XG5cbiAgLy8gcHJvamVuLndlYi5SZWFjdFByb2plY3QgLT4gd2ViLlJlYWN0UHJvamVjdFxuICBjb25zdCB0eXBlbmFtZSA9IGZxbi5zdWJzdHJpbmcoZnFuLmluZGV4T2YoJy4nKSArIDEpO1xuXG4gIGNvbnN0IGRvY3N1cmwgPSBgaHR0cHM6Ly9naXRodWIuY29tL3Byb2plbi9wcm9qZW4vYmxvYi9tYXN0ZXIvQVBJLm1kI3Byb2plbi0ke3R5cGVuYW1lLnRvTG9jYWxlTG93ZXJDYXNlKCl9YDtcbiAgbGV0IHBqaWQgPSB0eXBlaW5mby5kb2NzPy5jdXN0b20/LnBqaWQgPz8gZGVjYW1lbGl6ZSh0eXBlbmFtZSkucmVwbGFjZSgvX3Byb2plY3QkLywgJycpO1xuICByZXR1cm4ge1xuICAgIG1vZHVsZU5hbWU6IHR5cGVpbmZvLmFzc2VtYmx5LFxuICAgIHR5cGVuYW1lLFxuICAgIHBqaWQsXG4gICAgZnFuLFxuICAgIG9wdGlvbnM6IGRpc2NvdmVyT3B0aW9ucyhqc2lpLCBmcW4pLnNvcnQoKG8xLCBvMikgPT4gbzEubmFtZS5sb2NhbGVDb21wYXJlKG8yLm5hbWUpKSxcbiAgICBkb2NzOiB0eXBlaW5mby5kb2NzPy5zdW1tYXJ5LFxuICAgIGRvY3N1cmwsXG4gIH0gYXMgUHJvamVjdFR5cGU7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiByZWFkSnNpaU1hbmlmZXN0KGpzaWlGcW46IHN0cmluZyk6IGFueSB7XG4gIGxldCBbbW9kdWxlTmFtZV0gPSBqc2lpRnFuLnNwbGl0KCcuJyk7XG4gIGlmIChtb2R1bGVOYW1lID09PSAncHJvamVuJykge1xuICAgIG1vZHVsZU5hbWUgPSBQUk9KRU5fTU9EVUxFX1JPT1Q7XG4gIH1cblxuICBjb25zdCBqc2lpTWFuaWZlc3RGaWxlID0gcmVxdWlyZS5yZXNvbHZlKGAke21vZHVsZU5hbWV9Ly5qc2lpYCk7XG4gIHJldHVybiBmcy5yZWFkSnNvblN5bmMoanNpaU1hbmlmZXN0RmlsZSk7XG59XG5cbmZ1bmN0aW9uIGRpc2NvdmVyT3B0aW9ucyhqc2lpOiBKc2lpVHlwZXMsIGZxbjogc3RyaW5nKTogUHJvamVjdE9wdGlvbltdIHtcbiAgY29uc3Qgb3B0aW9uczogeyBbbmFtZTogc3RyaW5nXTogUHJvamVjdE9wdGlvbiB9ID0ge307XG4gIGNvbnN0IHBhcmFtcyA9IGpzaWlbZnFuXT8uaW5pdGlhbGl6ZXI/LnBhcmFtZXRlcnMgPz8gW107XG4gIGNvbnN0IG9wdGlvbnNQYXJhbSA9IHBhcmFtc1swXTtcbiAgY29uc3Qgb3B0aW9uc1R5cGVGcW4gPSBvcHRpb25zUGFyYW0/LnR5cGU/LmZxbjtcblxuICBpZiAocGFyYW1zLmxlbmd0aCA+IDEgfHwgKHBhcmFtcy5sZW5ndGggPT09IDEgJiYgb3B0aW9uc1BhcmFtPy5uYW1lICE9PSAnb3B0aW9ucycpKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBjb25zdHJ1Y3RvciBmb3IgcHJvamVjdCAke2Zxbn0gbXVzdCBoYXZlIGEgc2luZ2xlIFwib3B0aW9uc1wiIGFyZ3VtZW50IG9mIGEgc3RydWN0IHR5cGUuIGdvdCAke0pTT04uc3RyaW5naWZ5KHBhcmFtcyl9YCk7XG4gIH1cblxuICBhZGRPcHRpb25zKG9wdGlvbnNUeXBlRnFuKTtcblxuICBjb25zdCBvcHRzID0gT2JqZWN0LnZhbHVlcyhvcHRpb25zKTtcblxuICByZXR1cm4gb3B0cy5zb3J0KChhLCBiKSA9PiBhLnN3aXRjaC5sb2NhbGVDb21wYXJlKGIuc3dpdGNoKSk7XG5cbiAgZnVuY3Rpb24gYWRkT3B0aW9ucyhvZnFuPzogc3RyaW5nLCBiYXNlUGF0aDogc3RyaW5nW10gPSBbXSwgb3B0aW9uYWwgPSBmYWxzZSkge1xuICAgIGlmICghb2Zxbikge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNvbnN0IHN0cnVjdCA9IGpzaWlbb2Zxbl07XG4gICAgaWYgKCFzdHJ1Y3QpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgdW5hYmxlIHRvIGZpbmQgb3B0aW9ucyB0eXBlICR7b2Zxbn0gZm9yIHByb2plY3QgJHtmcW59YCk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBwcm9wIG9mIHN0cnVjdC5wcm9wZXJ0aWVzID8/IFtdKSB7XG4gICAgICBjb25zdCBwcm9wUGF0aCA9IFsuLi5iYXNlUGF0aCwgcHJvcC5uYW1lXTtcblxuICAgICAgLy8gcHJvdGVjdCBhZ2FpbnN0IGRvdWJsZS1ib29raW5nXG4gICAgICBpZiAocHJvcC5uYW1lIGluIG9wdGlvbnMpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBkdXBsaWNhdGUgb3B0aW9uIFwiJHtwcm9wLm5hbWV9XCIgaW4gJHtmcW59IChhbHJlYWR5IGRlY2xhcmVkIGluICR7b3B0aW9uc1twcm9wLm5hbWVdLnBhcmVudH0pYCk7XG4gICAgICB9XG5cbiAgICAgIGxldCB0eXBlTmFtZTtcbiAgICAgIGlmIChwcm9wLnR5cGU/LnByaW1pdGl2ZSkge1xuICAgICAgICB0eXBlTmFtZSA9IHByb3AudHlwZT8ucHJpbWl0aXZlOyAvLyBlLmcuICdzdHJpbmcnLCAnYm9vbGVhbicsICdudW1iZXInXG4gICAgICB9IGVsc2UgaWYgKHByb3AudHlwZT8uZnFuKSB7XG4gICAgICAgIHR5cGVOYW1lID0gcHJvcC50eXBlPy5mcW4uc3BsaXQoJy4nKS5wb3AoKTsgLy8gcHJvamVuLk5vZGVQcm9qZWN0T3B0aW9ucyAtPiBOb2RlUHJvamVjdE9wdGlvbnNcbiAgICAgIH0gZWxzZSB7IC8vIGFueSBvdGhlciB0eXBlcyBzdWNoIGFzIGNvbGxlY3Rpb24gdHlwZXNcbiAgICAgICAgdHlwZU5hbWUgPSAndW5rbm93bic7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IGlzT3B0aW9uYWwgPSBvcHRpb25hbCB8fCBwcm9wLm9wdGlvbmFsO1xuICAgICAgbGV0IGRlZmF1bHRWYWx1ZSA9IHByb3AuZG9jcz8uZGVmYXVsdDtcblxuICAgICAgaWYgKGRlZmF1bHRWYWx1ZSA9PT0gJ3VuZGVmaW5lZCcpIHtcbiAgICAgICAgZGVmYXVsdFZhbHVlID0gdW5kZWZpbmVkO1xuICAgICAgfVxuXG4gICAgICAvLyBpZiB0aGlzIGlzIGEgbWFuZGF0b3J5IG9wdGlvbiBhbmQgd2UgaGF2ZSBhIGRlZmF1bHQgdmFsdWUsIGl0IGhhcyB0byBiZSBKU09OLXBhcnNhYmxlIHRvIHRoZSBjb3JyZWN0IHR5cGVcbiAgICAgIGlmICghaXNPcHRpb25hbCAmJiBkZWZhdWx0VmFsdWUpIHtcbiAgICAgICAgaWYgKCFwcm9wLnR5cGU/LnByaW1pdGl2ZSkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgcmVxdWlyZWQgb3B0aW9uIFwiJHtwcm9wLm5hbWV9XCIgd2l0aCBhIEBkZWZhdWx0IG11c3QgdXNlIHByaW1pdGl2ZSB0eXBlcyAoc3RyaW5nLCBudW1iZXIgb3IgYm9vbGVhbikuIHR5cGUgZm91bmQgaXM6ICR7dHlwZU5hbWV9YCk7XG4gICAgICAgIH1cblxuICAgICAgICBjaGVja0RlZmF1bHRJc1BhcnNhYmxlKHByb3AubmFtZSwgZGVmYXVsdFZhbHVlLCBwcm9wLnR5cGU/LnByaW1pdGl2ZSk7XG4gICAgICB9XG5cbiAgICAgIG9wdGlvbnNbcHJvcC5uYW1lXSA9IGZpbHRlclVuZGVmaW5lZCh7XG4gICAgICAgIHBhdGg6IHByb3BQYXRoLFxuICAgICAgICBwYXJlbnQ6IHN0cnVjdC5uYW1lLFxuICAgICAgICBuYW1lOiBwcm9wLm5hbWUsXG4gICAgICAgIGRvY3M6IHByb3AuZG9jcy5zdW1tYXJ5LFxuICAgICAgICB0eXBlOiB0eXBlTmFtZSxcbiAgICAgICAgc3dpdGNoOiBwcm9wUGF0aC5tYXAocCA9PiBkZWNhbWVsaXplKHApLnJlcGxhY2UoL18vZywgJy0nKSkuam9pbignLScpLFxuICAgICAgICBkZWZhdWx0OiBkZWZhdWx0VmFsdWUsXG4gICAgICAgIG9wdGlvbmFsOiBpc09wdGlvbmFsLFxuICAgICAgICBkZXByZWNhdGVkOiBwcm9wLmRvY3Muc3RhYmlsaXR5ID09PSAnZGVwcmVjYXRlZCcgPyB0cnVlIDogdW5kZWZpbmVkLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBpZmMgb2Ygc3RydWN0LmludGVyZmFjZXMgPz8gW10pIHtcbiAgICAgIGFkZE9wdGlvbnMoaWZjKTtcbiAgICB9XG4gIH1cbn1cblxuZnVuY3Rpb24gZmlsdGVyVW5kZWZpbmVkKG9iajogYW55KSB7XG4gIGNvbnN0IHJldDogYW55ID0ge307XG4gIGZvciAoY29uc3QgW2ssIHZdIG9mIE9iamVjdC5lbnRyaWVzKG9iaikpIHtcbiAgICBpZiAodiAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXRba10gPSB2O1xuICAgIH1cbiAgfVxuICByZXR1cm4gcmV0O1xufVxuXG5mdW5jdGlvbiBpc1Byb2plY3RUeXBlKGpzaWk6IEpzaWlUeXBlcywgZnFuOiBzdHJpbmcpIHtcbiAgY29uc3QgdHlwZSA9IGpzaWlbZnFuXTtcblxuICBpZiAodHlwZS5raW5kICE9PSAnY2xhc3MnKSB7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG4gIGlmICh0eXBlLmFic3RyYWN0KSB7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG5cbiAgaWYgKHR5cGUuZG9jcz8uZGVwcmVjYXRlZCkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIGxldCBjdXJyID0gdHlwZTtcbiAgd2hpbGUgKHRydWUpIHtcbiAgICBpZiAoY3Vyci5mcW4gPT09IFBST0pFQ1RfQkFTRV9GUU4pIHtcbiAgICAgIHJldHVybiB0cnVlO1xuICAgIH1cblxuICAgIGlmICghY3Vyci5iYXNlKSB7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuXG4gICAgY3VyciA9IGpzaWlbY3Vyci5iYXNlXTtcbiAgICBpZiAoIWN1cnIpIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG4gIH1cbn1cblxuZnVuY3Rpb24gY2hlY2tEZWZhdWx0SXNQYXJzYWJsZShwcm9wOiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcsIHR5cGU6IHN0cmluZykge1xuICAvLyBtYWNyb3MgYXJlIHBhc3MtdGhyb3VnaFxuICBpZiAodmFsdWUuc3RhcnRzV2l0aCgnJCcpKSB7XG4gICAgcmV0dXJuO1xuICB9XG4gIHRyeSB7XG4gICAgY29uc3QgcGFyc2VkID0gSlNPTi5wYXJzZSh2YWx1ZSk7XG4gICAgaWYgKHR5cGVvZihwYXJzZWQpICE9PSB0eXBlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYGNhbm5vdCBwYXJzZSBAZGVmYXVsdCB2YWx1ZSBmb3IgbWFuZGF0b3J5IG9wdGlvbiAke3Byb3B9IGFzIGEgJHt0eXBlfTogJHtwYXJzZWR9YCk7XG4gICAgfVxuXG4gIH0gY2F0Y2ggKGUpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYHVuYWJsZSB0byBKU09OLnBhcnNlKCkgdmFsdWUgXCIke3ZhbHVlfVwiIHNwZWNpZmllZCBhcyBAZGVmYXVsdCBmb3IgbWFuZGF0b3J5IG9wdGlvbiBcIiR7cHJvcH1cIjogJHtlLm1lc3NhZ2V9YCk7XG4gIH1cbn0iXX0=