"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const os = require("os");
const path = require("path");
const vm = require("vm");
const fs = require("fs-extra");
const inquirer = require("inquirer");
const yargs = require("yargs");
const inventory = require("../../inventory");
const render_options_1 = require("../../javascript/render-options");
const logging = require("../../logging");
const util_1 = require("../../util");
const macros_1 = require("../macros");
class Command {
    constructor() {
        this.command = 'new [PROJECT-TYPE] [OPTIONS]';
        this.describe = 'Creates a new projen project';
    }
    builder(args) {
        var _a;
        args.positional('PROJECT-TYPE', { describe: 'optional only when --from is used and there is a single project type in the external module', type: 'string' });
        args.option('synth', { type: 'boolean', default: true, desc: 'Synthesize after creating .projenrc.js' });
        args.option('comments', { type: 'boolean', default: true, desc: 'Include commented out options in .projenrc.js (use --no-comments to disable)' });
        args.option('from', { type: 'string', alias: 'f', desc: 'External jsii npm module to create project from. Supports any package spec supported by yarn (such as "my-pack@^2.0")' });
        args.example('projen new awscdk-app-ts', 'Creates a new project of built-in type "awscdk-app-ts"');
        args.example('projen new --from projen-vue@^2', 'Creates a new project from an external module "projen-vue" with the specified version');
        for (const type of inventory.discover()) {
            args.command(type.pjid, (_a = type.docs) !== null && _a !== void 0 ? _a : '', {
                builder: cargs => {
                    var _a, _b, _c;
                    cargs.showHelpOnFail(true);
                    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
                        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean') {
                            continue; // we don't support non-primitive fields as command line options
                        }
                        let desc = [(_c = (_b = option.docs) === null || _b === void 0 ? void 0 : _b.replace(/\ *\.$/, '')) !== null && _c !== void 0 ? _c : ''];
                        const required = !option.optional;
                        let defaultValue;
                        if (option.default && option.default !== 'undefined') {
                            if (!required) {
                                // if the field is not required, just describe the default but don't actually assign a value
                                desc.push(`[default: ${option.default.replace(/^\ *-/, '').replace(/\.$/, '').trim()}]`);
                            }
                            else {
                                // if the field is required and we have a @default, then assign
                                // the value here so it appears in `--help`
                                defaultValue = renderDefault(option.default);
                            }
                        }
                        cargs.option(option.switch, {
                            group: required ? 'Required:' : 'Optional:',
                            type: option.type,
                            description: desc.join(' '),
                            default: defaultValue,
                            required,
                        });
                    }
                    return cargs;
                },
                handler: argv => newProject(process.cwd(), type, argv),
            });
        }
        return args;
    }
    async handler(args) {
        // handle --from which means we want to first install a jsii module and then
        // create a project defined within this module.
        if (args.from) {
            return newProjectFromModule(process.cwd(), args.from, args);
        }
        // project type is defined but was not matched by yargs, so print the list of supported types
        if (args.projectType) {
            console.log(`Invalid project type ${args.projectType}. Supported types:`);
            for (const pjid of inventory.discover().map(x => x.pjid)) {
                console.log(`  ${pjid}`);
            }
            return;
        }
        // Handles the use case that nothing was specified since PROJECT-TYPE is now an optional positional parameter
        yargs.showHelp();
    }
}
/**
 * Creates a new project with defaults.
 *
 * This function creates the project type in-process (with in VM) and calls
 * `.synth()` on it (if `options.synth` is not `false`).
 *
 * At the moment, it also generates a `.projenrc.js` file with the same code
 * that was just executed. In the future, this will also be done by the project
 * type, so we can easily support multiple languages of projenrc.
 */
function createProject(opts) {
    // Default project resolution location
    let mod = '../../index';
    // External projects need to load the module from the modules directory
    if (opts.type.moduleName !== 'projen') {
        try {
            mod = path.dirname(require.resolve(path.join(opts.type.moduleName, 'package.json'), { paths: [process.cwd()] }));
        }
        catch (err) {
            throw new Error(`External project module '${opts.type.moduleName}' could not be resolved.`);
        }
    }
    // pass the FQN of the project type to the project initializer so it can
    // generate the projenrc file.
    const js = render_options_1.renderJavaScriptOptions({
        bootstrap: true,
        comments: false,
        type: opts.type,
        args: opts.params,
    });
    const newProjectCode = `const project = new ${opts.type.typename}(${js});`;
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const module = require(mod);
    const ctx = vm.createContext(module);
    process.env.PROJEN_DISABLE_POST = (!opts.post).toString();
    vm.runInContext([
        newProjectCode,
        opts.synth ? 'project.synth();' : '',
    ].join('\n'), ctx);
}
/**
 * Given a value from "@default", processes macros and returns a stringified
 * (quoted) result.
 *
 * @returns a javascript primitive (could be a string, number or boolean)
 */
function renderDefault(value) {
    var _a;
    return (_a = macros_1.tryProcessMacro(value)) !== null && _a !== void 0 ? _a : JSON.parse(value);
}
/**
 * Converts yargs command line switches to project type props.
 * @param type Project type
 * @param argv Command line switches
 */
function commandLineToProps(type, argv) {
    var _a;
    const props = {};
    // initialize props with default values
    for (const prop of type.options) {
        if (prop.default && prop.default !== 'undefined' && !prop.optional) {
            props[prop.name] = renderDefault(prop.default);
        }
    }
    for (const [arg, value] of Object.entries(argv)) {
        for (const prop of type.options) {
            if (prop.switch === arg) {
                let curr = props;
                const queue = [...prop.path];
                while (true) {
                    const p = queue.shift();
                    if (!p) {
                        break;
                    }
                    if (queue.length === 0) {
                        curr[p] = value;
                    }
                    else {
                        curr[p] = (_a = curr[p]) !== null && _a !== void 0 ? _a : {};
                        curr = curr[p];
                    }
                }
            }
        }
    }
    return props;
}
/**
 * Generates a new project from an external module.
 *
 * @param spec The name of the external module to load
 * @param args Command line arguments (incl. project type)
 */
async function newProjectFromModule(baseDir, spec, args) {
    var _a;
    const specDependencyInfo = yarnAdd(baseDir, spec);
    // collect projects by looking up all .jsii modules in `node_modules`.
    const modulesDir = path.join(baseDir, 'node_modules');
    const modules = fs.readdirSync(modulesDir).map(file => path.join(modulesDir, file));
    const projects = inventory
        .discover(...modules)
        .filter(x => x.moduleName !== 'projen'); // filter built-in project types
    if (projects.length < 1) {
        throw new Error(`No projects found after installing ${spec}. The module must export at least one class which extends projen.Project`);
    }
    const requested = args.projectType;
    const types = projects.map(p => p.pjid);
    // if user did not specify a project type but the module has more than one, we need them to tell us which one...
    if (!requested && projects.length > 1) {
        throw new Error(`Multiple projects found after installing ${spec}: ${types.join(',')}. Please specify a project name.\nExample: npx projen new --from ${spec} ${types[0]}`);
    }
    // if user did not specify a type (and we know we have only one), the select it. otherwise, search by pjid.
    const type = !requested ? projects[0] : projects.find(p => p.pjid === requested);
    if (!type) {
        throw new Error(`Project type ${requested} not found. Found ${types.join(',')}`);
    }
    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean') {
            continue; // we don't support non-primitive fields as command line options
        }
        if (option.default && option.default !== 'undefined') {
            if (!option.optional) {
                const defaultValue = renderDefault(option.default);
                args[option.name] = defaultValue;
                args[option.switch] = defaultValue;
            }
        }
    }
    // include a dev dependency for the external module
    await newProject(baseDir, type, args, {
        devDeps: JSON.stringify([specDependencyInfo]),
    });
}
/**
 * Generates a new project.
 * @param type Project type
 * @param args Command line arguments
 * @param additionalProps Additional parameters to include in .projenrc.js
 */
async function newProject(baseDir, type, args, additionalProps) {
    // convert command line arguments to project props using type information
    const props = commandLineToProps(type, args);
    // merge in additional props if specified
    for (const [k, v] of Object.entries(additionalProps !== null && additionalProps !== void 0 ? additionalProps : {})) {
        props[k] = v;
    }
    createProject({
        dir: baseDir,
        type,
        params: props,
        comments: args.comments,
        synth: args.synth,
        post: args.post,
    });
    // interactive git and github setup
    const gitFolder = path.resolve(baseDir, '.git');
    let pushInitialToGithub = false;
    if (!fs.existsSync(gitFolder)) {
        pushInitialToGithub = await askAboutGit(baseDir);
    }
    if (pushInitialToGithub) {
        util_1.exec('git add .', { cwd: baseDir });
        util_1.exec('git commit -m \'Initial commit generated by projen\'', { cwd: baseDir });
        util_1.exec('git branch -M main', { cwd: baseDir });
        util_1.exec('git push --set-upstream origin main', { cwd: baseDir });
    }
}
/**
 * Installs the npm module (through `yarn add`) to node_modules under `projectDir`.
 * @param spec The npm package spec (e.g. foo@^1.2)
 * @returns String info for the project devDeps (e.g. foo@^1.2 or foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz)
 */
function yarnAdd(baseDir, spec) {
    const packageJsonPath = path.join(baseDir, 'package.json');
    const packageJsonExisted = fs.existsSync(packageJsonPath);
    let dependencyInfo = spec;
    // workaround: yarn fails to extract tgz if it contains '@' in the name, so we
    // create a temp copy called pkg.tgz and install from there.
    // see: https://github.com/yarnpkg/yarn/issues/6339
    if (spec.endsWith('.tgz') && spec.includes('@')) {
        // if user passes in a file spec then we have to specify the project name and the package location
        // (e.g foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz)
        const moduleName = spec.split('/').slice(-1)[0].trim().split('@')[0].trim(); // Example: ./cdk-project/dist/js/cdk-project@1.0.0.jsii.tgz
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'projen-'));
        try {
            const copy = path.join(tmpdir, 'pkg.tgz');
            fs.copyFileSync(spec, copy);
            spec = copy;
            dependencyInfo = `${moduleName}@${spec}`;
            logging.info(`installing external module ${spec}...`);
            util_1.exec(`yarn add --modules-folder=${baseDir}/node_modules --silent --no-lockfile --dev ${spec}`, { cwd: baseDir });
        }
        finally {
            // Clean up after ourselves!
            fs.removeSync(tmpdir);
        }
    }
    // if package.json did not exist before calling yarn add, we should remove it
    // so we can start off clean.
    if (!packageJsonExisted) {
        fs.removeSync(packageJsonPath);
    }
    return dependencyInfo;
}
/**
 * Returns the last path element for use as a repository name default.
 */
function repoName() {
    return path.basename(path.basename(process.cwd()));
}
async function askAboutGit(cwd) {
    logging.info('We notice that you do not have a local git repository.');
    const { setUpGit } = await inquirer.prompt([
        {
            type: 'confirm',
            name: 'setUpGit',
            message: 'Do you want to set that up now?',
        },
    ]);
    if (setUpGit) {
        const { plan } = await inquirer.prompt(githubPlanOptions);
        const { gh, git } = plan;
        if (!git && !gh) {
            util_1.exec('git init', { cwd });
            logging.info('Ok, we have run \'git init\' for you! Have a great day.');
        }
        if (git) {
            const { gitRepoURL } = await inquirer.prompt([
                {
                    type: 'input',
                    name: 'gitRepoURL',
                    message: 'What is the repo? (example: https://github.com/projen/projen)',
                },
            ]);
            util_1.exec('git init', { cwd });
            let formattedGitRepoURL = gitRepoURL;
            if (!gitRepoURL.includes('https')) {
                formattedGitRepoURL = `https://github.com/${gitRepoURL}`;
            }
            util_1.exec(`git remote add origin ${formattedGitRepoURL}`, { cwd });
            logging.info(`Great! We have run 'git init' for you and set the remote to ${formattedGitRepoURL}`);
        }
        if (!git && gh) {
            logging.info('Ok! We will make you a repository on GitHub.');
            const ghCLIPath = util_1.execOrUndefined(`${os.platform() === 'win32' ? 'where' : 'which'} gh`, { cwd });
            if (!ghCLIPath) {
                logging.warn('Looks like you do not have the GitHub CLI installed. Please go to https://cli.github.com/ to install and try again.');
            }
            else {
                const { gitProjectName } = await inquirer.prompt([
                    {
                        type: 'input',
                        name: 'gitProjectName',
                        message: 'What would you like to name it?',
                        default: repoName(),
                    },
                ]);
                logging.info(`Wow! ${gitProjectName} is such a great name!`);
                util_1.exec('git init', { cwd });
                util_1.exec(`gh repo create ${gitProjectName}`, { cwd });
                return true;
            }
        }
    }
    return false;
}
const githubPlanOptions = [
    {
        type: 'list',
        name: 'plan',
        message: 'We\'ll need some more info. Please choose one:',
        choices: [
            {
                value: {
                    git: true,
                },
                name: 'I already have a git repository',
            },
            {
                value: {
                    gh: true,
                    git: false,
                },
                name: 'I don\'t have a git repository and want to make one on GitHub',
            },
            {
                value: {
                    gh: false,
                    git: false,
                },
                name: 'I don\'t have a git repository and I don\'t want to use GitHub',
            },
        ],
    },
];
module.exports = new Command();
//# sourceMappingURL=data:application/json;base64,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