"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// tests for `projen new`: we run `projen new` for each supported project type
// and compare against a golden snapshot.
const child_process_1 = require("child_process");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const inventory = require("../inventory");
const util_1 = require("./util");
for (const type of inventory.discover()) {
    test(`projen new ${type.pjid}`, () => {
        const outdir = util_1.mkdtemp();
        try {
            const projectdir = createProjectDir(outdir);
            // execute `projen new PJID --no-synth` in the project directory
            util_1.execProjenCLI(projectdir, ['new', '--no-synth', type.pjid]);
            // compare generated .projenrc.js to the snapshot
            const actual = util_1.directorySnapshot(projectdir, {
                excludeGlobs: [
                    '.git/**',
                ],
            });
            expect(actual).toMatchSnapshot();
        }
        finally {
            fs_extra_1.removeSync(outdir);
        }
    });
}
test('post-synthesis option enabled', () => {
    const project = new util_1.TestProject();
    expect(util_1.synthSnapshotWithPost(project)['.postsynth']).toContain('postsynth');
});
test('post-synthesis option disabled', () => {
    const project = new util_1.TestProject();
    expect(util_1.synthSnapshot(project)['.postsynth']).toBeUndefined();
});
function createProjectDir(workdir) {
    // create project under "my-project" so that basedir is deterministic
    const projectdir = path_1.join(workdir, 'my-project');
    fs_extra_1.mkdirSync(projectdir);
    const git = (command) => child_process_1.execSync(`git ${command}`, { cwd: projectdir });
    git('init');
    git('remote add origin git@boom.com:foo/bar.git');
    git('config user.name "My User Name"');
    git('config user.email "my@user.email.com"');
    return projectdir;
}
//# sourceMappingURL=data:application/json;base64,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