"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderCondition = exports.Effect = exports.Targets = exports.parseLambdaPermission = exports.parseStatements = exports.Statement = void 0;
const util_1 = require("../util");
// namespace object imports won't work in the bundle for function exports
// eslint-disable-next-line @typescript-eslint/no-require-imports
const deepEqual = require('fast-deep-equal');
class Statement {
    constructor(statement) {
        if (typeof statement === 'string') {
            this.sid = undefined;
            this.effect = Effect.Unknown;
            this.resources = new Targets({}, '', '');
            this.actions = new Targets({}, '', '');
            this.principals = new Targets({}, '', '');
            this.condition = undefined;
            this.serializedIntrinsic = statement;
        }
        else {
            this.sid = expectString(statement.Sid);
            this.effect = expectEffect(statement.Effect);
            this.resources = new Targets(statement, 'Resource', 'NotResource');
            this.actions = new Targets(statement, 'Action', 'NotAction');
            this.principals = new Targets(statement, 'Principal', 'NotPrincipal');
            this.condition = statement.Condition;
            this.serializedIntrinsic = undefined;
        }
    }
    /**
     * Whether this statement is equal to the other statement
     */
    equal(other) {
        return (this.sid === other.sid
            && this.effect === other.effect
            && this.serializedIntrinsic === other.serializedIntrinsic
            && this.resources.equal(other.resources)
            && this.actions.equal(other.actions)
            && this.principals.equal(other.principals)
            && deepEqual(this.condition, other.condition));
    }
    render() {
        return this.serializedIntrinsic
            ? {
                resource: this.serializedIntrinsic,
                effect: '',
                action: '',
                principal: this.principals.render(),
                condition: '',
            }
            : {
                resource: this.resources.render(),
                effect: this.effect,
                action: this.actions.render(),
                principal: this.principals.render(),
                condition: renderCondition(this.condition),
            };
    }
    /**
     * Return a machine-readable version of the changes.
     * This is only used in tests.
     *
     * @internal
     */
    _toJson() {
        return this.serializedIntrinsic
            ? this.serializedIntrinsic
            : util_1.deepRemoveUndefined({
                sid: this.sid,
                effect: this.effect,
                resources: this.resources._toJson(),
                principals: this.principals._toJson(),
                actions: this.actions._toJson(),
                condition: this.condition,
            });
    }
    /**
     * Whether this is a negative statement
     *
     * A statement is negative if any of its targets are negative, inverted
     * if the Effect is Deny.
     */
    get isNegativeStatement() {
        const notTarget = this.actions.not || this.principals.not || this.resources.not;
        return this.effect === Effect.Allow ? notTarget : !notTarget;
    }
}
exports.Statement = Statement;
/**
 * Parse a list of statements from undefined, a Statement, or a list of statements
 */
function parseStatements(x) {
    if (x === undefined) {
        x = [];
    }
    if (!Array.isArray(x)) {
        x = [x];
    }
    return x.map((s) => new Statement(s));
}
exports.parseStatements = parseStatements;
/**
 * Parse a Statement from a Lambda::Permission object
 *
 * This is actually what Lambda adds to the policy document if you call AddPermission.
 */
function parseLambdaPermission(x) {
    // Construct a statement from
    const statement = {
        Effect: 'Allow',
        Action: x.Action,
        Resource: x.FunctionName,
    };
    if (x.Principal !== undefined) {
        if (x.Principal === '*') {
            // *
            statement.Principal = '*';
        }
        else if (/^\d{12}$/.test(x.Principal)) {
            // Account number
            statement.Principal = { AWS: `arn:aws:iam::${x.Principal}:root` };
        }
        else {
            // Assume it's a service principal
            // We might get this wrong vs. the previous one for tokens. Nothing to be done
            // about that. It's only for human readable consumption after all.
            statement.Principal = { Service: x.Principal };
        }
    }
    if (x.SourceArn !== undefined) {
        if (statement.Condition === undefined) {
            statement.Condition = {};
        }
        statement.Condition.ArnLike = { 'AWS:SourceArn': x.SourceArn };
    }
    if (x.SourceAccount !== undefined) {
        if (statement.Condition === undefined) {
            statement.Condition = {};
        }
        statement.Condition.StringEquals = { 'AWS:SourceAccount': x.SourceAccount };
    }
    if (x.EventSourceToken !== undefined) {
        if (statement.Condition === undefined) {
            statement.Condition = {};
        }
        statement.Condition.StringEquals = { 'lambda:EventSourceToken': x.EventSourceToken };
    }
    return new Statement(statement);
}
exports.parseLambdaPermission = parseLambdaPermission;
/**
 * Targets for a field
 */
class Targets {
    constructor(statement, positiveKey, negativeKey) {
        if (negativeKey in statement) {
            this.values = forceListOfStrings(statement[negativeKey]);
            this.not = true;
        }
        else {
            this.values = forceListOfStrings(statement[positiveKey]);
            this.not = false;
        }
        this.values.sort();
    }
    get empty() {
        return this.values.length === 0;
    }
    /**
     * Whether this set of targets is equal to the other set of targets
     */
    equal(other) {
        return this.not === other.not && deepEqual(this.values.sort(), other.values.sort());
    }
    /**
     * If the current value set is empty, put this in it
     */
    replaceEmpty(replacement) {
        if (this.empty) {
            this.values.push(replacement);
        }
    }
    /**
     * If the actions contains a '*', replace with this string.
     */
    replaceStar(replacement) {
        for (let i = 0; i < this.values.length; i++) {
            if (this.values[i] === '*') {
                this.values[i] = replacement;
            }
        }
        this.values.sort();
    }
    /**
     * Render into a summary table cell
     */
    render() {
        return this.not
            ? this.values.map(s => `NOT ${s}`).join('\n')
            : this.values.join('\n');
    }
    /**
     * Return a machine-readable version of the changes.
     * This is only used in tests.
     *
     * @internal
     */
    _toJson() {
        return { not: this.not, values: this.values };
    }
}
exports.Targets = Targets;
var Effect;
(function (Effect) {
    Effect["Unknown"] = "Unknown";
    Effect["Allow"] = "Allow";
    Effect["Deny"] = "Deny";
})(Effect = exports.Effect || (exports.Effect = {}));
function expectString(x) {
    return typeof x === 'string' ? x : undefined;
}
function expectEffect(x) {
    if (x === Effect.Allow || x === Effect.Deny) {
        return x;
    }
    return Effect.Unknown;
}
function forceListOfStrings(x) {
    if (typeof x === 'string') {
        return [x];
    }
    if (typeof x === 'undefined' || x === null) {
        return [];
    }
    if (Array.isArray(x)) {
        return x.map(e => forceListOfStrings(e).join(','));
    }
    if (typeof x === 'object' && x !== null) {
        const ret = [];
        for (const [key, value] of Object.entries(x)) {
            ret.push(...forceListOfStrings(value).map(s => `${key}:${s}`));
        }
        return ret;
    }
    return [`${x}`];
}
/**
 * Render the Condition column
 */
function renderCondition(condition) {
    if (!condition || Object.keys(condition).length === 0) {
        return '';
    }
    const jsonRepresentation = JSON.stringify(condition, undefined, 2);
    // The JSON representation looks like this:
    //
    //  {
    //    "ArnLike": {
    //      "AWS:SourceArn": "${MyTopic86869434}"
    //    }
    //  }
    //
    // We can make it more compact without losing information by getting rid of the outermost braces
    // and the indentation.
    const lines = jsonRepresentation.split('\n');
    return lines.slice(1, lines.length - 1).map(s => s.substr(2)).join('\n');
}
exports.renderCondition = renderCondition;
//# sourceMappingURL=data:application/json;base64,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