"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FactName = exports.Fact = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_entities_1 = require("./aws-entities");
/**
 * A database of regional information.
 */
class Fact {
    constructor() {
        throw new Error('Use the static methods of Fact instead!');
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     *          may not be an exhaustive list of all available AWS regions.
     */
    static get regions() {
        // Return by copy to ensure no modifications can be made to the undelying constant.
        return Array.from(aws_entities_1.AWS_REGIONS);
    }
    /**
     * Retrieves a fact from this Fact database.
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name   the name of the fact being looked up (see the `FactName` class for details)
     *
     * @returns the fact value if it is known, and `undefined` otherwise.
     */
    static find(region, name) {
        const regionFacts = this.database[region];
        return regionFacts && regionFacts[name];
    }
    /**
     * Retrieve a fact from the Fact database. (retrieval will fail if the specified region or
     * fact name does not exist.)
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name the name of the fact being looked up (see the `FactName` class for details)
     */
    static requireFact(region, name) {
        const foundFact = this.find(region, name);
        if (!foundFact) {
            throw new Error(`No fact ${name} could be found for region: ${region} and name: ${name}`);
        }
        return foundFact;
    }
    /**
     * Registers a new fact in this Fact database.
     *
     * @param fact           the new fact to be registered.
     * @param allowReplacing whether new facts can replace existing facts or not.
     */
    static register(fact, allowReplacing = false) {
        jsiiDeprecationWarnings._aws_cdk_region_info_IFact(fact);
        const regionFacts = this.database[fact.region] || (this.database[fact.region] = {});
        if (fact.name in regionFacts && regionFacts[fact.name] !== fact.value && !allowReplacing) {
            throw new Error(`Region ${fact.region} already has a fact ${fact.name}, with value ${regionFacts[fact.name]}`);
        }
        if (fact.value !== undefined) {
            regionFacts[fact.name] = fact.value;
        }
    }
    /**
     * Removes a fact from the database.
     * @param region the region for which the fact is to be removed.
     * @param name   the name of the fact to remove.
     * @param value  the value that should be removed (removal will fail if the value is specified, but does not match the
     *               current stored value).
     */
    static unregister(region, name, value) {
        const regionFacts = this.database[region] || {};
        if (name in regionFacts && value && regionFacts[name] !== value) {
            throw new Error(`Attempted to remove ${name} from ${region} with value ${value}, but the fact's value is ${regionFacts[name]}`);
        }
        delete regionFacts[name];
    }
}
exports.Fact = Fact;
_a = JSII_RTTI_SYMBOL_1;
Fact[_a] = { fqn: "@aws-cdk/region-info.Fact", version: "1.146.0" };
Fact.database = {};
/**
 * All standardized fact names.
 */
class FactName {
    /**
     * The ARN of CloudWatch Lambda Insights for a version (e.g. 1.0.98.0)
     */
    static cloudwatchLambdaInsightsVersion(version, arch) {
        // if we are provided an architecture use that, otherwise
        // default to x86_64 for backwards compatibility
        const suffix = version.split('.').join('_') + `_${arch !== null && arch !== void 0 ? arch : 'x86_64'}`;
        return `cloudwatch-lambda-insights-version:${suffix}`;
    }
    /**
     * The name of the regional service principal for a given service.
     *
     * @param service the service name, either simple (e.g: `s3`, `codedeploy`) or qualified (e.g: `s3.amazonaws.com`).
     *                The `.amazonaws.com` and `.amazonaws.com.cn` domains are stripped from service names, so they are
     *                canonicalized in that respect.
     */
    static servicePrincipal(service) {
        return `service-principal:${service.replace(/\.amazonaws\.com(\.cn)?$/, '')}`;
    }
}
exports.FactName = FactName;
_b = JSII_RTTI_SYMBOL_1;
FactName[_b] = { fqn: "@aws-cdk/region-info.FactName", version: "1.146.0" };
/**
 * The name of the partition for a region (e.g: 'aws', 'aws-cn', ...)
 */
FactName.PARTITION = 'partition';
/**
 * The domain suffix for a region (e.g: 'amazonaws.com`)
 */
FactName.DOMAIN_SUFFIX = 'domainSuffix';
/**
 * Whether the AWS::CDK::Metadata CloudFormation Resource is available in-region or not. The value is a boolean
 * modelled as `YES` or `NO`.
 */
FactName.CDK_METADATA_RESOURCE_AVAILABLE = 'cdk:metadata-resource:available';
/**
 * The endpoint used for hosting S3 static websites
 */
FactName.S3_STATIC_WEBSITE_ENDPOINT = 's3-static-website:endpoint';
/**
 * The endpoint used for aliasing S3 static websites in Route 53
 */
FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID = 's3-static-website:route-53-hosted-zone-id';
/**
 * The hosted zone ID used by Route 53 to alias a EBS environment endpoint in this region (e.g: Z2O1EMRO9K5GLX)
 */
FactName.EBS_ENV_ENDPOINT_HOSTED_ZONE_ID = 'ebs-environment:route-53-hosted-zone-id';
/**
 * The prefix for VPC Endpoint Service names,
 * cn.com.amazonaws.vpce for China regions,
 * com.amazonaws.vpce otherwise.
 */
FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'vpcEndpointServiceNamePrefix';
/**
 * The account for ELBv2 in this region
 */
FactName.ELBV2_ACCOUNT = 'elbv2Account';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the
 * AWS Deep Learning Containers images in a given region.
 */
FactName.DLC_REPOSITORY_ACCOUNT = 'dlcRepositoryAccount';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the
 * AWS App Mesh Envoy Proxy images in a given region.
 */
FactName.APPMESH_ECR_ACCOUNT = 'appMeshRepositoryAccount';
/**
 * The CIDR block used by Kinesis Data Firehose servers.
 */
FactName.FIREHOSE_CIDR_BLOCK = 'firehoseCidrBlock';
//# sourceMappingURL=data:application/json;base64,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