"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatTable = void 0;
const chalk = require("chalk");
const stringWidth = require("string-width");
const table = require("table");
/**
 * Render a two-dimensional array to a visually attractive table
 *
 * First row is considered the table header.
 */
function formatTable(cells, columns) {
    return table.table(cells, {
        border: TABLE_BORDER_CHARACTERS,
        columns: buildColumnConfig(columns !== undefined ? calculcateColumnWidths(cells, columns) : undefined),
        drawHorizontalLine: (line) => {
            // Numbering like this: [line 0] [header = row[0]] [line 1] [row 1] [line 2] [content 2] [line 3]
            return (line < 2 || line === cells.length) || lineBetween(cells[line - 1], cells[line]);
        },
    }).trimRight();
}
exports.formatTable = formatTable;
/**
 * Whether we should draw a line between two rows
 *
 * Draw horizontal line if 2nd column values are different.
 */
function lineBetween(rowA, rowB) {
    return rowA[1] !== rowB[1];
}
function buildColumnConfig(widths) {
    if (widths === undefined) {
        return undefined;
    }
    const ret = {};
    widths.forEach((width, i) => {
        if (width === undefined) {
            return;
        }
        ret[i] = { width };
    });
    return ret;
}
/**
 * Calculate column widths given a terminal width
 *
 * We do this by calculating a fair share for every column. Extra width smaller
 * than the fair share is evenly distributed over all columns that exceed their
 * fair share.
 */
function calculcateColumnWidths(rows, terminalWidth) {
    // The terminal is sometimes reported to be 0. Also if the terminal is VERY narrow,
    // just assume a reasonable minimum size.
    terminalWidth = Math.max(terminalWidth, 40);
    // use 'string-width' to not count ANSI chars as actual character width
    const columns = rows[0].map((_, i) => Math.max(...rows.map(row => stringWidth(String(row[i])))));
    // If we have no terminal width, do nothing
    const contentWidth = terminalWidth - 2 - columns.length * 3;
    // If we don't exceed the terminal width, do nothing
    if (sum(columns) <= contentWidth) {
        return columns;
    }
    const fairShare = Math.min(contentWidth / columns.length);
    const smallColumns = columns.filter(w => w < fairShare);
    let distributableWidth = contentWidth - sum(smallColumns);
    const fairDistributable = Math.floor(distributableWidth / (columns.length - smallColumns.length));
    const ret = new Array();
    for (const requestedWidth of columns) {
        if (requestedWidth < fairShare) {
            // Small column gets what they want
            ret.push(requestedWidth);
        }
        else {
            // Last column gets all remaining, otherwise get fair redist share
            const width = distributableWidth < 2 * fairDistributable ? distributableWidth : fairDistributable;
            ret.push(width);
            distributableWidth -= width;
        }
    }
    return ret;
}
function sum(xs) {
    let total = 0;
    for (const x of xs) {
        total += x;
    }
    return total;
}
// What color the table is going to be
const tableColor = chalk.gray;
// Unicode table characters with a color
const TABLE_BORDER_CHARACTERS = {
    topBody: tableColor('─'),
    topJoin: tableColor('┬'),
    topLeft: tableColor('┌'),
    topRight: tableColor('┐'),
    bottomBody: tableColor('─'),
    bottomJoin: tableColor('┴'),
    bottomLeft: tableColor('└'),
    bottomRight: tableColor('┘'),
    bodyLeft: tableColor('│'),
    bodyRight: tableColor('│'),
    bodyJoin: tableColor('│'),
    joinBody: tableColor('─'),
    joinLeft: tableColor('├'),
    joinRight: tableColor('┤'),
    joinJoin: tableColor('┼'),
};
//# sourceMappingURL=data:application/json;base64,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