"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FactName = exports.Fact = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_entities_1 = require("./aws-entities");
/**
 * A database of regional information.
 *
 * @stability stable
 */
class Fact {
    constructor() {
        throw new Error('Use the static methods of Fact instead!');
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     * may not be an exhaustive list of all available AWS regions.
     * @stability stable
     */
    static get regions() {
        // Return by copy to ensure no modifications can be made to the undelying constant.
        return Array.from(aws_entities_1.AWS_REGIONS);
    }
    /**
     * Retrieves a fact from this Fact database.
     *
     * @param region the name of the region (e.g: `us-east-1`).
     * @param name the name of the fact being looked up (see the `FactName` class for details).
     * @returns the fact value if it is known, and `undefined` otherwise.
     * @stability stable
     */
    static find(region, name) {
        const regionFacts = this.database[region];
        return regionFacts && regionFacts[name];
    }
    /**
     * Retrieve a fact from the Fact database.
     *
     * (retrieval will fail if the specified region or
     * fact name does not exist.)
     *
     * @param region the name of the region (e.g: `us-east-1`).
     * @param name the name of the fact being looked up (see the `FactName` class for details).
     * @stability stable
     */
    static requireFact(region, name) {
        const foundFact = this.find(region, name);
        if (!foundFact) {
            throw new Error(`No fact ${name} could be found for region: ${region} and name: ${name}`);
        }
        return foundFact;
    }
    /**
     * Registers a new fact in this Fact database.
     *
     * @param fact the new fact to be registered.
     * @param allowReplacing whether new facts can replace existing facts or not.
     * @stability stable
     */
    static register(fact, allowReplacing = false) {
        jsiiDeprecationWarnings._aws_cdk_region_info_IFact(fact);
        const regionFacts = this.database[fact.region] || (this.database[fact.region] = {});
        if (fact.name in regionFacts && regionFacts[fact.name] !== fact.value && !allowReplacing) {
            throw new Error(`Region ${fact.region} already has a fact ${fact.name}, with value ${regionFacts[fact.name]}`);
        }
        if (fact.value !== undefined) {
            regionFacts[fact.name] = fact.value;
        }
    }
    /**
     * Removes a fact from the database.
     *
     * @param region the region for which the fact is to be removed.
     * @param name the name of the fact to remove.
     * @param value the value that should be removed (removal will fail if the value is specified, but does not match the current stored value).
     * @stability stable
     */
    static unregister(region, name, value) {
        const regionFacts = this.database[region] || {};
        if (name in regionFacts && value && regionFacts[name] !== value) {
            throw new Error(`Attempted to remove ${name} from ${region} with value ${value}, but the fact's value is ${regionFacts[name]}`);
        }
        delete regionFacts[name];
    }
}
exports.Fact = Fact;
_a = JSII_RTTI_SYMBOL_1;
Fact[_a] = { fqn: "@aws-cdk/region-info.Fact", version: "1.144.0" };
Fact.database = {};
/**
 * All standardized fact names.
 *
 * @stability stable
 */
class FactName {
    /**
     * The ARN of CloudWatch Lambda Insights for a version (e.g. 1.0.98.0).
     *
     * @stability stable
     */
    static cloudwatchLambdaInsightsVersion(version, arch) {
        // if we are provided an architecture use that, otherwise
        // default to x86_64 for backwards compatibility
        const suffix = version.split('.').join('_') + `_${arch !== null && arch !== void 0 ? arch : 'x86_64'}`;
        return `cloudwatch-lambda-insights-version:${suffix}`;
    }
    /**
     * The name of the regional service principal for a given service.
     *
     * @param service the service name, either simple (e.g: `s3`, `codedeploy`) or qualified (e.g: `s3.amazonaws.com`). The `.amazonaws.com` and `.amazonaws.com.cn` domains are stripped from service names, so they are canonicalized in that respect.
     * @stability stable
     */
    static servicePrincipal(service) {
        return `service-principal:${service.replace(/\.amazonaws\.com(\.cn)?$/, '')}`;
    }
}
exports.FactName = FactName;
_b = JSII_RTTI_SYMBOL_1;
FactName[_b] = { fqn: "@aws-cdk/region-info.FactName", version: "1.144.0" };
/**
 * The name of the partition for a region (e.g: 'aws', 'aws-cn', ...).
 *
 * @stability stable
 */
FactName.PARTITION = 'partition';
/**
 * The domain suffix for a region (e.g: 'amazonaws.com`).
 *
 * @stability stable
 */
FactName.DOMAIN_SUFFIX = 'domainSuffix';
/**
 * Whether the AWS::CDK::Metadata CloudFormation Resource is available in-region or not.
 *
 * The value is a boolean
 * modelled as `YES` or `NO`.
 *
 * @stability stable
 */
FactName.CDK_METADATA_RESOURCE_AVAILABLE = 'cdk:metadata-resource:available';
/**
 * The endpoint used for hosting S3 static websites.
 *
 * @stability stable
 */
FactName.S3_STATIC_WEBSITE_ENDPOINT = 's3-static-website:endpoint';
/**
 * The endpoint used for aliasing S3 static websites in Route 53.
 *
 * @stability stable
 */
FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID = 's3-static-website:route-53-hosted-zone-id';
/**
 * The hosted zone ID used by Route 53 to alias a EBS environment endpoint in this region (e.g: Z2O1EMRO9K5GLX).
 *
 * @stability stable
 */
FactName.EBS_ENV_ENDPOINT_HOSTED_ZONE_ID = 'ebs-environment:route-53-hosted-zone-id';
/**
 * The prefix for VPC Endpoint Service names, cn.com.amazonaws.vpce for China regions, com.amazonaws.vpce otherwise.
 *
 * @stability stable
 */
FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'vpcEndpointServiceNamePrefix';
/**
 * The account for ELBv2 in this region.
 *
 * @stability stable
 */
FactName.ELBV2_ACCOUNT = 'elbv2Account';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the AWS Deep Learning Containers images in a given region.
 *
 * @stability stable
 */
FactName.DLC_REPOSITORY_ACCOUNT = 'dlcRepositoryAccount';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the AWS App Mesh Envoy Proxy images in a given region.
 *
 * @stability stable
 */
FactName.APPMESH_ECR_ACCOUNT = 'appMeshRepositoryAccount';
/**
 * The CIDR block used by Kinesis Data Firehose servers.
 *
 * @stability stable
 */
FactName.FIREHOSE_CIDR_BLOCK = 'firehoseCidrBlock';
//# sourceMappingURL=data:application/json;base64,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