"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateRegionMap = exports.partitionInformation = exports.regionsBefore = exports.before = exports.AWS_SERVICES = exports.AWS_REGIONS = exports.AWS_REGIONS_AND_RULES = exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN = exports.RULE_SSM_PRINCIPALS_ARE_REGIONAL = void 0;
/**
 * After this point, SSM only creates regional principals
 */
exports.RULE_SSM_PRINCIPALS_ARE_REGIONAL = Symbol('SSM_PRINCIPALS_ARE_REGIONAL');
/**
 * After this point, S3 website domains look like `s3-website.REGION.s3.amazonaws.com`
 *
 * Before this point, S3 website domains look like `s3-website-REGION.s3.amazonaws.com`.
 */
exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN = Symbol('S3_WEBSITE_REGIONAL_SUBDOMAIN');
/**
 * List of AWS region, ordered by launch date (oldest to newest)
 *
 * The significance of this is that standards and conventions change over time.
 * Generally, as rules are changed they only apply to new regions, and old
 * regions are left as-is.
 *
 * We mix the list of regions with a list of rules that were introduced over
 * time (rules are symbols).
 *
 * Therefore, if we want to know if a rule applies to a certain region, we
 * only need to check its position in the list and compare it to when a
 * rule was introduced.
 */
exports.AWS_REGIONS_AND_RULES = [
    'us-east-1',
    'eu-west-1',
    'us-west-1',
    'ap-southeast-1',
    'ap-northeast-1',
    'us-gov-west-1',
    'us-west-2',
    'sa-east-1',
    'ap-southeast-2',
    exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN,
    'cn-north-1',
    'us-iso-east-1',
    'eu-central-1',
    'ap-northeast-2',
    'ap-south-1',
    'us-east-2',
    'ca-central-1',
    'eu-west-2',
    'us-isob-east-1',
    'cn-northwest-1',
    'eu-west-3',
    'ap-northeast-3',
    'us-gov-east-1',
    'eu-north-1',
    exports.RULE_SSM_PRINCIPALS_ARE_REGIONAL,
    'ap-east-1',
    'me-south-1',
    'eu-south-1',
    'af-south-1',
    'us-iso-west-1',
    'eu-south-2',
    'ap-southeast-3',
];
/**
 * The names of all (known) AWS regions
 *
 * Not in the list ==> no built-in data for that region.
 */
exports.AWS_REGIONS = exports.AWS_REGIONS_AND_RULES
    .filter((x) => typeof x === 'string')
    .sort();
/**
 * Possibly non-exaustive list of all service names, used to locate service principals.
 *
 * Not in the list ==> default service principal mappings.
 */
exports.AWS_SERVICES = [
    'application-autoscaling',
    'autoscaling',
    'codedeploy',
    'ec2',
    'events',
    'lambda',
    'logs',
    's3',
    'ssm',
    'sns',
    'sqs',
    'states',
].sort();
/**
 * Whether or not a region predates a given rule (or region).
 *
 * Unknown region => we have to assume no.
 */
function before(region, ruleOrRegion) {
    const ruleIx = exports.AWS_REGIONS_AND_RULES.indexOf(ruleOrRegion);
    if (ruleIx === -1) {
        throw new Error(`Unknown rule: ${String(ruleOrRegion)}`);
    }
    const regionIx = exports.AWS_REGIONS_AND_RULES.indexOf(region);
    return regionIx === -1 ? false : regionIx < ruleIx;
}
exports.before = before;
/**
 * Return all regions before a given rule was introduced (or region)
 */
function regionsBefore(ruleOrRegion) {
    const ruleIx = exports.AWS_REGIONS_AND_RULES.indexOf(ruleOrRegion);
    if (ruleIx === -1) {
        throw new Error(`Unknown rule: ${String(ruleOrRegion)}`);
    }
    return exports.AWS_REGIONS_AND_RULES.slice(0, ruleIx)
        .filter((entry) => typeof entry === 'string')
        .sort();
}
exports.regionsBefore = regionsBefore;
const PARTITION_MAP = {
    'default': { partition: 'aws', domainSuffix: 'amazonaws.com' },
    'cn-': { partition: 'aws-cn', domainSuffix: 'amazonaws.com.cn' },
    'us-gov-': { partition: 'aws-us-gov', domainSuffix: 'amazonaws.com' },
    'us-iso-': { partition: 'aws-iso', domainSuffix: 'c2s.ic.gov' },
    'us-isob-': { partition: 'aws-iso-b', domainSuffix: 'sc2s.sgov.gov' },
};
function partitionInformation(region) {
    for (const [prefix, info] of Object.entries(PARTITION_MAP)) {
        if (region.startsWith(prefix)) {
            return info;
        }
    }
    return PARTITION_MAP.default;
}
exports.partitionInformation = partitionInformation;
/**
 * Build a lookup map for all regions
 */
function generateRegionMap(cb) {
    const ret = {};
    for (const region of exports.AWS_REGIONS) {
        ret[region] = cb(region);
    }
    return ret;
}
exports.generateRegionMap = generateRegionMap;
//# sourceMappingURL=data:application/json;base64,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