"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.synth = void 0;
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const common_1 = require("../common");
const logging = require("../logging");
const project_1 = require("../project");
const projenModule = path.dirname(require.resolve('../../package.json'));
async function synth(runtime, options) {
    var _a;
    const workdir = runtime.workdir;
    const rcfile = path.resolve(workdir, (_a = options.rcfile) !== null && _a !== void 0 ? _a : common_1.PROJEN_RC); // TODO: support non javascript projenrc (e.g. java projects)
    // if --rc points to .projenrc.js, then behave as if it wasn't specified.
    if (rcfile === path.resolve(workdir, common_1.PROJEN_RC)) {
        delete options.rcfile;
    }
    // if there are no tasks, we assume this is not a projen project (modern
    // projects must at least have the "default" task).
    if (runtime.tasks.length === 0 && !fs.existsSync(rcfile)) {
        logging.error('Unable to find projen project. Use "projen new" to create a new project.');
        process.exit(1);
    }
    // run synth once
    const success = await trySynth();
    if (options.watch) {
        // if we are in watch mode, start the watch loop
        watchLoop();
    }
    else if (!success) {
        // make sure exit code is non-zero if we are not in watch mode
        process.exit(1);
    }
    async function trySynth() {
        // determine if post synthesis tasks should be executed (e.g. "yarn install").
        process.env.PROJEN_DISABLE_POST = (!options.post).toString();
        try {
            const defaultTask = runtime.tasks.find(t => t.name === project_1.Project.DEFAULT_TASK);
            // if "--rc" is specified, ignore the default task
            if (defaultTask) {
                if (!options.rcfile) {
                    runtime.runTask(defaultTask.name);
                    return true;
                }
                else {
                    logging.warn('Default task skipped. Trying legacy synthesis since --rc is specified');
                }
            }
            // for backwards compatibility, if there is a .projenrc.js file, default to "node .projenrc.js"
            if (tryLegacySynth()) {
                return true;
            }
            throw new Error('Unable to find a task named "default"');
        }
        catch (e) {
            logging.error(`Synthesis failed: ${e.message}`);
            return false;
        }
    }
    function watchLoop() {
        if (!fs.existsSync(rcfile)) {
            throw new Error(`--watch is only supported for projects with "${rcfile}"`);
        }
        logging.info(`Watching for changes in ${rcfile}...`);
        const watch = fs.watch(rcfile);
        watch.on('change', () => {
            process.stdout.write('\x1Bc'); // clear screen
            watch.close();
            trySynth()
                .then(() => watchLoop())
                .catch(() => watchLoop());
        });
    }
    function tryLegacySynth() {
        const rcdir = path.dirname(rcfile);
        if (!fs.existsSync(rcfile)) {
            return false;
        }
        // if node_modules/projen is not a directory or does not exist, create a
        // temporary symlink to the projen that we are currently running in order to
        // allow .projenrc.js to `require()` it.
        const nodeModules = path.resolve(rcdir, 'node_modules');
        const projenModulePath = path.resolve(nodeModules, 'projen');
        if (!fs.existsSync(path.join(projenModulePath, 'package.json')) || !fs.statSync(projenModulePath).isDirectory()) {
            fs.removeSync(projenModulePath);
            fs.mkdirpSync(nodeModules);
            fs.symlinkSync(projenModule, projenModulePath, (os.platform() === 'win32') ? 'junction' : null);
        }
        child_process_1.spawnSync(process.execPath, [rcfile], { stdio: 'inherit' });
        return true;
    }
}
exports.synth = synth;
//# sourceMappingURL=data:application/json;base64,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