"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.directorySnapshot = exports.synthSnapshotWithPost = exports.synthSnapshot = exports.mkdtemp = exports.execProjenCLI = exports.TestProject = void 0;
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const glob_1 = require("glob");
const __1 = require("..");
const logging = require("../logging");
const util_1 = require("../util");
const PROJEN_CLI = require.resolve('../../bin/projen');
logging.disable(); // no logging during tests
class TestProject extends __1.Project {
    constructor(options = {}) {
        const tmpdir = mkdtemp();
        super({
            name: 'my-project',
            outdir: tmpdir,
            clobber: false,
            logging: {
                level: __1.LogLevel.OFF,
            },
            ...options,
        });
    }
    postSynthesize() {
        fs.writeFileSync(path.join(this.outdir, '.postsynth'), '# postsynth');
    }
}
exports.TestProject = TestProject;
function execProjenCLI(workdir, args = []) {
    const command = [
        process.execPath,
        PROJEN_CLI,
        ...args,
    ];
    return util_1.exec(command.map(x => `"${x}"`).join(' '), { cwd: workdir, stdio: 'inherit' });
}
exports.execProjenCLI = execProjenCLI;
function mkdtemp() {
    return fs.mkdtempSync(path.join(os.tmpdir(), 'projen-test-'));
}
exports.mkdtemp = mkdtemp;
function synthSnapshot(project) {
    const ENV_PROJEN_DISABLE_POST = process.env.PROJEN_DISABLE_POST;
    try {
        process.env.PROJEN_DISABLE_POST = 'true';
        project.synth();
        return directorySnapshot(project.outdir);
    }
    finally {
        fs.removeSync(project.outdir);
        // values assigned to process.env.XYZ are automatically converted to strings
        if (ENV_PROJEN_DISABLE_POST === undefined) {
            delete process.env.PROJEN_DISABLE_POST;
        }
        else {
            process.env.PROJEN_DISABLE_POST = ENV_PROJEN_DISABLE_POST;
        }
    }
}
exports.synthSnapshot = synthSnapshot;
function synthSnapshotWithPost(project) {
    try {
        project.synth();
        return directorySnapshot(project.outdir);
    }
    finally {
        fs.removeSync(project.outdir);
    }
}
exports.synthSnapshotWithPost = synthSnapshotWithPost;
function directorySnapshot(root, options = {}) {
    var _a;
    const output = {};
    const files = glob_1.glob.sync('**', {
        ignore: (_a = options.excludeGlobs) !== null && _a !== void 0 ? _a : [],
        cwd: root,
        nodir: true,
        dot: true,
    }); // returns relative file paths with POSIX separators
    for (const file of files) {
        const filePath = path.join(root, file);
        let content;
        if (path.extname(filePath) === '.json') {
            content = fs.readJsonSync(filePath);
        }
        else {
            content = fs.readFileSync(filePath, 'utf-8');
        }
        output[file] = content;
    }
    return output;
}
exports.directorySnapshot = directorySnapshot;
//# sourceMappingURL=data:application/json;base64,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