"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Default = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_entities_1 = require("./aws-entities");
/**
 * Provides default values for certain regional information points.
 *
 * @stability stable
 */
class Default {
    constructor() { }
    /**
     * Computes a "standard" AWS Service principal for a given service, region and suffix.
     *
     * This is useful for example when
     * you need to compute a service principal name, but you do not have a synthesize-time region literal available (so
     * all you have is `{ "Ref": "AWS::Region" }`). This way you get the same defaulting behavior that is normally used
     * for built-in data.
     *
     * @param serviceFqn the name of the service (s3, s3.amazonaws.com, ...).
     * @param region the region in which the service principal is needed.
     * @param urlSuffix deprecated and ignored.
     * @stability stable
     */
    static servicePrincipal(serviceFqn, region, urlSuffix) {
        const serviceName = extractSimpleName(serviceFqn);
        if (!serviceName) {
            // Return "service" if it does not look like any of the following:
            // - s3
            // - s3.amazonaws.com
            // - s3.amazonaws.com.cn
            // - s3.c2s.ic.gov
            // - s3.sc2s.sgov.gov
            return serviceFqn;
        }
        function determineConfiguration(service) {
            function universal(s) { return `${s}.amazonaws.com`; }
            ;
            function partitional(s, _, u) { return `${s}.${u}`; }
            ;
            function regional(s, r) { return `${s}.${r}.amazonaws.com`; }
            ;
            function regionalPartitional(s, r, u) { return `${s}.${r}.${u}`; }
            ;
            // Exceptions for Service Principals in us-iso-*
            const US_ISO_EXCEPTIONS = new Set([
                'cloudhsm',
                'config',
                'states',
                'workspaces',
            ]);
            // Account for idiosyncratic Service Principals in `us-iso-*` regions
            if (region.startsWith('us-iso-') && US_ISO_EXCEPTIONS.has(service)) {
                switch (service) {
                    // Services with universal principal
                    case ('states'):
                        return universal;
                    // Services with a partitional principal
                    default:
                        return partitional;
                }
            }
            // Exceptions for Service Principals in us-isob-*
            const US_ISOB_EXCEPTIONS = new Set([
                'dms',
                'states',
            ]);
            // Account for idiosyncratic Service Principals in `us-isob-*` regions
            if (region.startsWith('us-isob-') && US_ISOB_EXCEPTIONS.has(service)) {
                switch (service) {
                    // Services with universal principal
                    case ('states'):
                        return universal;
                    // Services with a partitional principal
                    default:
                        return partitional;
                }
            }
            switch (service) {
                // SSM turned from global to regional at some point
                case 'ssm':
                    return aws_entities_1.before(region, aws_entities_1.RULE_SSM_PRINCIPALS_ARE_REGIONAL)
                        ? universal
                        : regional;
                // CodeDeploy is regional+partitional in CN, only regional everywhere else
                case 'codedeploy':
                    return region.startsWith('cn-')
                        ? regionalPartitional
                        : regional;
                // Services with a regional AND partitional principal
                case 'logs':
                    return regionalPartitional;
                // Services with a regional principal
                case 'states':
                    return regional;
                // Services with a partitional principal
                case 'ec2':
                    return partitional;
                // Services with a universal principal across all regions/partitions (the default case)
                default:
                    return universal;
            }
        }
        ;
        const configuration = determineConfiguration(serviceName);
        return configuration(serviceName, region, urlSuffix);
    }
}
exports.Default = Default;
_a = JSII_RTTI_SYMBOL_1;
Default[_a] = { fqn: "@aws-cdk/region-info.Default", version: "1.142.0" };
/**
 * The default value for a VPC Endpoint Service name prefix, useful if you do not have a synthesize-time region literal available (all you have is `{ "Ref": "AWS::Region" }`).
 *
 * @stability stable
 */
Default.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'com.amazonaws.vpce';
function extractSimpleName(serviceFqn) {
    const matches = serviceFqn.match(/^([^.]+)(?:(?:\.amazonaws\.com(?:\.cn)?)|(?:\.c2s\.ic\.gov)|(?:\.sc2s\.sgov\.gov))?$/);
    return matches ? matches[1] : undefined;
}
//# sourceMappingURL=data:application/json;base64,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