"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Manifest = exports.VERSION_MISMATCH = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const jsonschema = require("jsonschema");
const semver = require("semver");
const assembly = require("./cloud-assembly");
/* eslint-disable @typescript-eslint/no-var-requires */
/* eslint-disable @typescript-eslint/no-require-imports */
// this prefix is used by the CLI to identify this specific error.
// in which case we want to instruct the user to upgrade his CLI.
// see exec.ts#createAssembly
exports.VERSION_MISMATCH = 'Cloud assembly schema version mismatch';
const ASSETS_SCHEMA = require('../schema/assets.schema.json');
const ASSEMBLY_SCHEMA = require('../schema/cloud-assembly.schema.json');
/**
 * Version is shared for both manifests
 */
const SCHEMA_VERSION = require('../schema/cloud-assembly.version.json').version;
/**
 * Protocol utility class.
 *
 * @stability stable
 */
class Manifest {
    constructor() { }
    /**
     * Validates and saves the cloud assembly manifest to file.
     *
     * @param manifest - manifest.
     * @param filePath - output file path.
     * @stability stable
     */
    static saveAssemblyManifest(manifest, filePath) {
        jsiiDeprecationWarnings._aws_cdk_cloud_assembly_schema_AssemblyManifest(manifest);
        Manifest.saveManifest(manifest, filePath, ASSEMBLY_SCHEMA, Manifest.patchStackTagsOnWrite);
    }
    /**
     * Load and validates the cloud assembly manifest from file.
     *
     * @param filePath - path to the manifest file.
     * @stability stable
     */
    static loadAssemblyManifest(filePath) {
        return Manifest.loadManifest(filePath, ASSEMBLY_SCHEMA, Manifest.patchStackTagsOnRead);
    }
    /**
     * Validates and saves the asset manifest to file.
     *
     * @param manifest - manifest.
     * @param filePath - output file path.
     * @stability stable
     */
    static saveAssetManifest(manifest, filePath) {
        jsiiDeprecationWarnings._aws_cdk_cloud_assembly_schema_AssetManifest(manifest);
        Manifest.saveManifest(manifest, filePath, ASSETS_SCHEMA, Manifest.patchStackTagsOnRead);
    }
    /**
     * Load and validates the asset manifest from file.
     *
     * @param filePath - path to the manifest file.
     * @stability stable
     */
    static loadAssetManifest(filePath) {
        return this.loadManifest(filePath, ASSETS_SCHEMA);
    }
    /**
     * Fetch the current schema version number.
     *
     * @stability stable
     */
    static version() {
        return SCHEMA_VERSION;
    }
    /**
     * (deprecated) Deprecated.
     *
     * @deprecated use `saveAssemblyManifest()`
     */
    static save(manifest, filePath) { jsiiDeprecationWarnings.print("@aws-cdk/cloud-assembly-schema.Manifest#save", "use `saveAssemblyManifest()`"); jsiiDeprecationWarnings._aws_cdk_cloud_assembly_schema_AssemblyManifest(manifest); return this.saveAssemblyManifest(manifest, filePath); }
    /**
     * (deprecated) Deprecated.
     *
     * @deprecated use `loadAssemblyManifest()`
     */
    static load(filePath) { jsiiDeprecationWarnings.print("@aws-cdk/cloud-assembly-schema.Manifest#load", "use `loadAssemblyManifest()`"); return this.loadAssemblyManifest(filePath); }
    static validate(manifest, schema) {
        function parseVersion(version) {
            const ver = semver.valid(version);
            if (!ver) {
                throw new Error(`Invalid semver string: "${version}"`);
            }
            return ver;
        }
        const maxSupported = parseVersion(Manifest.version());
        const actual = parseVersion(manifest.version);
        // first validate the version should be accepted.
        if (semver.gt(actual, maxSupported)) {
            // we use a well known error prefix so that the CLI can identify this specific error
            // and print some more context to the user.
            throw new Error(`${exports.VERSION_MISMATCH}: Maximum schema version supported is ${maxSupported}, but found ${actual}`);
        }
        // now validate the format is good.
        const validator = new jsonschema.Validator();
        const result = validator.validate(manifest, schema, {
            // does exist but is not in the TypeScript definitions
            nestedErrors: true,
            allowUnknownAttributes: false,
        });
        if (!result.valid) {
            throw new Error(`Invalid assembly manifest:\n${result}`);
        }
    }
    static saveManifest(manifest, filePath, schema, preprocess) {
        let withVersion = { ...manifest, version: Manifest.version() };
        Manifest.validate(withVersion, schema);
        if (preprocess) {
            withVersion = preprocess(withVersion);
        }
        fs.writeFileSync(filePath, JSON.stringify(withVersion, undefined, 2));
    }
    static loadManifest(filePath, schema, preprocess) {
        let obj = JSON.parse(fs.readFileSync(filePath, { encoding: 'utf-8' }));
        if (preprocess) {
            obj = preprocess(obj);
        }
        Manifest.validate(obj, schema);
        return obj;
    }
    /**
     * This requires some explaining...
     *
     * We previously used `{ Key, Value }` for the object that represents a stack tag. (Notice the casing)
     * @link https://github.com/aws/aws-cdk/blob/v1.27.0/packages/aws-cdk/lib/api/cxapp/stacks.ts#L427.
     *
     * When that object moved to this package, it had to be JSII compliant, which meant the property
     * names must be `camelCased`, and not `PascalCased`. This meant it no longer matches the structure in the `manifest.json` file.
     * In order to support current manifest files, we have to translate the `PascalCased` representation to the new `camelCased` one.
     *
     * Note that the serialization itself still writes `PascalCased` because it relates to how CloudFormation expects it.
     *
     * Ideally, we would start writing the `camelCased` and translate to how CloudFormation expects it when needed. But this requires nasty
     * backwards-compatibility code and it just doesn't seem to be worth the effort.
     */
    static patchStackTagsOnRead(manifest) {
        return Manifest.replaceStackTags(manifest, tags => tags.map((diskTag) => ({
            key: diskTag.Key,
            value: diskTag.Value,
        })));
    }
    /**
     * See explanation on `patchStackTagsOnRead`
     *
     * Translate stack tags metadata if it has the "right" casing.
     */
    static patchStackTagsOnWrite(manifest) {
        return Manifest.replaceStackTags(manifest, tags => tags.map(memTag => 
        // Might already be uppercased (because stack synthesis generates it in final form yet)
        ('Key' in memTag ? memTag : { Key: memTag.key, Value: memTag.value })));
    }
    /**
     * Recursively replace stack tags in the stack metadata
     */
    static replaceStackTags(manifest, fn) {
        // Need to add in the `noUndefined`s because otherwise jest snapshot tests are going to freak out
        // about the keys with values that are `undefined` (even though they would never be JSON.stringified)
        return noUndefined({
            ...manifest,
            artifacts: mapValues(manifest.artifacts, artifact => {
                if (artifact.type !== assembly.ArtifactType.AWS_CLOUDFORMATION_STACK) {
                    return artifact;
                }
                return noUndefined({
                    ...artifact,
                    metadata: mapValues(artifact.metadata, metadataEntries => metadataEntries.map(metadataEntry => {
                        if (metadataEntry.type !== assembly.ArtifactMetadataEntryType.STACK_TAGS || !metadataEntry.data) {
                            return metadataEntry;
                        }
                        return {
                            ...metadataEntry,
                            data: fn(metadataEntry.data),
                        };
                    })),
                });
            }),
        });
    }
}
exports.Manifest = Manifest;
_a = JSII_RTTI_SYMBOL_1;
Manifest[_a] = { fqn: "@aws-cdk/cloud-assembly-schema.Manifest", version: "1.142.0" };
function mapValues(xs, fn) {
    if (!xs) {
        return undefined;
    }
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        ret[k] = fn(v);
    }
    return ret;
}
function noUndefined(xs) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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