"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FactName = exports.Fact = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_entities_1 = require("./aws-entities");
/**
 * A database of regional information.
 *
 * @stability stable
 */
class Fact {
    constructor() {
        throw new Error('Use the static methods of Fact instead!');
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     * may not be an exhaustive list of all available AWS regions.
     * @stability stable
     */
    static get regions() {
        return aws_entities_1.AWS_REGIONS;
    }
    /**
     * Retrieves a fact from this Fact database.
     *
     * @param region the name of the region (e.g: `us-east-1`).
     * @param name the name of the fact being looked up (see the `FactName` class for details).
     * @returns the fact value if it is known, and `undefined` otherwise.
     * @stability stable
     */
    static find(region, name) {
        const regionFacts = this.database[region];
        return regionFacts && regionFacts[name];
    }
    /**
     * Retrieve a fact from the Fact database.
     *
     * (retrieval will fail if the specified region or
     * fact name does not exist.)
     *
     * @param region the name of the region (e.g: `us-east-1`).
     * @param name the name of the fact being looked up (see the `FactName` class for details).
     * @stability stable
     */
    static requireFact(region, name) {
        const foundFact = this.find(region, name);
        if (!foundFact) {
            throw new Error(`No fact ${name} could be found for region: ${region} and name: ${name}`);
        }
        return foundFact;
    }
    /**
     * Registers a new fact in this Fact database.
     *
     * @param fact the new fact to be registered.
     * @param allowReplacing whether new facts can replace existing facts or not.
     * @stability stable
     */
    static register(fact, allowReplacing = false) {
        jsiiDeprecationWarnings._aws_cdk_region_info_IFact(fact);
        const regionFacts = this.database[fact.region] || (this.database[fact.region] = {});
        if (fact.name in regionFacts && regionFacts[fact.name] !== fact.value && !allowReplacing) {
            throw new Error(`Region ${fact.region} already has a fact ${fact.name}, with value ${regionFacts[fact.name]}`);
        }
        if (fact.value !== undefined) {
            regionFacts[fact.name] = fact.value;
        }
    }
    /**
     * Removes a fact from the database.
     *
     * @param region the region for which the fact is to be removed.
     * @param name the name of the fact to remove.
     * @param value the value that should be removed (removal will fail if the value is specified, but does not match the current stored value).
     * @stability stable
     */
    static unregister(region, name, value) {
        const regionFacts = this.database[region] || {};
        if (name in regionFacts && value && regionFacts[name] !== value) {
            throw new Error(`Attempted to remove ${name} from ${region} with value ${value}, but the fact's value is ${regionFacts[name]}`);
        }
        delete regionFacts[name];
    }
}
exports.Fact = Fact;
_a = JSII_RTTI_SYMBOL_1;
Fact[_a] = { fqn: "@aws-cdk/region-info.Fact", version: "1.140.0" };
Fact.database = {};
/**
 * All standardized fact names.
 *
 * @stability stable
 */
class FactName {
    /**
     * The ARN of CloudWatch Lambda Insights for a version (e.g. 1.0.98.0).
     *
     * @stability stable
     */
    static cloudwatchLambdaInsightsVersion(version, arch) {
        // if we are provided an architecture use that, otherwise
        // default to x86_64 for backwards compatibility
        const suffix = version.split('.').join('_') + `_${arch !== null && arch !== void 0 ? arch : 'x86_64'}`;
        return `cloudwatch-lambda-insights-version:${suffix}`;
    }
    /**
     * The name of the regional service principal for a given service.
     *
     * @param service the service name, either simple (e.g: `s3`, `codedeploy`) or qualified (e.g: `s3.amazonaws.com`). The `.amazonaws.com` and `.amazonaws.com.cn` domains are stripped from service names, so they are canonicalized in that respect.
     * @stability stable
     */
    static servicePrincipal(service) {
        return `service-principal:${service.replace(/\.amazonaws\.com(\.cn)?$/, '')}`;
    }
}
exports.FactName = FactName;
_b = JSII_RTTI_SYMBOL_1;
FactName[_b] = { fqn: "@aws-cdk/region-info.FactName", version: "1.140.0" };
/**
 * The name of the partition for a region (e.g: 'aws', 'aws-cn', ...).
 *
 * @stability stable
 */
FactName.PARTITION = 'partition';
/**
 * The domain suffix for a region (e.g: 'amazonaws.com`).
 *
 * @stability stable
 */
FactName.DOMAIN_SUFFIX = 'domainSuffix';
/**
 * Whether the AWS::CDK::Metadata CloudFormation Resource is available in-region or not.
 *
 * The value is a boolean
 * modelled as `YES` or `NO`.
 *
 * @stability stable
 */
FactName.CDK_METADATA_RESOURCE_AVAILABLE = 'cdk:metadata-resource:available';
/**
 * The endpoint used for hosting S3 static websites.
 *
 * @stability stable
 */
FactName.S3_STATIC_WEBSITE_ENDPOINT = 's3-static-website:endpoint';
/**
 * The endpoint used for aliasing S3 static websites in Route 53.
 *
 * @stability stable
 */
FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID = 's3-static-website:route-53-hosted-zone-id';
/**
 * The hosted zone ID used by Route 53 to alias a EBS environment endpoint in this region (e.g: Z2O1EMRO9K5GLX).
 *
 * @stability stable
 */
FactName.EBS_ENV_ENDPOINT_HOSTED_ZONE_ID = 'ebs-environment:route-53-hosted-zone-id';
/**
 * The prefix for VPC Endpoint Service names, cn.com.amazonaws.vpce for China regions, com.amazonaws.vpce otherwise.
 *
 * @stability stable
 */
FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'vpcEndpointServiceNamePrefix';
/**
 * The account for ELBv2 in this region.
 *
 * @stability stable
 */
FactName.ELBV2_ACCOUNT = 'elbv2Account';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the AWS Deep Learning Containers images in a given region.
 *
 * @stability stable
 */
FactName.DLC_REPOSITORY_ACCOUNT = 'dlcRepositoryAccount';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the AWS App Mesh Envoy Proxy images in a given region.
 *
 * @stability stable
 */
FactName.APPMESH_ECR_ACCOUNT = 'appMeshRepositoryAccount';
/**
 * The CIDR block used by Kinesis Data Firehose servers.
 *
 * @stability stable
 */
FactName.FIREHOSE_CIDR_BLOCK = 'firehoseCidrBlock';
//# sourceMappingURL=data:application/json;base64,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