"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderCondition = exports.Effect = exports.Targets = exports.parseLambdaPermission = exports.parseStatements = exports.Statement = void 0;
const deepEqual = require("fast-deep-equal");
const util_1 = require("../util");
class Statement {
    constructor(statement) {
        if (typeof statement === 'string') {
            this.sid = undefined;
            this.effect = Effect.Unknown;
            this.resources = new Targets({}, '', '');
            this.actions = new Targets({}, '', '');
            this.principals = new Targets({}, '', '');
            this.condition = undefined;
            this.serializedIntrinsic = statement;
        }
        else {
            this.sid = expectString(statement.Sid);
            this.effect = expectEffect(statement.Effect);
            this.resources = new Targets(statement, 'Resource', 'NotResource');
            this.actions = new Targets(statement, 'Action', 'NotAction');
            this.principals = new Targets(statement, 'Principal', 'NotPrincipal');
            this.condition = statement.Condition;
            this.serializedIntrinsic = undefined;
        }
    }
    /**
     * Whether this statement is equal to the other statement
     */
    equal(other) {
        return (this.sid === other.sid
            && this.effect === other.effect
            && this.serializedIntrinsic === other.serializedIntrinsic
            && this.resources.equal(other.resources)
            && this.actions.equal(other.actions)
            && this.principals.equal(other.principals)
            && deepEqual(this.condition, other.condition));
    }
    render() {
        return this.serializedIntrinsic
            ? {
                resource: this.serializedIntrinsic,
                effect: '',
                action: '',
                principal: this.principals.render(),
                condition: '',
            }
            : {
                resource: this.resources.render(),
                effect: this.effect,
                action: this.actions.render(),
                principal: this.principals.render(),
                condition: renderCondition(this.condition),
            };
    }
    /**
     * Return a machine-readable version of the changes.
     * This is only used in tests.
     *
     * @internal
     */
    _toJson() {
        return this.serializedIntrinsic
            ? this.serializedIntrinsic
            : util_1.deepRemoveUndefined({
                sid: this.sid,
                effect: this.effect,
                resources: this.resources._toJson(),
                principals: this.principals._toJson(),
                actions: this.actions._toJson(),
                condition: this.condition,
            });
    }
    /**
     * Whether this is a negative statement
     *
     * A statement is negative if any of its targets are negative, inverted
     * if the Effect is Deny.
     */
    get isNegativeStatement() {
        const notTarget = this.actions.not || this.principals.not || this.resources.not;
        return this.effect === Effect.Allow ? notTarget : !notTarget;
    }
}
exports.Statement = Statement;
/**
 * Parse a list of statements from undefined, a Statement, or a list of statements
 */
function parseStatements(x) {
    if (x === undefined) {
        x = [];
    }
    if (!Array.isArray(x)) {
        x = [x];
    }
    return x.map((s) => new Statement(s));
}
exports.parseStatements = parseStatements;
/**
 * Parse a Statement from a Lambda::Permission object
 *
 * This is actually what Lambda adds to the policy document if you call AddPermission.
 */
function parseLambdaPermission(x) {
    // Construct a statement from
    const statement = {
        Effect: 'Allow',
        Action: x.Action,
        Resource: x.FunctionName,
    };
    if (x.Principal !== undefined) {
        if (x.Principal === '*') {
            // *
            statement.Principal = '*';
        }
        else if (/^\d{12}$/.test(x.Principal)) {
            // Account number
            statement.Principal = { AWS: `arn:aws:iam::${x.Principal}:root` };
        }
        else {
            // Assume it's a service principal
            // We might get this wrong vs. the previous one for tokens. Nothing to be done
            // about that. It's only for human readable consumption after all.
            statement.Principal = { Service: x.Principal };
        }
    }
    if (x.SourceArn !== undefined) {
        if (statement.Condition === undefined) {
            statement.Condition = {};
        }
        statement.Condition.ArnLike = { 'AWS:SourceArn': x.SourceArn };
    }
    if (x.SourceAccount !== undefined) {
        if (statement.Condition === undefined) {
            statement.Condition = {};
        }
        statement.Condition.StringEquals = { 'AWS:SourceAccount': x.SourceAccount };
    }
    if (x.EventSourceToken !== undefined) {
        if (statement.Condition === undefined) {
            statement.Condition = {};
        }
        statement.Condition.StringEquals = { 'lambda:EventSourceToken': x.EventSourceToken };
    }
    return new Statement(statement);
}
exports.parseLambdaPermission = parseLambdaPermission;
/**
 * Targets for a field
 */
class Targets {
    constructor(statement, positiveKey, negativeKey) {
        if (negativeKey in statement) {
            this.values = forceListOfStrings(statement[negativeKey]);
            this.not = true;
        }
        else {
            this.values = forceListOfStrings(statement[positiveKey]);
            this.not = false;
        }
        this.values.sort();
    }
    get empty() {
        return this.values.length === 0;
    }
    /**
     * Whether this set of targets is equal to the other set of targets
     */
    equal(other) {
        return this.not === other.not && deepEqual(this.values.sort(), other.values.sort());
    }
    /**
     * If the current value set is empty, put this in it
     */
    replaceEmpty(replacement) {
        if (this.empty) {
            this.values.push(replacement);
        }
    }
    /**
     * If the actions contains a '*', replace with this string.
     */
    replaceStar(replacement) {
        for (let i = 0; i < this.values.length; i++) {
            if (this.values[i] === '*') {
                this.values[i] = replacement;
            }
        }
        this.values.sort();
    }
    /**
     * Render into a summary table cell
     */
    render() {
        return this.not
            ? this.values.map(s => `NOT ${s}`).join('\n')
            : this.values.join('\n');
    }
    /**
     * Return a machine-readable version of the changes.
     * This is only used in tests.
     *
     * @internal
     */
    _toJson() {
        return { not: this.not, values: this.values };
    }
}
exports.Targets = Targets;
var Effect;
(function (Effect) {
    Effect["Unknown"] = "Unknown";
    Effect["Allow"] = "Allow";
    Effect["Deny"] = "Deny";
})(Effect = exports.Effect || (exports.Effect = {}));
function expectString(x) {
    return typeof x === 'string' ? x : undefined;
}
function expectEffect(x) {
    if (x === Effect.Allow || x === Effect.Deny) {
        return x;
    }
    return Effect.Unknown;
}
function forceListOfStrings(x) {
    if (typeof x === 'string') {
        return [x];
    }
    if (typeof x === 'undefined' || x === null) {
        return [];
    }
    if (Array.isArray(x)) {
        return x.map(e => forceListOfStrings(e).join(','));
    }
    if (typeof x === 'object' && x !== null) {
        const ret = [];
        for (const [key, value] of Object.entries(x)) {
            ret.push(...forceListOfStrings(value).map(s => `${key}:${s}`));
        }
        return ret;
    }
    return [`${x}`];
}
/**
 * Render the Condition column
 */
function renderCondition(condition) {
    if (!condition || Object.keys(condition).length === 0) {
        return '';
    }
    const jsonRepresentation = JSON.stringify(condition, undefined, 2);
    // The JSON representation looks like this:
    //
    //  {
    //    "ArnLike": {
    //      "AWS:SourceArn": "${MyTopic86869434}"
    //    }
    //  }
    //
    // We can make it more compact without losing information by getting rid of the outermost braces
    // and the indentation.
    const lines = jsonRepresentation.split('\n');
    return lines.slice(1, lines.length - 1).map(s => s.substr(2)).join('\n');
}
exports.renderCondition = renderCondition;
//# sourceMappingURL=data:application/json;base64,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