"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LegacyStackSynthesizer = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const cfn_fn_1 = require("../cfn-fn");
const construct_compat_1 = require("../construct-compat");
const asset_parameters_1 = require("../private/asset-parameters");
const _shared_1 = require("./_shared");
const stack_synthesizer_1 = require("./stack-synthesizer");
/**
 * The well-known name for the docker image asset ECR repository. All docker
 * image assets will be pushed into this repository with an image tag based on
 * the source hash.
 */
const ASSETS_ECR_REPOSITORY_NAME = 'aws-cdk/assets';
/**
 * This allows users to work around the fact that the ECR repository is
 * (currently) not configurable by setting this context key to their desired
 * repository name. The CLI will auto-create this ECR repository if it's not
 * already created.
 */
const ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY = 'assets-ecr-repository-name';
/**
 * Use the original deployment environment.
 *
 * This deployment environment is restricted in cross-environment deployments,
 * CI/CD deployments, and will use up CloudFormation parameters in your template.
 *
 * This is the only StackSynthesizer that supports customizing asset behavior
 * by overriding `Stack.addFileAsset()` and `Stack.addDockerImageAsset()`.
 *
 * @stability stable
 */
class LegacyStackSynthesizer extends stack_synthesizer_1.StackSynthesizer {
    constructor() {
        super(...arguments);
        this.cycle = false;
        /**
         * The image ID of all the docker image assets that were already added to this
         * stack (to avoid duplication).
         */
        this.addedImageAssets = new Set();
    }
    /**
     * Bind to the stack this environment is going to be used on.
     *
     * Must be called before any of the other methods are called.
     *
     * @stability stable
     */
    bind(stack) {
        jsiiDeprecationWarnings._aws_cdk_core_Stack(stack);
        if (this.stack !== undefined) {
            throw new Error('A StackSynthesizer can only be used for one Stack: create a new instance to use with a different Stack');
        }
        this.stack = stack;
    }
    /**
     * Register a File Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addFileAsset(asset) {
        jsiiDeprecationWarnings._aws_cdk_core_FileAssetSource(asset);
        _shared_1.assertBound(this.stack);
        // Backwards compatibility hack. We have a number of conflicting goals here:
        //
        // - We want put the actual logic in this class
        // - We ALSO want to keep supporting people overriding Stack.addFileAsset (for backwards compatibility,
        // because that mechanism is currently used to make CI/CD scenarios work)
        // - We ALSO want to allow both entry points from user code (our own framework
        // code will always call stack.deploymentMechanism.addFileAsset() but existing users
        // may still be calling `stack.addFileAsset()` directly.
        //
        // Solution: delegate call to the stack, but if the stack delegates back to us again
        // then do the actual logic.
        //
        // @deprecated: this can be removed for v2
        if (this.cycle) {
            return this.doAddFileAsset(asset);
        }
        this.cycle = true;
        try {
            const stack = this.stack;
            return withoutDeprecationWarnings(() => stack.addFileAsset(asset));
        }
        finally {
            this.cycle = false;
        }
    }
    /**
     * Register a Docker Image Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addDockerImageAsset(asset) {
        jsiiDeprecationWarnings._aws_cdk_core_DockerImageAssetSource(asset);
        _shared_1.assertBound(this.stack);
        // See `addFileAsset` for explanation.
        // @deprecated: this can be removed for v2
        if (this.cycle) {
            return this.doAddDockerImageAsset(asset);
        }
        this.cycle = true;
        try {
            const stack = this.stack;
            return withoutDeprecationWarnings(() => stack.addDockerImageAsset(asset));
        }
        finally {
            this.cycle = false;
        }
    }
    /**
     * Synthesize the associated stack to the session.
     *
     * @stability stable
     */
    synthesize(session) {
        jsiiDeprecationWarnings._aws_cdk_core_ISynthesisSession(session);
        _shared_1.assertBound(this.stack);
        this.synthesizeStackTemplate(this.stack, session);
        // Just do the default stuff, nothing special
        this.emitStackArtifact(this.stack, session);
    }
    doAddDockerImageAsset(asset) {
        var _b, _c;
        _shared_1.assertBound(this.stack);
        // check if we have an override from context
        const repositoryNameOverride = this.stack.node.tryGetContext(ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY);
        const repositoryName = (_c = (_b = asset.repositoryName) !== null && _b !== void 0 ? _b : repositoryNameOverride) !== null && _c !== void 0 ? _c : ASSETS_ECR_REPOSITORY_NAME;
        const imageTag = asset.sourceHash;
        const assetId = asset.sourceHash;
        // only add every image (identified by source hash) once for each stack that uses it.
        if (!this.addedImageAssets.has(assetId)) {
            if (!asset.directoryName) {
                throw new Error(`LegacyStackSynthesizer does not support this type of file asset: ${JSON.stringify(asset)}`);
            }
            const metadata = {
                repositoryName,
                imageTag,
                id: assetId,
                packaging: 'container-image',
                path: asset.directoryName,
                sourceHash: asset.sourceHash,
                buildArgs: asset.dockerBuildArgs,
                target: asset.dockerBuildTarget,
                file: asset.dockerFile,
            };
            this.stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
            this.addedImageAssets.add(assetId);
        }
        return {
            imageUri: `${this.stack.account}.dkr.ecr.${this.stack.region}.${this.stack.urlSuffix}/${repositoryName}:${imageTag}`,
            repositoryName,
        };
    }
    doAddFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        let params = this.assetParameters.node.tryFindChild(asset.sourceHash);
        if (!params) {
            params = new asset_parameters_1.FileAssetParameters(this.assetParameters, asset.sourceHash);
            if (!asset.fileName || !asset.packaging) {
                throw new Error(`LegacyStackSynthesizer does not support this type of file asset: ${JSON.stringify(asset)}`);
            }
            const metadata = {
                path: asset.fileName,
                id: asset.sourceHash,
                packaging: asset.packaging,
                sourceHash: asset.sourceHash,
                s3BucketParameter: params.bucketNameParameter.logicalId,
                s3KeyParameter: params.objectKeyParameter.logicalId,
                artifactHashParameter: params.artifactHashParameter.logicalId,
            };
            this.stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
        }
        const bucketName = params.bucketNameParameter.valueAsString;
        // key is prefix|postfix
        const encodedKey = params.objectKeyParameter.valueAsString;
        const s3Prefix = cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const s3Filename = cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const objectKey = `${s3Prefix}${s3Filename}`;
        const httpUrl = `https://s3.${this.stack.region}.${this.stack.urlSuffix}/${bucketName}/${objectKey}`;
        const s3ObjectUrl = `s3://${bucketName}/${objectKey}`;
        return { bucketName, objectKey, httpUrl, s3ObjectUrl, s3Url: httpUrl };
    }
    get assetParameters() {
        _shared_1.assertBound(this.stack);
        if (!this._assetParameters) {
            this._assetParameters = new construct_compat_1.Construct(this.stack, 'AssetParameters');
        }
        return this._assetParameters;
    }
}
exports.LegacyStackSynthesizer = LegacyStackSynthesizer;
_a = JSII_RTTI_SYMBOL_1;
LegacyStackSynthesizer[_a] = { fqn: "@aws-cdk/core.LegacyStackSynthesizer", version: "1.140.0" };
function withoutDeprecationWarnings(block) {
    const orig = process.env.JSII_DEPRECATED;
    process.env.JSII_DEPRECATED = 'quiet';
    try {
        return block();
    }
    finally {
        process.env.JSII_DEPRECATED = orig;
    }
}
//# sourceMappingURL=data:application/json;base64,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