"use strict";
// --------------------------------------------------------------------------------
// This file defines context keys that enable certain features that are
// implemented behind a flag in order to preserve backwards compatibility for
// existing apps. When a new app is initialized through `cdk init`, the CLI will
// automatically add enable these features by adding them to the generated
// `cdk.json` file. In the next major release of the CDK, these feature flags
// will be removed and will become the default behavior.
// See https://github.com/aws/aws-cdk-rfcs/blob/master/text/0055-feature-flags.md
// --------------------------------------------------------------------------------
Object.defineProperty(exports, "__esModule", { value: true });
exports.futureFlagDefault = exports.FUTURE_FLAGS_EXPIRED = exports.NEW_PROJECT_DEFAULT_CONTEXT = exports.FUTURE_FLAGS = exports.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME = exports.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER = exports.TARGET_PARTITIONS = exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021 = exports.LAMBDA_RECOGNIZE_VERSION_PROPS = exports.EFS_DEFAULT_ENCRYPTION_AT_REST = exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID = exports.RDS_LOWERCASE_DB_IDENTIFIER = exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT = exports.S3_GRANT_WRITE_WITHOUT_ACL = exports.KMS_DEFAULT_KEY_POLICIES = exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = exports.DOCKER_IGNORE_SUPPORT = exports.STACK_RELATIVE_EXPORTS_CONTEXT = exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT = exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = void 0;
/**
 * If this is set, multiple stacks can use the same stack name (e.g. deployed to
 * different environments). This means that the name of the synthesized template
 * file will be based on the construct path and not on the defined `stackName`
 * of the stack.
 *
 * This is a "future flag": the feature is disabled by default for backwards
 * compatibility, but new projects created using `cdk init` will have this
 * enabled through the generated `cdk.json`.
 */
exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = '@aws-cdk/core:enableStackNameDuplicates';
/**
 * IF this is set, `cdk diff` will always exit with 0.
 *
 * Use `cdk diff --fail` to exit with 1 if there's a diff.
 */
exports.ENABLE_DIFF_NO_FAIL_CONTEXT = 'aws-cdk:enableDiffNoFail';
/** @deprecated use `ENABLE_DIFF_NO_FAIL_CONTEXT` */
exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT;
/**
 * Switch to new stack synthesis method which enable CI/CD
 */
exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = '@aws-cdk/core:newStyleStackSynthesis';
/**
 * Name exports based on the construct paths relative to the stack, rather than the global construct path
 *
 * Combined with the stack name this relative construct path is good enough to
 * ensure uniqueness, and makes the export names robust against refactoring
 * the location of the stack in the construct tree (specifically, moving the Stack
 * into a Stage).
 */
exports.STACK_RELATIVE_EXPORTS_CONTEXT = '@aws-cdk/core:stackRelativeExports';
/**
 * DockerImageAsset properly supports `.dockerignore` files by default
 *
 * If this flag is not set, the default behavior for `DockerImageAsset` is to use
 * glob semantics for `.dockerignore` files. If this flag is set, the default behavior
 * is standard Docker ignore semantics.
 *
 * This is a feature flag as the old behavior was technically incorrect but
 * users may have come to depend on it.
 */
exports.DOCKER_IGNORE_SUPPORT = '@aws-cdk/aws-ecr-assets:dockerIgnoreSupport';
/**
 * Secret.secretName for an "owned" secret will attempt to parse the secretName from the ARN,
 * rather than the default full resource name, which includes the SecretsManager suffix.
 *
 * If this flag is not set, Secret.secretName will include the SecretsManager suffix, which cannot be directly
 * used by SecretsManager.DescribeSecret, and must be parsed by the user first (e.g., Fn:Join, Fn:Select, Fn:Split).
 */
exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = '@aws-cdk/aws-secretsmanager:parseOwnedSecretName';
/**
 * KMS Keys start with a default key policy that grants the account access to administer the key,
 * mirroring the behavior of the KMS SDK/CLI/Console experience. Users may override the default key
 * policy by specifying their own.
 *
 * If this flag is not set, the default key policy depends on the setting of the `trustAccountIdentities`
 * flag. If false (the default, for backwards-compatibility reasons), the default key policy somewhat
 * resemebles the default admin key policy, but with the addition of 'GenerateDataKey' permissions. If
 * true, the policy matches what happens when this feature flag is set.
 *
 * Additionally, if this flag is not set and the user supplies a custom key policy, this will be appended
 * to the key's default policy (rather than replacing it).
 */
exports.KMS_DEFAULT_KEY_POLICIES = '@aws-cdk/aws-kms:defaultKeyPolicies';
/**
 * Change the old 's3:PutObject*' permission to 's3:PutObject' on Bucket,
 * as the former includes 's3:PutObjectAcl',
 * which could be used to grant read/write object access to IAM principals in other accounts.
 * Use a feature flag to make sure existing customers who might be relying
 * on the overly-broad permissions are not broken.
 */
exports.S3_GRANT_WRITE_WITHOUT_ACL = '@aws-cdk/aws-s3:grantWriteWithoutAcl';
/**
 * ApplicationLoadBalancedServiceBase, ApplicationMultipleTargetGroupServiceBase,
 * NetworkLoadBalancedServiceBase, NetworkMultipleTargetGroupServiceBase, and
 * QueueProcessingServiceBase currently determine a default value for the desired count of
 * a CfnService if a desiredCount is not provided.
 *
 * If this flag is not set, the default behaviour for CfnService.desiredCount is to set a
 * desiredCount of 1, if one is not provided. If true, a default will not be defined for
 * CfnService.desiredCount and as such desiredCount will be undefined, if one is not provided.
 *
 * This is a feature flag as the old behavior was technically incorrect, but
 * users may have come to depend on it.
 */
exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT = '@aws-cdk/aws-ecs-patterns:removeDefaultDesiredCount';
/**
 * ServerlessCluster.clusterIdentifier currently can has uppercase letters,
 * and ServerlessCluster pass it through to CfnDBCluster.dbClusterIdentifier.
 * The identifier is saved as lowercase string in AWS and is resolved as original string in CloudFormation.
 *
 * If this flag is not set, original value that one set to ServerlessCluster.clusterIdentifier
 * is passed to CfnDBCluster.dbClusterIdentifier.
 * If this flag is true, ServerlessCluster.clusterIdentifier is converted into a string containing
 * only lowercase characters by the `toLowerCase` function and passed to CfnDBCluster.dbClusterIdentifier.
 *
 * This feature flag make correct the ServerlessCluster.clusterArn when
 * clusterIdentifier contains a Upper case letters.
 */
exports.RDS_LOWERCASE_DB_IDENTIFIER = '@aws-cdk/aws-rds:lowercaseDbIdentifier';
/**
 * The UsagePlanKey resource connects an ApiKey with a UsagePlan. API Gateway does not allow more than one UsagePlanKey
 * for any given UsagePlan and ApiKey combination. For this reason, CloudFormation cannot replace this resource without
 * either the UsagePlan or ApiKey changing.
 *
 * The feature addition to support multiple UsagePlanKey resources - 142bd0e2 - recognized this and attempted to keep
 * existing UsagePlanKey logical ids unchanged.
 * However, this intentionally caused the logical id of the UsagePlanKey to be sensitive to order. That is, when
 * the 'first' UsagePlanKey resource is removed, the logical id of the 'second' assumes what was originally the 'first',
 * which again is disallowed.
 *
 * In effect, there is no way to get out of this mess in a backwards compatible way, while supporting existing stacks.
 * This flag changes the logical id layout of UsagePlanKey to not be sensitive to order.
 */
exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID = '@aws-cdk/aws-apigateway:usagePlanKeyOrderInsensitiveId';
/**
 * Enable this feature flag to have elastic file systems encrypted at rest by default.
 *
 * Encryption can also be configured explicitly using the `encrypted` property.
 */
exports.EFS_DEFAULT_ENCRYPTION_AT_REST = '@aws-cdk/aws-efs:defaultEncryptionAtRest';
/**
 * Enable this feature flag to opt in to the updated logical id calculation for Lambda Version created using the
 * `fn.currentVersion`.
 *
 * The previous calculation incorrectly considered properties of the `AWS::Lambda::Function` resource that did
 * not constitute creating a new Version.
 *
 * See 'currentVersion' section in the aws-lambda module's README for more details.
 */
exports.LAMBDA_RECOGNIZE_VERSION_PROPS = '@aws-cdk/aws-lambda:recognizeVersionProps';
/**
 * Enable this feature flag to have cloudfront distributions use the security policy TLSv1.2_2021 by default.
 *
 * The security policy can also be configured explicitly using the `minimumProtocolVersion` property.
 */
exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021 = '@aws-cdk/aws-cloudfront:defaultSecurityPolicyTLSv1.2_2021';
/**
 * What regions to include in lookup tables of environment agnostic stacks
 *
 * Has no effect on stacks that have a defined region, but will limit the amount
 * of unnecessary regions included in stacks without a known region.
 *
 * The type of this value should be a list of strings.
 */
exports.TARGET_PARTITIONS = '@aws-cdk/core:target-partitions';
/**
 * Enable this feature flag to configure default logging behavior for the ECS Service Extensions. This will enable the
 * `awslogs` log driver for the application container of the service to send the container logs to CloudWatch Logs.
 *
 * This is a feature flag as the new behavior provides a better default experience for the users.
 */
exports.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER = '@aws-cdk-containers/ecs-service-extensions:enableDefaultLogDriver';
/**
 * Enable this feature flag to have Launch Templates generated by the `InstanceRequireImdsv2Aspect` use unique names.
 *
 * Previously, the generated Launch Template names were only unique within a stack because they were based only on the
 * `Instance` construct ID. If another stack that has an `Instance` with the same construct ID is deployed in the same
 * account and region, the deployments would always fail as the generated Launch Template names were the same.
 *
 * The new implementation addresses this issue by generating the Launch Template name with the `Names.uniqueId` method.
 */
exports.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME = '@aws-cdk/aws-ec2:uniqueImdsv2TemplateName';
/**
 * This map includes context keys and values for feature flags that enable
 * capabilities "from the future", which we could not introduce as the default
 * behavior due to backwards compatibility for existing projects.
 *
 * New projects generated through `cdk init` will include these flags in their
 * generated `cdk.json` file.
 *
 * When we release the next major version of the CDK, we will flip the logic of
 * these features and clean up the `cdk.json` generated by `cdk init`.
 *
 * Tests must cover the default (disabled) case and the future (enabled) case.
 */
exports.FUTURE_FLAGS = {
    [exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID]: true,
    [exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: true,
    [exports.ENABLE_DIFF_NO_FAIL_CONTEXT]: true,
    [exports.STACK_RELATIVE_EXPORTS_CONTEXT]: true,
    [exports.DOCKER_IGNORE_SUPPORT]: true,
    [exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME]: true,
    [exports.KMS_DEFAULT_KEY_POLICIES]: true,
    [exports.S3_GRANT_WRITE_WITHOUT_ACL]: true,
    [exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT]: true,
    [exports.RDS_LOWERCASE_DB_IDENTIFIER]: true,
    [exports.EFS_DEFAULT_ENCRYPTION_AT_REST]: true,
    [exports.LAMBDA_RECOGNIZE_VERSION_PROPS]: true,
    [exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021]: true,
    [exports.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER]: true,
    [exports.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME]: true,
};
/**
 * Values that will be set by default in a new project, which are not necessarily booleans (and don't expire)
 */
exports.NEW_PROJECT_DEFAULT_CONTEXT = {
    [exports.TARGET_PARTITIONS]: ['aws', 'aws-cn'],
};
/**
 * The list of future flags that are now expired. This is going to be used to identify
 * and block usages of old feature flags in the new major version of CDK.
 */
exports.FUTURE_FLAGS_EXPIRED = [];
/**
 * The set of defaults that should be applied if the feature flag is not
 * explicitly configured.
 */
const FUTURE_FLAGS_DEFAULTS = {
    [exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID]: false,
    [exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: false,
    [exports.ENABLE_DIFF_NO_FAIL_CONTEXT]: false,
    [exports.STACK_RELATIVE_EXPORTS_CONTEXT]: false,
    [exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT]: false,
    [exports.DOCKER_IGNORE_SUPPORT]: false,
    [exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME]: false,
    [exports.KMS_DEFAULT_KEY_POLICIES]: false,
    [exports.S3_GRANT_WRITE_WITHOUT_ACL]: false,
    [exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT]: false,
    [exports.RDS_LOWERCASE_DB_IDENTIFIER]: false,
    [exports.EFS_DEFAULT_ENCRYPTION_AT_REST]: false,
    [exports.LAMBDA_RECOGNIZE_VERSION_PROPS]: false,
    [exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021]: false,
    [exports.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER]: false,
    [exports.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME]: false,
};
function futureFlagDefault(flag) {
    return FUTURE_FLAGS_DEFAULTS[flag];
}
exports.futureFlagDefault = futureFlagDefault;
//# sourceMappingURL=data:application/json;base64,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