"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const os = require("os");
const path = require("path");
const vm = require("vm");
const fs = require("fs-extra");
const inquirer = require("inquirer");
const yargs = require("yargs");
const common_1 = require("../../common");
const inventory = require("../../inventory");
const logging = require("../../logging");
const util_1 = require("../../util");
const macros_1 = require("../macros");
class Command {
    constructor() {
        this.command = 'new [PROJECT-TYPE] [OPTIONS]';
        this.describe = 'Creates a new projen project';
    }
    builder(args) {
        var _a;
        args.positional('PROJECT-TYPE', { describe: 'optional only when --from is used and there is a single project type in the external module', type: 'string' });
        args.option('synth', { type: 'boolean', default: true, desc: 'Synthesize after creating .projenrc.js' });
        args.option('comments', { type: 'boolean', default: true, desc: 'Include commented out options in .projenrc.js (use --no-comments to disable)' });
        args.option('from', { type: 'string', alias: 'f', desc: 'External jsii npm module to create project from. Supports any package spec supported by yarn (such as "my-pack@^2.0")' });
        args.example('projen new awscdk-app-ts', 'Creates a new project of built-in type "awscdk-app-ts"');
        args.example('projen new --from projen-vue@^2', 'Creates a new project from an external module "projen-vue" with the specified version');
        for (const type of inventory.discover()) {
            args.command(type.pjid, (_a = type.docs) !== null && _a !== void 0 ? _a : '', {
                builder: cargs => {
                    var _a, _b, _c;
                    cargs.showHelpOnFail(true);
                    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
                        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean') {
                            continue; // we don't support non-primitive fields as command line options
                        }
                        let desc = [(_c = (_b = option.docs) === null || _b === void 0 ? void 0 : _b.replace(/\ *\.$/, '')) !== null && _c !== void 0 ? _c : ''];
                        const required = !option.optional;
                        let defaultValue;
                        if (option.default && option.default !== 'undefined') {
                            if (!required) {
                                // if the field is not required, just describe the default but don't actually assign a value
                                desc.push(`[default: ${option.default.replace(/^\ *-/, '').replace(/\.$/, '').trim()}]`);
                            }
                            else {
                                // if the field is required and we have a @default, then assign
                                // the value here so it appears in `--help`
                                defaultValue = renderDefault(option.default);
                            }
                        }
                        cargs.option(option.switch, {
                            group: required ? 'Required:' : 'Optional:',
                            type: option.type,
                            description: desc.join(' '),
                            default: defaultValue,
                            required,
                        });
                    }
                    return cargs;
                },
                handler: argv => newProject(process.cwd(), type, argv),
            });
        }
        return args;
    }
    async handler(args) {
        // handle --from which means we want to first install a jsii module and then
        // create a project defined within this module.
        if (args.from) {
            return newProjectFromModule(process.cwd(), args.from, args);
        }
        // project type is defined but was not matched by yargs, so print the list of supported types
        if (args.projectType) {
            console.log(`Invalid project type ${args.projectType}. Supported types:`);
            for (const pjid of inventory.discover().map(x => x.pjid)) {
                console.log(`  ${pjid}`);
            }
            return;
        }
        // Handles the use case that nothing was specified since PROJECT-TYPE is now an optional positional parameter
        yargs.showHelp();
    }
}
/**
 * Creates a new project with defaults.
 *
 * This function creates the project type in-process (with in VM) and calls
 * `.synth()` on it (if `options.synth` is not `false`).
 *
 * At the moment, it also generates a `.projenrc.js` file with the same code
 * that was just executed. In the future, this will also be done by the project
 * type, so we can easily support multiple languages of projenrc.
 */
function createProject(opts) {
    // Default project resolution location
    let mod = '../../index';
    // External projects need to load the module from the modules directory
    if (opts.type.moduleName !== 'projen') {
        try {
            mod = path.dirname(require.resolve(path.join(opts.type.moduleName, 'package.json'), { paths: [process.cwd()] }));
        }
        catch (err) {
            throw new Error(`External project module '${opts.type.moduleName}' could not be resolved.`);
        }
    }
    // pass the FQN of the project type to the project initializer so it can
    // generate the projenrc file.
    opts.params.jsiiFqn = JSON.stringify(opts.type.fqn);
    const newProjectCode = `const project = new ${opts.type.typename}(${renderParams(opts)});`;
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const module = require(mod);
    const ctx = vm.createContext(module);
    process.env.PROJEN_DISABLE_POST = (!opts.post).toString();
    vm.runInContext([
        newProjectCode,
        opts.synth ? 'project.synth();' : '',
    ].join('\n'), ctx);
    const configPath = path.join(opts.dir, common_1.PROJEN_RC);
    if (fs.existsSync(configPath)) {
        logging.error(`Directory ${opts.dir} already contains ${common_1.PROJEN_RC}`);
        process.exit(1);
    }
    const [importName] = opts.type.typename.split('.');
    const lines = [
        `const { ${importName} } = require('${opts.type.moduleName}');`,
        '',
        newProjectCode,
        '',
        'project.synth();',
        '',
    ];
    fs.writeFileSync(configPath, lines.join('\n'));
    logging.info(`Created ${common_1.PROJEN_RC} for ${opts.type.typename}`);
}
function makePadding(paddingLength) {
    return ' '.repeat(paddingLength);
}
/**
 * Prints all parameters that can be used in a project type, alongside their descriptions.
 *
 * Parameters in `params` that aren't undefined are rendered as defaults,
 * while all other parameters are rendered as commented out.
 *
 * @param type Project type
 * @param params Object with parameter default values
 * @param comments Whether to include optional parameters in commented out form
 */
function renderParams(opts) {
    var _a, _b, _c;
    // preprocessing
    const renders = {};
    const optionsWithDefaults = [];
    const optionsByModule = {}; // only options without defaults
    for (const option of opts.type.options) {
        if (option.deprecated) {
            continue;
        }
        const optionName = option.name;
        let paramRender;
        if (opts.params[optionName] !== undefined) {
            paramRender = `${optionName}: ${opts.params[optionName]},`;
            optionsWithDefaults.push(optionName);
        }
        else {
            const defaultValue = ((_a = option.default) === null || _a === void 0 ? void 0 : _a.startsWith('-')) ? undefined : ((_b = option.default) !== null && _b !== void 0 ? _b : undefined);
            paramRender = `// ${optionName}: ${defaultValue === null || defaultValue === void 0 ? void 0 : defaultValue.replace(/"(.+)"/, '\'$1\'')},`; // single quotes
            const parentModule = option.parent;
            optionsByModule[parentModule] = (_c = optionsByModule[parentModule]) !== null && _c !== void 0 ? _c : [];
            optionsByModule[parentModule].push(option);
        }
        renders[optionName] = paramRender;
    }
    // alphabetize
    const marginSize = Math.max(...Object.values(renders).map(str => str.length));
    optionsWithDefaults.sort();
    for (const parentModule in optionsByModule) {
        optionsByModule[parentModule].sort((o1, o2) => o1.name.localeCompare(o2.name));
    }
    // generate rendering
    const tab = makePadding(2);
    const result = [];
    result.push('{');
    // render options with defaults
    for (const optionName of optionsWithDefaults) {
        result.push(`${tab}${renders[optionName]}`);
    }
    if (result.length > 1) {
        result.push('');
    }
    // render options without defaults
    if (opts.comments) {
        for (const [moduleName, options] of Object.entries(optionsByModule).sort()) {
            result.push(`${tab}/* ${moduleName} */`);
            for (const option of options) {
                const paramRender = renders[option.name];
                result.push(`${tab}${paramRender}${makePadding(marginSize - paramRender.length + 2)}/* ${option.docs} */`);
            }
            result.push('');
        }
    }
    if (result[result.length - 1] === '') {
        result.pop();
    }
    result.push('}');
    return result.join('\n');
}
/**
 * Given a value from "@default", processes macros and returns a stringified
 * (quoted) result.
 *
 * @returns a javascript primitive (could be a string, number or boolean)
 */
function renderDefault(value) {
    var _a;
    return (_a = macros_1.tryProcessMacro(value)) !== null && _a !== void 0 ? _a : JSON.parse(value);
}
/**
 * Converts yargs command line switches to project type props.
 * @param type Project type
 * @param argv Command line switches
 */
function commandLineToProps(type, argv) {
    var _a;
    const props = {};
    // initialize props with default values
    for (const prop of type.options) {
        if (prop.default && prop.default !== 'undefined' && !prop.optional) {
            props[prop.name] = renderDefault(prop.default);
        }
    }
    for (const [arg, value] of Object.entries(argv)) {
        for (const prop of type.options) {
            if (prop.switch === arg) {
                let curr = props;
                const queue = [...prop.path];
                while (true) {
                    const p = queue.shift();
                    if (!p) {
                        break;
                    }
                    if (queue.length === 0) {
                        let val = value;
                        // if this is a string, then single quote it
                        if (val && typeof (val) === 'string') {
                            val = JSON.stringify(val).replace(/"(.+)"/, '\'$1\'');
                        }
                        curr[p] = val;
                    }
                    else {
                        curr[p] = (_a = curr[p]) !== null && _a !== void 0 ? _a : {};
                        curr = curr[p];
                    }
                }
            }
        }
    }
    return props;
}
/**
 * Generates a new project from an external module.
 *
 * @param spec The name of the external module to load
 * @param args Command line arguments (incl. project type)
 */
async function newProjectFromModule(baseDir, spec, args) {
    var _a;
    const specDependencyInfo = yarnAdd(baseDir, spec);
    // collect projects by looking up all .jsii modules in `node_modules`.
    const modulesDir = path.join(baseDir, 'node_modules');
    const modules = fs.readdirSync(modulesDir).map(file => path.join(modulesDir, file));
    const projects = inventory
        .discover(...modules)
        .filter(x => x.moduleName !== 'projen'); // filter built-in project types
    if (projects.length < 1) {
        throw new Error(`No projects found after installing ${spec}. The module must export at least one class which extends projen.Project`);
    }
    const requested = args.projectType;
    const types = projects.map(p => p.pjid);
    // if user did not specify a project type but the module has more than one, we need them to tell us which one...
    if (!requested && projects.length > 1) {
        throw new Error(`Multiple projects found after installing ${spec}: ${types.join(',')}. Please specify a project name.\nExample: npx projen new --from ${spec} ${types[0]}`);
    }
    // if user did not specify a type (and we know we have only one), the select it. otherwise, search by pjid.
    const type = !requested ? projects[0] : projects.find(p => p.pjid === requested);
    if (!type) {
        throw new Error(`Project type ${requested} not found. Found ${types.join(',')}`);
    }
    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean') {
            continue; // we don't support non-primitive fields as command line options
        }
        if (option.default && option.default !== 'undefined') {
            if (!option.optional) {
                const defaultValue = renderDefault(option.default);
                args[option.name] = defaultValue;
                args[option.switch] = defaultValue;
            }
        }
    }
    // include a dev dependency for the external module
    await newProject(baseDir, type, args, {
        devDeps: JSON.stringify([specDependencyInfo]),
    });
}
/**
 * Generates a new project.
 * @param type Project type
 * @param args Command line arguments
 * @param additionalProps Additional parameters to include in .projenrc.js
 */
async function newProject(baseDir, type, args, additionalProps) {
    // convert command line arguments to project props using type information
    const props = commandLineToProps(type, args);
    // merge in additional props if specified
    for (const [k, v] of Object.entries(additionalProps !== null && additionalProps !== void 0 ? additionalProps : {})) {
        props[k] = v;
    }
    createProject({
        dir: baseDir,
        type,
        params: props,
        comments: args.comments,
        synth: args.synth,
        post: args.post,
    });
    // interactive git and github setup
    const gitFolder = path.resolve(baseDir, '.git');
    let pushInitialToGithub = false;
    if (!fs.existsSync(gitFolder)) {
        pushInitialToGithub = await askAboutGit(baseDir);
    }
    if (pushInitialToGithub) {
        util_1.exec('git add .', { cwd: baseDir });
        util_1.exec('git commit -m \'Initial commit generated by projen\'', { cwd: baseDir });
        util_1.exec('git branch -M main', { cwd: baseDir });
        util_1.exec('git push --set-upstream origin main', { cwd: baseDir });
    }
}
/**
 * Installs the npm module (through `yarn add`) to node_modules under `projectDir`.
 * @param spec The npm package spec (e.g. foo@^1.2)
 * @returns String info for the project devDeps (e.g. foo@^1.2 or foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz)
 */
function yarnAdd(baseDir, spec) {
    const packageJsonPath = path.join(baseDir, 'package.json');
    const packageJsonExisted = fs.existsSync(packageJsonPath);
    let dependencyInfo = spec;
    // workaround: yarn fails to extract tgz if it contains '@' in the name, so we
    // create a temp copy called pkg.tgz and install from there.
    // see: https://github.com/yarnpkg/yarn/issues/6339
    if (spec.endsWith('.tgz') && spec.includes('@')) {
        // if user passes in a file spec then we have to specify the project name and the package location
        // (e.g foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz)
        const moduleName = spec.split('/').slice(-1)[0].trim().split('@')[0].trim(); // Example: ./cdk-project/dist/js/cdk-project@1.0.0.jsii.tgz
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'projen-'));
        const copy = path.join(tmpdir, 'pkg.tgz');
        fs.copyFileSync(spec, copy);
        spec = copy;
        dependencyInfo = `${moduleName}@${spec}`;
    }
    logging.info(`installing external module ${spec}...`);
    util_1.exec(`yarn add --modules-folder=${baseDir}/node_modules --silent --no-lockfile --dev ${spec}`, { cwd: baseDir });
    // if package.json did not exist before calling yarn add, we should remove it
    // so we can start off clean.
    if (!packageJsonExisted) {
        fs.removeSync(packageJsonPath);
    }
    return dependencyInfo;
}
/**
 * Returns the last path element for use as a repository name default.
 */
function repoName() {
    return path.basename(path.basename(process.cwd()));
}
async function askAboutGit(cwd) {
    logging.info('We notice that you do not have a local git repository.');
    const { setUpGit } = await inquirer.prompt([
        {
            type: 'confirm',
            name: 'setUpGit',
            message: 'Do you want to set that up now?',
        },
    ]);
    if (setUpGit) {
        const { plan } = await inquirer.prompt(githubPlanOptions);
        const { gh, git } = plan;
        if (!git && !gh) {
            util_1.exec('git init', { cwd });
            logging.info('Ok, we have run \'git init\' for you! Have a great day.');
        }
        if (git) {
            const { gitRepoURL } = await inquirer.prompt([
                {
                    type: 'input',
                    name: 'gitRepoURL',
                    message: 'What is the repo? (example: https://github.com/projen/projen)',
                },
            ]);
            util_1.exec('git init', { cwd });
            let formattedGitRepoURL = gitRepoURL;
            if (!gitRepoURL.includes('https')) {
                formattedGitRepoURL = `https://github.com/${gitRepoURL}`;
            }
            util_1.exec(`git remote add origin ${formattedGitRepoURL}`, { cwd });
            logging.info(`Great! We have run 'git init' for you and set the remote to ${formattedGitRepoURL}`);
        }
        if (!git && gh) {
            logging.info('Ok! We will make you a repository on GitHub.');
            const ghCLIPath = util_1.execOrUndefined(`${os.platform() === 'win32' ? 'where' : 'which'} gh`, { cwd });
            if (!ghCLIPath) {
                logging.warn('Looks like you do not have the GitHub CLI installed. Please go to https://cli.github.com/ to install and try again.');
            }
            else {
                const { gitProjectName } = await inquirer.prompt([
                    {
                        type: 'input',
                        name: 'gitProjectName',
                        message: 'What would you like to name it?',
                        default: repoName(),
                    },
                ]);
                logging.info(`Wow! ${gitProjectName} is such a great name!`);
                util_1.exec('git init', { cwd });
                util_1.exec(`gh repo create ${gitProjectName}`, { cwd });
                return true;
            }
        }
    }
    return false;
}
const githubPlanOptions = [
    {
        type: 'list',
        name: 'plan',
        message: 'We\'ll need some more info. Please choose one:',
        choices: [
            {
                value: {
                    git: true,
                },
                name: 'I already have a git repository',
            },
            {
                value: {
                    gh: true,
                    git: false,
                },
                name: 'I don\'t have a git repository and want to make one on GitHub',
            },
            {
                value: {
                    gh: false,
                    git: false,
                },
                name: 'I don\'t have a git repository and I don\'t want to use GitHub',
            },
        ],
    },
];
module.exports = new Command();
//# sourceMappingURL=data:application/json;base64,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