"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RegionInfo = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_entities_1 = require("./aws-entities");
const fact_1 = require("./fact");
/**
 * Information pertaining to an AWS region.
 *
 * @stability stable
 */
class RegionInfo {
    constructor(name) {
        this.name = name;
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     * may not be an exaustive list of all available AWS regions.
     * @stability stable
     */
    static get regions() {
        return fact_1.Fact.regions.map(RegionInfo.get);
    }
    /**
     * Retrieves a collection of all fact values for all regions that fact is defined in.
     *
     * @param factName the name of the fact to retrieve values for.
     * @returns a mapping with AWS region codes as the keys,
     * and the fact in the given region as the value for that key
     * @stability stable
     */
    static regionMap(factName) {
        const ret = {};
        for (const regionInfo of RegionInfo.regions) {
            const fact = fact_1.Fact.find(regionInfo.name, factName);
            if (fact) {
                ret[regionInfo.name] = fact;
            }
        }
        return ret;
    }
    /**
     * Retrieves a collection of all fact values for all regions, limited to some partitions.
     *
     * @param factName the name of the fact to retrieve values for.
     * @param partitions list of partitions to retrieve facts for.
     * @returns a mapping with AWS region codes as the keys,
     * and the fact in the given region as the value for that key
     * @stability stable
     */
    static limitedRegionMap(factName, partitions) {
        const ret = {};
        for (const [region, value] of Object.entries(RegionInfo.regionMap(factName))) {
            if (partitions.includes(aws_entities_1.partitionInformation(region).partition)) {
                ret[region] = value;
            }
        }
        return ret;
    }
    /**
     * Obtain region info for a given region name.
     *
     * @param name the name of the region (e.g: us-east-1).
     * @stability stable
     */
    static get(name) {
        return new RegionInfo(name);
    }
    /**
     * Whether the `AWS::CDK::Metadata` CloudFormation Resource is available in this region or not.
     *
     * @stability stable
     */
    get cdkMetadataResourceAvailable() {
        return fact_1.Fact.find(this.name, fact_1.FactName.CDK_METADATA_RESOURCE_AVAILABLE) === 'YES';
    }
    /**
     * The domain name suffix (e.g: amazonaws.com) for this region.
     *
     * @stability stable
     */
    get domainSuffix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DOMAIN_SUFFIX);
    }
    /**
     * The name of the ARN partition for this region (e.g: aws).
     *
     * @stability stable
     */
    get partition() {
        return fact_1.Fact.find(this.name, fact_1.FactName.PARTITION);
    }
    /**
     * The endpoint used by S3 static website hosting in this region (e.g: s3-static-website-us-east-1.amazonaws.com).
     *
     * @stability stable
     */
    get s3StaticWebsiteEndpoint() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ENDPOINT);
    }
    /**
     * The hosted zone ID used by Route 53 to alias a S3 static website in this region (e.g: Z2O1EMRO9K5GLX).
     *
     * @stability stable
     */
    get s3StaticWebsiteHostedZoneId() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID);
    }
    /**
     * The hosted zone ID used by Route 53 to alias a EBS environment endpoint in this region (e.g: Z2O1EMRO9K5GLX).
     *
     * @stability stable
     */
    get ebsEnvEndpointHostedZoneId() {
        return fact_1.Fact.find(this.name, fact_1.FactName.EBS_ENV_ENDPOINT_HOSTED_ZONE_ID);
    }
    /**
     * The prefix for VPC Endpoint Service names, cn.com.amazonaws.vpce for China regions, com.amazonaws.vpce otherwise.
     *
     * @stability stable
     */
    get vpcEndpointServiceNamePrefix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX);
    }
    /**
     * The name of the service principal for a given service in this region.
     *
     * @param service the service name (e.g: s3.amazonaws.com).
     * @stability stable
     */
    servicePrincipal(service) {
        return fact_1.Fact.find(this.name, fact_1.FactName.servicePrincipal(service));
    }
    /**
     * The account ID for ELBv2 in this region.
     *
     * @stability stable
     */
    get elbv2Account() {
        return fact_1.Fact.find(this.name, fact_1.FactName.ELBV2_ACCOUNT);
    }
    /**
     * The ID of the AWS account that owns the public ECR repository containing the AWS Deep Learning Containers images in this region.
     *
     * @stability stable
     */
    get dlcRepositoryAccount() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DLC_REPOSITORY_ACCOUNT);
    }
    /**
     * The ARN of the CloudWatch Lambda Insights extension, for the given version.
     *
     * @param insightsVersion the version (e.g. 1.0.98.0).
     * @param architecture the Lambda Function architecture (e.g. 'x86_64' or 'arm64').
     * @stability stable
     */
    cloudwatchLambdaInsightsArn(insightsVersion, architecture) {
        return fact_1.Fact.find(this.name, fact_1.FactName.cloudwatchLambdaInsightsVersion(insightsVersion, architecture));
    }
    /**
     * The ID of the AWS account that owns the public ECR repository that contains the AWS App Mesh Envoy Proxy images in a given region.
     *
     * @stability stable
     */
    get appMeshRepositoryAccount() {
        return fact_1.Fact.find(this.name, fact_1.FactName.APPMESH_ECR_ACCOUNT);
    }
    /**
     * The CIDR block used by Kinesis Data Firehose servers.
     *
     * @stability stable
     */
    get firehoseCidrBlock() {
        return fact_1.Fact.find(this.name, fact_1.FactName.FIREHOSE_CIDR_BLOCK);
    }
}
exports.RegionInfo = RegionInfo;
_a = JSII_RTTI_SYMBOL_1;
RegionInfo[_a] = { fqn: "@aws-cdk/region-info.RegionInfo", version: "1.138.2" };
//# sourceMappingURL=data:application/json;base64,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