"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateRegionMap = exports.partitionInformation = exports.regionsBefore = exports.before = exports.AWS_SERVICES = exports.AWS_REGIONS = exports.AWS_REGIONS_AND_RULES = exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN = exports.RULE_SSM_PRINCIPALS_ARE_REGIONAL = void 0;
// Rule prefix
const RULE_ = 'RULE_';
/**
 * After this point, SSM only creates regional principals
 */
exports.RULE_SSM_PRINCIPALS_ARE_REGIONAL = `${RULE_}SSM_PRINCIPALS_ARE_REGIONAL`;
/**
 * After this point, S3 website domains look like `s3-website.REGION.s3.amazonaws.com`
 *
 * Before this point, S3 website domains look like `s3-website-REGION.s3.amazonaws.com`.
 */
exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN = `${RULE_}S3_WEBSITE_REGIONAL_SUBDOMAIN`;
/**
 * List of AWS region, ordered by launch date (oldest to newest)
 *
 * The significance of this is that standards and conventions change over time.
 * Generally, as rules are changed they only apply to new regions, and old
 * regions are left as-is.
 *
 * We mix the list of regions with a list of rules that were introduced over
 * time (rules are strings starting with `RULE_`).
 *
 * Therefore, if we want to know if a rule applies to a certain region, we
 * only need to check its position in the list and compare it to when a
 * rule was introduced.
 */
exports.AWS_REGIONS_AND_RULES = [
    'us-east-1',
    'eu-west-1',
    'us-west-1',
    'ap-southeast-1',
    'ap-northeast-1',
    'us-gov-west-1',
    'us-west-2',
    'sa-east-1',
    'ap-southeast-2',
    exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN,
    'cn-north-1',
    'us-iso-east-1',
    'eu-central-1',
    'ap-northeast-2',
    'ap-south-1',
    'us-east-2',
    'ca-central-1',
    'eu-west-2',
    'us-isob-east-1',
    'cn-northwest-1',
    'eu-west-3',
    'ap-northeast-3',
    'us-gov-east-1',
    'eu-north-1',
    exports.RULE_SSM_PRINCIPALS_ARE_REGIONAL,
    'ap-east-1',
    'me-south-1',
    'eu-south-1',
    'af-south-1',
    'us-iso-west-1',
    'eu-south-2',
    'ap-southeast-3',
];
/**
 * The names of all (known) AWS regions
 *
 * Not in the list ==> no built-in data for that region.
 */
exports.AWS_REGIONS = exports.AWS_REGIONS_AND_RULES
    .filter((x) => !x.startsWith(RULE_))
    .sort();
/**
 * Possibly non-exaustive list of all service names, used to locate service principals.
 *
 * Not in the list ==> default service principal mappings.
 */
exports.AWS_SERVICES = [
    'application-autoscaling',
    'autoscaling',
    'codedeploy',
    'ec2',
    'events',
    'lambda',
    'logs',
    's3',
    'ssm',
    'sns',
    'sqs',
    'states',
].sort();
/**
 * Whether or not a region predates a given rule (or region).
 *
 * Unknown region => we have to assume no.
 */
function before(region, ruleOrRegion) {
    const ruleIx = exports.AWS_REGIONS_AND_RULES.indexOf(ruleOrRegion);
    if (ruleIx === -1) {
        throw new Error(`Unknown rule: ${ruleOrRegion}`);
    }
    const regionIx = exports.AWS_REGIONS_AND_RULES.indexOf(region);
    return regionIx === -1 ? false : regionIx < ruleIx;
}
exports.before = before;
/**
 * Return all regions before a given rule was introduced (or region)
 */
function regionsBefore(ruleOrRegion) {
    const ruleIx = exports.AWS_REGIONS_AND_RULES.indexOf(ruleOrRegion);
    if (ruleIx === -1) {
        throw new Error(`Unknown rule: ${ruleOrRegion}`);
    }
    return exports.AWS_REGIONS_AND_RULES.filter((_, i) => i < ruleIx).sort();
}
exports.regionsBefore = regionsBefore;
const PARTITION_MAP = {
    'default': { partition: 'aws', domainSuffix: 'amazonaws.com' },
    'cn-': { partition: 'aws-cn', domainSuffix: 'amazonaws.com.cn' },
    'us-gov-': { partition: 'aws-us-gov', domainSuffix: 'amazonaws.com' },
    'us-iso-': { partition: 'aws-iso', domainSuffix: 'c2s.ic.gov' },
    'us-isob-': { partition: 'aws-iso-b', domainSuffix: 'sc2s.sgov.gov' },
};
function partitionInformation(region) {
    for (const [prefix, info] of Object.entries(PARTITION_MAP)) {
        if (region.startsWith(prefix)) {
            return info;
        }
    }
    return PARTITION_MAP.default;
}
exports.partitionInformation = partitionInformation;
/**
 * Build a lookup map for all regions
 */
function generateRegionMap(cb) {
    const ret = {};
    for (const region of exports.AWS_REGIONS) {
        ret[region] = cb(region);
    }
    return ret;
}
exports.generateRegionMap = generateRegionMap;
//# sourceMappingURL=data:application/json;base64,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