"use strict";
/*
 * Invoked as part of the "build" script of this package,
 * this script takes all specification fragments in the
 * `spec-source` folder and generates a unified specification
 * document at `spec/specification.json`.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const md5 = require("md5");
const massage_spec_1 = require("./massage-spec");
async function main() {
    const inputDir = path.join(process.cwd(), 'spec-source');
    const outDir = path.join(process.cwd(), 'spec');
    await generateResourceSpecification(inputDir, path.join(outDir, 'specification.json'));
    await mergeSpecificationFromDirs(path.join(inputDir, 'cfn-lint'), path.join(outDir, 'cfn-lint.json'));
    await fs.copyFile(path.join(inputDir, 'cfn-docs', 'cfn-docs.json'), path.join(outDir, 'cfn-docs.json'));
}
/**
 * Generate CloudFormation resource specification from sources and patches
 */
async function generateResourceSpecification(inputDir, outFile) {
    const spec = { PropertyTypes: {}, ResourceTypes: {}, Fingerprint: '' };
    const files = await fs.readdir(inputDir);
    for (const file of files.filter(n => n.endsWith('.json')).sort()) {
        const data = await fs.readJson(path.join(inputDir, file));
        if (file.indexOf('patch') === -1) {
            massage_spec_1.decorateResourceTypes(data);
            massage_spec_1.forEachSection(spec, data, massage_spec_1.merge);
        }
        else {
            massage_spec_1.forEachSection(spec, data, massage_spec_1.patch);
        }
    }
    massage_spec_1.massageSpec(spec);
    spec.Fingerprint = md5(JSON.stringify(massage_spec_1.normalize(spec)));
    await fs.mkdirp(path.dirname(outFile));
    await fs.writeJson(outFile, spec, { spaces: 2 });
}
/**
 * Generate Cfnlint spec annotations from sources and patches
 */
async function mergeSpecificationFromDirs(inputDir, outFile) {
    const spec = {};
    for (const child of await fs.readdir(inputDir)) {
        const fullPath = path.join(inputDir, child);
        if (!(await fs.stat(fullPath)).isDirectory()) {
            continue;
        }
        const subspec = await loadMergedSpec(fullPath);
        spec[child] = subspec;
    }
    await fs.mkdirp(path.dirname(outFile));
    await fs.writeJson(outFile, spec, { spaces: 2 });
}
/**
 * Load all files in the given directory, merge them and apply patches in the order found
 *
 * The base structure is always an empty object
 */
async function loadMergedSpec(inputDir) {
    const structure = {};
    const files = await fs.readdir(inputDir);
    for (const file of files.filter(n => n.endsWith('.json')).sort()) {
        const data = await fs.readJson(path.join(inputDir, file));
        if (file.indexOf('patch') === -1) {
            // Copy properties from current object into structure, adding/overwriting whatever is found
            Object.assign(structure, data);
        }
        else {
            // Apply the loaded file as a patch onto the current structure
            massage_spec_1.patch(structure, data);
        }
    }
    return structure;
}
main()
    .catch(e => {
    // eslint-disable-next-line no-console
    console.error(e.stack);
    process.exit(-1);
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVpbGQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidWlsZC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7O0dBS0c7O0FBRUgsNkJBQTZCO0FBQzdCLCtCQUErQjtBQUMvQiwyQkFBMkI7QUFFM0IsaURBQTZHO0FBRTdHLEtBQUssVUFBVSxJQUFJO0lBQ2pCLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUFFLGFBQWEsQ0FBQyxDQUFDO0lBQ3pELE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBRWhELE1BQU0sNkJBQTZCLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLG9CQUFvQixDQUFDLENBQUMsQ0FBQztJQUN2RixNQUFNLDBCQUEwQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFVBQVUsQ0FBQyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLGVBQWUsQ0FBQyxDQUFDLENBQUM7SUFDdEcsTUFBTSxFQUFFLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFVBQVUsRUFBRSxlQUFlLENBQUMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxlQUFlLENBQUMsQ0FBQyxDQUFDO0FBQzFHLENBQUM7QUFFRDs7R0FFRztBQUNILEtBQUssVUFBVSw2QkFBNkIsQ0FBQyxRQUFnQixFQUFFLE9BQWU7SUFDNUUsTUFBTSxJQUFJLEdBQXlCLEVBQUUsYUFBYSxFQUFFLEVBQUUsRUFBRSxhQUFhLEVBQUUsRUFBRSxFQUFFLFdBQVcsRUFBRSxFQUFFLEVBQUUsQ0FBQztJQUU3RixNQUFNLEtBQUssR0FBRyxNQUFNLEVBQUUsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDekMsS0FBSyxNQUFNLElBQUksSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFO1FBQ2hFLE1BQU0sSUFBSSxHQUFHLE1BQU0sRUFBRSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQzFELElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRTtZQUNoQyxvQ0FBcUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUM1Qiw2QkFBYyxDQUFDLElBQUksRUFBRSxJQUFJLEVBQUUsb0JBQUssQ0FBQyxDQUFDO1NBQ25DO2FBQU07WUFDTCw2QkFBYyxDQUFDLElBQUksRUFBRSxJQUFJLEVBQUUsb0JBQUssQ0FBQyxDQUFDO1NBQ25DO0tBQ0Y7SUFFRCwwQkFBVyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBRWxCLElBQUksQ0FBQyxXQUFXLEdBQUcsR0FBRyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsd0JBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFFeEQsTUFBTSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztJQUN2QyxNQUFNLEVBQUUsQ0FBQyxTQUFTLENBQUMsT0FBTyxFQUFFLElBQUksRUFBRSxFQUFFLE1BQU0sRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0FBQ25ELENBQUM7QUFFRDs7R0FFRztBQUNILEtBQUssVUFBVSwwQkFBMEIsQ0FBQyxRQUFnQixFQUFFLE9BQWU7SUFDekUsTUFBTSxJQUFJLEdBQVEsRUFBRSxDQUFDO0lBRXJCLEtBQUssTUFBTSxLQUFLLElBQUksTUFBTSxFQUFFLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFO1FBQzlDLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQzVDLElBQUksQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFO1lBQUUsU0FBUztTQUFFO1FBRTNELE1BQU0sT0FBTyxHQUFHLE1BQU0sY0FBYyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxPQUFPLENBQUM7S0FDdkI7SUFFRCxNQUFNLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO0lBQ3ZDLE1BQU0sRUFBRSxDQUFDLFNBQVMsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLEVBQUUsTUFBTSxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7QUFDbkQsQ0FBQztBQUVEOzs7O0dBSUc7QUFDSCxLQUFLLFVBQVUsY0FBYyxDQUFDLFFBQWdCO0lBQzVDLE1BQU0sU0FBUyxHQUFRLEVBQUUsQ0FBQztJQUUxQixNQUFNLEtBQUssR0FBRyxNQUFNLEVBQUUsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDekMsS0FBSyxNQUFNLElBQUksSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFO1FBQ2hFLE1BQU0sSUFBSSxHQUFHLE1BQU0sRUFBRSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQzFELElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRTtZQUNoQywyRkFBMkY7WUFDM0YsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLENBQUM7U0FDaEM7YUFBTTtZQUNMLDhEQUE4RDtZQUM5RCxvQkFBSyxDQUFDLFNBQVMsRUFBRSxJQUFJLENBQUMsQ0FBQztTQUN4QjtLQUNGO0lBRUQsT0FBTyxTQUFTLENBQUM7QUFDbkIsQ0FBQztBQUVELElBQUksRUFBRTtLQUNILEtBQUssQ0FBQyxDQUFDLENBQUMsRUFBRTtJQUNULHNDQUFzQztJQUN0QyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUN2QixPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFDbkIsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogSW52b2tlZCBhcyBwYXJ0IG9mIHRoZSBcImJ1aWxkXCIgc2NyaXB0IG9mIHRoaXMgcGFja2FnZSxcbiAqIHRoaXMgc2NyaXB0IHRha2VzIGFsbCBzcGVjaWZpY2F0aW9uIGZyYWdtZW50cyBpbiB0aGVcbiAqIGBzcGVjLXNvdXJjZWAgZm9sZGVyIGFuZCBnZW5lcmF0ZXMgYSB1bmlmaWVkIHNwZWNpZmljYXRpb25cbiAqIGRvY3VtZW50IGF0IGBzcGVjL3NwZWNpZmljYXRpb24uanNvbmAuXG4gKi9cblxuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzLWV4dHJhJztcbmltcG9ydCAqIGFzIG1kNSBmcm9tICdtZDUnO1xuaW1wb3J0IHsgc2NoZW1hIH0gZnJvbSAnLi4vbGliJztcbmltcG9ydCB7IGRlY29yYXRlUmVzb3VyY2VUeXBlcywgZm9yRWFjaFNlY3Rpb24sIG1hc3NhZ2VTcGVjLCBtZXJnZSwgbm9ybWFsaXplLCBwYXRjaCB9IGZyb20gJy4vbWFzc2FnZS1zcGVjJztcblxuYXN5bmMgZnVuY3Rpb24gbWFpbigpIHtcbiAgY29uc3QgaW5wdXREaXIgPSBwYXRoLmpvaW4ocHJvY2Vzcy5jd2QoKSwgJ3NwZWMtc291cmNlJyk7XG4gIGNvbnN0IG91dERpciA9IHBhdGguam9pbihwcm9jZXNzLmN3ZCgpLCAnc3BlYycpO1xuXG4gIGF3YWl0IGdlbmVyYXRlUmVzb3VyY2VTcGVjaWZpY2F0aW9uKGlucHV0RGlyLCBwYXRoLmpvaW4ob3V0RGlyLCAnc3BlY2lmaWNhdGlvbi5qc29uJykpO1xuICBhd2FpdCBtZXJnZVNwZWNpZmljYXRpb25Gcm9tRGlycyhwYXRoLmpvaW4oaW5wdXREaXIsICdjZm4tbGludCcpLCBwYXRoLmpvaW4ob3V0RGlyLCAnY2ZuLWxpbnQuanNvbicpKTtcbiAgYXdhaXQgZnMuY29weUZpbGUocGF0aC5qb2luKGlucHV0RGlyLCAnY2ZuLWRvY3MnLCAnY2ZuLWRvY3MuanNvbicpLCBwYXRoLmpvaW4ob3V0RGlyLCAnY2ZuLWRvY3MuanNvbicpKTtcbn1cblxuLyoqXG4gKiBHZW5lcmF0ZSBDbG91ZEZvcm1hdGlvbiByZXNvdXJjZSBzcGVjaWZpY2F0aW9uIGZyb20gc291cmNlcyBhbmQgcGF0Y2hlc1xuICovXG5hc3luYyBmdW5jdGlvbiBnZW5lcmF0ZVJlc291cmNlU3BlY2lmaWNhdGlvbihpbnB1dERpcjogc3RyaW5nLCBvdXRGaWxlOiBzdHJpbmcpIHtcbiAgY29uc3Qgc3BlYzogc2NoZW1hLlNwZWNpZmljYXRpb24gPSB7IFByb3BlcnR5VHlwZXM6IHt9LCBSZXNvdXJjZVR5cGVzOiB7fSwgRmluZ2VycHJpbnQ6ICcnIH07XG5cbiAgY29uc3QgZmlsZXMgPSBhd2FpdCBmcy5yZWFkZGlyKGlucHV0RGlyKTtcbiAgZm9yIChjb25zdCBmaWxlIG9mIGZpbGVzLmZpbHRlcihuID0+IG4uZW5kc1dpdGgoJy5qc29uJykpLnNvcnQoKSkge1xuICAgIGNvbnN0IGRhdGEgPSBhd2FpdCBmcy5yZWFkSnNvbihwYXRoLmpvaW4oaW5wdXREaXIsIGZpbGUpKTtcbiAgICBpZiAoZmlsZS5pbmRleE9mKCdwYXRjaCcpID09PSAtMSkge1xuICAgICAgZGVjb3JhdGVSZXNvdXJjZVR5cGVzKGRhdGEpO1xuICAgICAgZm9yRWFjaFNlY3Rpb24oc3BlYywgZGF0YSwgbWVyZ2UpO1xuICAgIH0gZWxzZSB7XG4gICAgICBmb3JFYWNoU2VjdGlvbihzcGVjLCBkYXRhLCBwYXRjaCk7XG4gICAgfVxuICB9XG5cbiAgbWFzc2FnZVNwZWMoc3BlYyk7XG5cbiAgc3BlYy5GaW5nZXJwcmludCA9IG1kNShKU09OLnN0cmluZ2lmeShub3JtYWxpemUoc3BlYykpKTtcblxuICBhd2FpdCBmcy5ta2RpcnAocGF0aC5kaXJuYW1lKG91dEZpbGUpKTtcbiAgYXdhaXQgZnMud3JpdGVKc29uKG91dEZpbGUsIHNwZWMsIHsgc3BhY2VzOiAyIH0pO1xufVxuXG4vKipcbiAqIEdlbmVyYXRlIENmbmxpbnQgc3BlYyBhbm5vdGF0aW9ucyBmcm9tIHNvdXJjZXMgYW5kIHBhdGNoZXNcbiAqL1xuYXN5bmMgZnVuY3Rpb24gbWVyZ2VTcGVjaWZpY2F0aW9uRnJvbURpcnMoaW5wdXREaXI6IHN0cmluZywgb3V0RmlsZTogc3RyaW5nKSB7XG4gIGNvbnN0IHNwZWM6IGFueSA9IHt9O1xuXG4gIGZvciAoY29uc3QgY2hpbGQgb2YgYXdhaXQgZnMucmVhZGRpcihpbnB1dERpcikpIHtcbiAgICBjb25zdCBmdWxsUGF0aCA9IHBhdGguam9pbihpbnB1dERpciwgY2hpbGQpO1xuICAgIGlmICghKGF3YWl0IGZzLnN0YXQoZnVsbFBhdGgpKS5pc0RpcmVjdG9yeSgpKSB7IGNvbnRpbnVlOyB9XG5cbiAgICBjb25zdCBzdWJzcGVjID0gYXdhaXQgbG9hZE1lcmdlZFNwZWMoZnVsbFBhdGgpO1xuICAgIHNwZWNbY2hpbGRdID0gc3Vic3BlYztcbiAgfVxuXG4gIGF3YWl0IGZzLm1rZGlycChwYXRoLmRpcm5hbWUob3V0RmlsZSkpO1xuICBhd2FpdCBmcy53cml0ZUpzb24ob3V0RmlsZSwgc3BlYywgeyBzcGFjZXM6IDIgfSk7XG59XG5cbi8qKlxuICogTG9hZCBhbGwgZmlsZXMgaW4gdGhlIGdpdmVuIGRpcmVjdG9yeSwgbWVyZ2UgdGhlbSBhbmQgYXBwbHkgcGF0Y2hlcyBpbiB0aGUgb3JkZXIgZm91bmRcbiAqXG4gKiBUaGUgYmFzZSBzdHJ1Y3R1cmUgaXMgYWx3YXlzIGFuIGVtcHR5IG9iamVjdFxuICovXG5hc3luYyBmdW5jdGlvbiBsb2FkTWVyZ2VkU3BlYyhpbnB1dERpcjogc3RyaW5nKSB7XG4gIGNvbnN0IHN0cnVjdHVyZTogYW55ID0ge307XG5cbiAgY29uc3QgZmlsZXMgPSBhd2FpdCBmcy5yZWFkZGlyKGlucHV0RGlyKTtcbiAgZm9yIChjb25zdCBmaWxlIG9mIGZpbGVzLmZpbHRlcihuID0+IG4uZW5kc1dpdGgoJy5qc29uJykpLnNvcnQoKSkge1xuICAgIGNvbnN0IGRhdGEgPSBhd2FpdCBmcy5yZWFkSnNvbihwYXRoLmpvaW4oaW5wdXREaXIsIGZpbGUpKTtcbiAgICBpZiAoZmlsZS5pbmRleE9mKCdwYXRjaCcpID09PSAtMSkge1xuICAgICAgLy8gQ29weSBwcm9wZXJ0aWVzIGZyb20gY3VycmVudCBvYmplY3QgaW50byBzdHJ1Y3R1cmUsIGFkZGluZy9vdmVyd3JpdGluZyB3aGF0ZXZlciBpcyBmb3VuZFxuICAgICAgT2JqZWN0LmFzc2lnbihzdHJ1Y3R1cmUsIGRhdGEpO1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyBBcHBseSB0aGUgbG9hZGVkIGZpbGUgYXMgYSBwYXRjaCBvbnRvIHRoZSBjdXJyZW50IHN0cnVjdHVyZVxuICAgICAgcGF0Y2goc3RydWN0dXJlLCBkYXRhKTtcbiAgICB9XG4gIH1cblxuICByZXR1cm4gc3RydWN0dXJlO1xufVxuXG5tYWluKClcbiAgLmNhdGNoKGUgPT4ge1xuICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1jb25zb2xlXG4gICAgY29uc29sZS5lcnJvcihlLnN0YWNrKTtcbiAgICBwcm9jZXNzLmV4aXQoLTEpO1xuICB9KTtcbiJdfQ==