"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TagType = exports.CfnResource = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxapi = require("@aws-cdk/cx-api");
// import required to be here, otherwise causes a cycle when running the generated JavaScript
/* eslint-disable import/order */
const cfn_element_1 = require("./cfn-element");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const constructs_1 = require("constructs");
const deps_1 = require("./deps");
const cfn_reference_1 = require("./private/cfn-reference");
const cloudformation_lang_1 = require("./private/cloudformation-lang");
const removal_policy_1 = require("./removal-policy");
const tag_manager_1 = require("./tag-manager");
const token_1 = require("./token");
const util_1 = require("./util");
/**
 * Represents a CloudFormation resource.
 *
 * @stability stable
 */
class CfnResource extends cfn_element_1.CfnRefElement {
    /**
     * Creates a resource construct.
     *
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        // MAINTAINERS NOTE: this class serves as the base class for the generated L1
        // ("CFN") resources (such as `s3.CfnBucket`). These resources will have a
        // property for each CloudFormation property of the resource. This means that
        // if at some point in the future a property is introduced with a name similar
        // to one of the properties here, it will be "masked" by the derived class. To
        // that end, we prefix all properties in this class with `cfnXxx` with the
        // hope to avoid those conflicts in the future.
        /**
         * Options for this resource, such as condition, update policy etc.
         *
         * @stability stable
         */
        this.cfnOptions = {};
        /**
         * An object to be merged on top of the entire resource definition.
         */
        this.rawOverrides = {};
        /**
         * Logical IDs of dependencies.
         *
         * Is filled during prepare().
         */
        this.dependsOn = new Set();
        jsiiDeprecationWarnings._aws_cdk_core_CfnResourceProps(props);
        if (!props.type) {
            throw new Error('The `type` property is required');
        }
        this.cfnResourceType = props.type;
        this._cfnProperties = props.properties || {};
        // if aws:cdk:enable-path-metadata is set, embed the current construct's
        // path in the CloudFormation template, so it will be possible to trace
        // back to the actual construct path.
        if (constructs_1.Node.of(this).tryGetContext(cxapi.PATH_METADATA_ENABLE_CONTEXT)) {
            this.addMetadata(cxapi.PATH_METADATA_KEY, constructs_1.Node.of(this).path);
        }
    }
    /**
     * Check whether the given construct is a CfnResource.
     *
     * @stability stable
     */
    static isCfnResource(construct) {
        return construct.cfnResourceType !== undefined;
    }
    /**
     * Sets the deletion policy of the resource based on the removal policy specified.
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be deleted (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * @stability stable
     */
    applyRemovalPolicy(policy, options = {}) {
        jsiiDeprecationWarnings._aws_cdk_core_RemovalPolicy(policy);
        jsiiDeprecationWarnings._aws_cdk_core_RemovalPolicyOptions(options);
        policy = policy || options.default || removal_policy_1.RemovalPolicy.RETAIN;
        let deletionPolicy;
        switch (policy) {
            case removal_policy_1.RemovalPolicy.DESTROY:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
                break;
            case removal_policy_1.RemovalPolicy.RETAIN:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
                break;
            case removal_policy_1.RemovalPolicy.SNAPSHOT:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
                break;
            default:
                throw new Error(`Invalid removal policy: ${policy}`);
        }
        this.cfnOptions.deletionPolicy = deletionPolicy;
        if (options.applyToUpdateReplacePolicy !== false) {
            this.cfnOptions.updateReplacePolicy = deletionPolicy;
        }
    }
    /**
     * Returns a token for an runtime attribute of this resource.
     *
     * Ideally, use generated attribute accessors (e.g. `resource.arn`), but this can be used for future compatibility
     * in case there is no generated attribute.
     *
     * @param attributeName The name of the attribute.
     * @stability stable
     */
    getAtt(attributeName) {
        return cfn_reference_1.CfnReference.for(this, attributeName);
    }
    /**
     * Adds an override to the synthesized CloudFormation resource.
     *
     * To add a
     * property override, either use `addPropertyOverride` or prefix `path` with
     * "Properties." (i.e. `Properties.TopicName`).
     *
     * If the override is nested, separate each nested level using a dot (.) in the path parameter.
     * If there is an array as part of the nesting, specify the index in the path.
     *
     * To include a literal `.` in the property name, prefix with a `\`. In most
     * programming languages you will need to write this as `"\\."` because the
     * `\` itself will need to be escaped.
     *
     * For example,
     * ```typescript
     * cfnResource.addOverride('Properties.GlobalSecondaryIndexes.0.Projection.NonKeyAttributes', ['myattribute']);
     * cfnResource.addOverride('Properties.GlobalSecondaryIndexes.1.ProjectionType', 'INCLUDE');
     * ```
     * would add the overrides
     * ```json
     * "Properties": {
     *    "GlobalSecondaryIndexes": [
     *      {
     *        "Projection": {
     *          "NonKeyAttributes": [ "myattribute" ]
     *          ...
     *        }
     *        ...
     *      },
     *      {
     *        "ProjectionType": "INCLUDE"
     *        ...
     *      },
     *    ]
     *    ...
     * }
     * ```
     *
     * @param path - The path of the property, you can use dot notation to override values in complex types.
     * @param value - The value.
     * @stability stable
     */
    addOverride(path, value) {
        const parts = splitOnPeriods(path);
        let curr = this.rawOverrides;
        while (parts.length > 1) {
            const key = parts.shift();
            // if we can't recurse further or the previous value is not an
            // object overwrite it with an object.
            const isObject = curr[key] != null && typeof (curr[key]) === 'object' && !Array.isArray(curr[key]);
            if (!isObject) {
                curr[key] = {};
            }
            curr = curr[key];
        }
        const lastKey = parts.shift();
        curr[lastKey] = value;
    }
    /**
     * Syntactic sugar for `addOverride(path, undefined)`.
     *
     * @param path The path of the value to delete.
     * @stability stable
     */
    addDeletionOverride(path) {
        this.addOverride(path, undefined);
    }
    /**
     * Adds an override to a resource property.
     *
     * Syntactic sugar for `addOverride("Properties.<...>", value)`.
     *
     * @param propertyPath The path of the property.
     * @param value The value.
     * @stability stable
     */
    addPropertyOverride(propertyPath, value) {
        this.addOverride(`Properties.${propertyPath}`, value);
    }
    /**
     * Adds an override that deletes the value of a property from the resource definition.
     *
     * @param propertyPath The path to the property.
     * @stability stable
     */
    addPropertyDeletionOverride(propertyPath) {
        this.addPropertyOverride(propertyPath, undefined);
    }
    /**
     * Indicates that this resource depends on another resource and cannot be provisioned unless the other resource has been successfully provisioned.
     *
     * This can be used for resources across stacks (or nested stack) boundaries
     * and the dependency will automatically be transferred to the relevant scope.
     *
     * @stability stable
     */
    addDependsOn(target) {
        jsiiDeprecationWarnings._aws_cdk_core_CfnResource(target);
        // skip this dependency if the target is not part of the output
        if (!target.shouldSynthesize()) {
            return;
        }
        deps_1.addDependency(this, target, `"${constructs_1.Node.of(this).path}" depends on "${constructs_1.Node.of(target).path}"`);
    }
    /**
     * Add a value to the CloudFormation Resource Metadata.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/metadata-section-structure.html
     *
     * Note that this is a different set of metadata from CDK node metadata; this
     * metadata ends up in the stack template under the resource, whereas CDK
     * node metadata ends up in the Cloud Assembly.
     * @stability stable
     */
    addMetadata(key, value) {
        if (!this.cfnOptions.metadata) {
            this.cfnOptions.metadata = {};
        }
        this.cfnOptions.metadata[key] = value;
    }
    /**
     * Retrieve a value value from the CloudFormation Resource Metadata.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/metadata-section-structure.html
     *
     * Note that this is a different set of metadata from CDK node metadata; this
     * metadata ends up in the stack template under the resource, whereas CDK
     * node metadata ends up in the Cloud Assembly.
     * @stability stable
     */
    getMetadata(key) {
        var _b;
        return (_b = this.cfnOptions.metadata) === null || _b === void 0 ? void 0 : _b[key];
    }
    /**
     * Returns a string representation of this construct.
     *
     * @returns a string representation of this resource
     * @stability stable
     */
    toString() {
        return `${super.toString()} [${this.cfnResourceType}]`;
    }
    /**
     * Called by the `addDependency` helper function in order to realize a direct
     * dependency between two resources that are directly defined in the same
     * stacks.
     *
     * Use `resource.addDependsOn` to define the dependency between two resources,
     * which also takes stack boundaries into account.
     *
     * @internal
     */
    _addResourceDependency(target) {
        this.dependsOn.add(target);
    }
    /**
     * Emits CloudFormation for this resource.
     * @internal
     */
    _toCloudFormation() {
        if (!this.shouldSynthesize()) {
            return {};
        }
        try {
            const ret = {
                Resources: {
                    // Post-Resolve operation since otherwise deepMerge is going to mix values into
                    // the Token objects returned by ignoreEmpty.
                    [this.logicalId]: new util_1.PostResolveToken({
                        Type: this.cfnResourceType,
                        Properties: util_1.ignoreEmpty(this.cfnProperties),
                        DependsOn: util_1.ignoreEmpty(renderDependsOn(this.dependsOn)),
                        CreationPolicy: util_1.capitalizePropertyNames(this, renderCreationPolicy(this.cfnOptions.creationPolicy)),
                        UpdatePolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.updatePolicy),
                        UpdateReplacePolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.updateReplacePolicy),
                        DeletionPolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.deletionPolicy),
                        Version: this.cfnOptions.version,
                        Description: this.cfnOptions.description,
                        Metadata: util_1.ignoreEmpty(this.cfnOptions.metadata),
                        Condition: this.cfnOptions.condition && this.cfnOptions.condition.logicalId,
                    }, resourceDef => {
                        const renderedProps = this.renderProperties(resourceDef.Properties || {});
                        if (renderedProps) {
                            const hasDefined = Object.values(renderedProps).find(v => v !== undefined);
                            resourceDef.Properties = hasDefined !== undefined ? renderedProps : undefined;
                        }
                        const resolvedRawOverrides = token_1.Tokenization.resolve(this.rawOverrides, {
                            scope: this,
                            resolver: cloudformation_lang_1.CLOUDFORMATION_TOKEN_RESOLVER,
                            // we need to preserve the empty elements here,
                            // as that's how removing overrides are represented as
                            removeEmpty: false,
                        });
                        return deepMerge(resourceDef, resolvedRawOverrides);
                    }),
                },
            };
            return ret;
        }
        catch (e) {
            // Change message
            e.message = `While synthesizing ${this.node.path}: ${e.message}`;
            // Adjust stack trace (make it look like node built it, too...)
            const trace = this.creationStack;
            if (trace) {
                const creationStack = ['--- resource created at ---', ...trace].join('\n  at ');
                const problemTrace = e.stack.substr(e.stack.indexOf(e.message) + e.message.length);
                e.stack = `${e.message}\n  ${creationStack}\n  --- problem discovered at ---${problemTrace}`;
            }
            // Re-throw
            throw e;
        }
        // returns the set of logical ID (tokens) this resource depends on
        // sorted by construct paths to ensure test determinism
        function renderDependsOn(dependsOn) {
            return Array
                .from(dependsOn)
                .sort((x, y) => x.node.path.localeCompare(y.node.path))
                .map(r => r.logicalId);
        }
        function renderCreationPolicy(policy) {
            if (!policy) {
                return undefined;
            }
            const result = { ...policy };
            if (policy.resourceSignal && policy.resourceSignal.timeout) {
                result.resourceSignal = policy.resourceSignal;
            }
            return result;
        }
    }
    /**
     * @stability stable
     */
    get cfnProperties() {
        const props = this._cfnProperties || {};
        if (tag_manager_1.TagManager.isTaggable(this)) {
            const tagsProp = {};
            tagsProp[this.tags.tagPropertyName] = this.tags.renderTags();
            return deepMerge(props, tagsProp);
        }
        return props;
    }
    /**
     * @stability stable
     */
    renderProperties(props) {
        return props;
    }
    /**
     * Return properties modified after initiation.
     *
     * Resources that expose mutable properties should override this function to
     * collect and return the properties object for this resource.
     *
     * @stability stable
     */
    get updatedProperites() {
        return this._cfnProperties;
    }
    /**
     * @stability stable
     */
    validateProperties(_properties) {
    }
    /**
     * Can be overridden by subclasses to determine if this resource will be rendered into the cloudformation template.
     *
     * @returns `true` if the resource should be included or `false` is the resource
     * should be omitted.
     * @stability stable
     */
    shouldSynthesize() {
        return true;
    }
}
exports.CfnResource = CfnResource;
_a = JSII_RTTI_SYMBOL_1;
CfnResource[_a] = { fqn: "@aws-cdk/core.CfnResource", version: "1.138.0" };
/**
 * @stability stable
 */
var TagType;
(function (TagType) {
    TagType["STANDARD"] = "StandardTag";
    TagType["AUTOSCALING_GROUP"] = "AutoScalingGroupTag";
    TagType["MAP"] = "StringToStringMap";
    TagType["KEY_VALUE"] = "KeyValue";
    TagType["NOT_TAGGABLE"] = "NotTaggable";
})(TagType = exports.TagType || (exports.TagType = {}));
/**
 * Merges `source` into `target`, overriding any existing values.
 * `null`s will cause a value to be deleted.
 */
function deepMerge(target, ...sources) {
    for (const source of sources) {
        if (typeof (source) !== 'object' || typeof (target) !== 'object') {
            throw new Error(`Invalid usage. Both source (${JSON.stringify(source)}) and target (${JSON.stringify(target)}) must be objects`);
        }
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (typeof (value) === 'object' && value != null && !Array.isArray(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof (target[key]) !== 'object') {
                    target[key] = {};
                }
                deepMerge(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof (output) === 'object' && Object.keys(output).length === 0) {
                    delete target[key];
                }
            }
            else if (value === undefined) {
                delete target[key];
            }
            else {
                target[key] = value;
            }
        }
    }
    return target;
}
/**
 * Split on periods while processing escape characters \
 */
function splitOnPeriods(x) {
    // Build this list in reverse because it's more convenient to get the "current"
    // item by doing ret[0] than by ret[ret.length - 1].
    const ret = [''];
    for (let i = 0; i < x.length; i++) {
        if (x[i] === '\\' && i + 1 < x.length) {
            ret[0] += x[i + 1];
            i++;
        }
        else if (x[i] === '.') {
            ret.unshift('');
        }
        else {
            ret[0] += x[i];
        }
    }
    ret.reverse();
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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