"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Default = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_entities_1 = require("./aws-entities");
/**
 * Provides default values for certain regional information points.
 *
 * @stability stable
 */
class Default {
    constructor() { }
    /**
     * Computes a "standard" AWS Service principal for a given service, region and suffix.
     *
     * This is useful for example when
     * you need to compute a service principal name, but you do not have a synthesize-time region literal available (so
     * all you have is `{ "Ref": "AWS::Region" }`). This way you get the same defaulting behavior that is normally used
     * for built-in data.
     *
     * @param serviceFqn the name of the service (s3, s3.amazonaws.com, ...).
     * @param region the region in which the service principal is needed.
     * @param urlSuffix deprecated and ignored.
     * @stability stable
     */
    static servicePrincipal(serviceFqn, region, urlSuffix) {
        const service = extractSimpleName(serviceFqn);
        if (!service) {
            // Return "service" if it does not look like any of the following:
            // - s3
            // - s3.amazonaws.com
            // - s3.amazonaws.com.cn
            // - s3.c2s.ic.gov
            // - s3.sc2s.sgov.gov
            return serviceFqn;
        }
        // Exceptions for Service Principals in us-iso-*
        const US_ISO_EXCEPTIONS = new Set([
            'cloudhsm',
            'config',
            'states',
            'workspaces',
        ]);
        // Account for idiosyncratic Service Principals in `us-iso-*` regions
        if (region.startsWith('us-iso-') && US_ISO_EXCEPTIONS.has(service)) {
            switch (service) {
                // Services with universal principal
                case ('states'):
                    return `${service}.amazonaws.com`;
                // Services with a partitional principal
                default:
                    return `${service}.${urlSuffix}`;
            }
        }
        // Exceptions for Service Principals in us-isob-*
        const US_ISOB_EXCEPTIONS = new Set([
            'dms',
            'states',
        ]);
        // Account for idiosyncratic Service Principals in `us-isob-*` regions
        if (region.startsWith('us-isob-') && US_ISOB_EXCEPTIONS.has(service)) {
            switch (service) {
                // Services with universal principal
                case ('states'):
                    return `${service}.amazonaws.com`;
                // Services with a partitional principal
                default:
                    return `${service}.${urlSuffix}`;
            }
        }
        // SSM turned from global to regional at some point
        if (service === 'ssm') {
            return aws_entities_1.before(region, aws_entities_1.RULE_SSM_PRINCIPALS_ARE_REGIONAL)
                ? `${service}.amazonaws.com`
                : `${service}.${region}.amazonaws.com`;
        }
        switch (service) {
            // Services with a regional AND partitional principal
            case 'codedeploy':
            case 'logs':
                return `${service}.${region}.${urlSuffix}`;
            // Services with a regional principal
            case 'states':
                return `${service}.${region}.amazonaws.com`;
            // Services with a partitional principal
            case 'ec2':
                return `${service}.${urlSuffix}`;
            // Services with a universal principal across all regions/partitions (the default case)
            default:
                return `${service}.amazonaws.com`;
        }
    }
}
exports.Default = Default;
_a = JSII_RTTI_SYMBOL_1;
Default[_a] = { fqn: "@aws-cdk/region-info.Default", version: "1.137.0" };
/**
 * The default value for a VPC Endpoint Service name prefix, useful if you do not have a synthesize-time region literal available (all you have is `{ "Ref": "AWS::Region" }`).
 *
 * @stability stable
 */
Default.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'com.amazonaws.vpce';
function extractSimpleName(serviceFqn) {
    const matches = serviceFqn.match(/^([^.]+)(?:(?:\.amazonaws\.com(?:\.cn)?)|(?:\.c2s\.ic\.gov)|(?:\.sc2s\.sgov\.gov))?$/);
    return matches ? matches[1] : undefined;
}
//# sourceMappingURL=data:application/json;base64,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