"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateRegionMap = exports.partitionInformation = exports.regionsBefore = exports.before = exports.AWS_SERVICES = exports.AWS_REGIONS = exports.AWS_REGIONS_AND_RULES = exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN = exports.RULE_SSM_PRINCIPALS_ARE_REGIONAL = void 0;
// Rule prefix
const RULE_ = 'RULE_';
/**
 * After this point, SSM only creates regional principals
 */
exports.RULE_SSM_PRINCIPALS_ARE_REGIONAL = `${RULE_}SSM_PRINCIPALS_ARE_REGIONAL`;
/**
 * After this point, S3 website domains look like `s3-website.REGION.s3.amazonaws.com`
 *
 * Before this point, S3 website domains look like `s3-website-REGION.s3.amazonaws.com`.
 */
exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN = `${RULE_}SSM_PRINCIPALS_ARE_REGIONAL`;
/**
 * List of AWS region, ordered by launch date (oldest to newest)
 *
 * The significance of this is that standards and conventions change over time.
 * Generally, as rules are changed they only apply to new regions, and old
 * regions are left as-is.
 *
 * We mix the list of regions with a list of rules that were introduced over
 * time (rules are strings starting with `RULE_`).
 *
 * Therefore, if we want to know if a rule applies to a certain region, we
 * only need to check its position in the list and compare it to when a
 * rule was introduced.
 */
exports.AWS_REGIONS_AND_RULES = [
    'us-east-1',
    'eu-west-1',
    'us-west-1',
    'ap-southeast-1',
    'ap-northeast-1',
    'us-gov-west-1',
    'us-west-2',
    'sa-east-1',
    'ap-southeast-2',
    exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN,
    'cn-north-1',
    'us-iso-east-1',
    'eu-central-1',
    'ap-northeast-2',
    'ap-south-1',
    'us-east-2',
    'ca-central-1',
    'eu-west-2',
    'us-isob-east-1',
    'cn-northwest-1',
    'eu-west-3',
    'ap-northeast-3',
    'us-gov-east-1',
    'eu-north-1',
    exports.RULE_SSM_PRINCIPALS_ARE_REGIONAL,
    'ap-east-1',
    'me-south-1',
    'eu-south-1',
    'af-south-1',
    'us-iso-west-1',
    'eu-south-2',
    'ap-southeast-3',
];
/**
 * The names of all (known) AWS regions
 *
 * Not in the list ==> no built-in data for that region.
 */
exports.AWS_REGIONS = exports.AWS_REGIONS_AND_RULES
    .filter((x) => !x.startsWith(RULE_))
    .sort();
/**
 * Possibly non-exaustive list of all service names, used to locate service principals.
 *
 * Not in the list ==> default service principal mappings.
 */
exports.AWS_SERVICES = [
    'application-autoscaling',
    'autoscaling',
    'codedeploy',
    'ec2',
    'events',
    'lambda',
    'logs',
    's3',
    'ssm',
    'sns',
    'sqs',
    'states',
].sort();
/**
 * Whether or not a region predates a given rule (or region).
 *
 * Unknown region => we have to assume no.
 */
function before(region, ruleOrRegion) {
    const ruleIx = exports.AWS_REGIONS_AND_RULES.indexOf(ruleOrRegion);
    if (ruleIx === -1) {
        throw new Error(`Unknown rule: ${ruleOrRegion}`);
    }
    const regionIx = exports.AWS_REGIONS_AND_RULES.indexOf(region);
    return regionIx === -1 ? false : regionIx < ruleIx;
}
exports.before = before;
/**
 * Return all regions before a given rule was introduced (or region)
 */
function regionsBefore(ruleOrRegion) {
    const ruleIx = exports.AWS_REGIONS_AND_RULES.indexOf(ruleOrRegion);
    if (ruleIx === -1) {
        throw new Error(`Unknown rule: ${ruleOrRegion}`);
    }
    return exports.AWS_REGIONS_AND_RULES.filter((_, i) => i < ruleIx).sort();
}
exports.regionsBefore = regionsBefore;
const PARTITION_MAP = {
    'default': { partition: 'aws', domainSuffix: 'amazonaws.com' },
    'cn-': { partition: 'aws-cn', domainSuffix: 'amazonaws.com.cn' },
    'us-gov-': { partition: 'aws-us-gov', domainSuffix: 'amazonaws.com' },
    'us-iso-': { partition: 'aws-iso', domainSuffix: 'c2s.ic.gov' },
    'us-isob-': { partition: 'aws-iso-b', domainSuffix: 'sc2s.sgov.gov' },
};
function partitionInformation(region) {
    for (const [prefix, info] of Object.entries(PARTITION_MAP)) {
        if (region.startsWith(prefix)) {
            return info;
        }
    }
    return PARTITION_MAP.default;
}
exports.partitionInformation = partitionInformation;
/**
 * Build a lookup map for all regions
 */
function generateRegionMap(cb) {
    const ret = {};
    for (const region of exports.AWS_REGIONS) {
        ret[region] = cb(region);
    }
    return ret;
}
exports.generateRegionMap = generateRegionMap;
//# sourceMappingURL=data:application/json;base64,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