"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FactName = exports.Fact = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * A database of regional information.
 *
 * @stability stable
 */
class Fact {
    constructor() {
        throw new Error('Use the static methods of Fact instead!');
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     * may not be an exhaustive list of all available AWS regions.
     * @stability stable
     */
    static get regions() {
        return Object.keys(this.database);
    }
    /**
     * Retrieves a fact from this Fact database.
     *
     * @param region the name of the region (e.g: `us-east-1`).
     * @param name the name of the fact being looked up (see the `FactName` class for details).
     * @returns the fact value if it is known, and `undefined` otherwise.
     * @stability stable
     */
    static find(region, name) {
        const regionFacts = this.database[region];
        return regionFacts && regionFacts[name];
    }
    /**
     * Retrieve a fact from the Fact database.
     *
     * (retrieval will fail if the specified region or
     * fact name does not exist.)
     *
     * @param region the name of the region (e.g: `us-east-1`).
     * @param name the name of the fact being looked up (see the `FactName` class for details).
     * @stability stable
     */
    static requireFact(region, name) {
        const foundFact = this.find(region, name);
        if (!foundFact) {
            throw new Error(`No fact ${name} could be found for region: ${region} and name: ${name}`);
        }
        return foundFact;
    }
    /**
     * Registers a new fact in this Fact database.
     *
     * @param fact the new fact to be registered.
     * @param allowReplacing whether new facts can replace existing facts or not.
     * @stability stable
     */
    static register(fact, allowReplacing = false) {
        jsiiDeprecationWarnings._aws_cdk_region_info_IFact(fact);
        const regionFacts = this.database[fact.region] || (this.database[fact.region] = {});
        if (fact.name in regionFacts && regionFacts[fact.name] !== fact.value && !allowReplacing) {
            throw new Error(`Region ${fact.region} already has a fact ${fact.name}, with value ${regionFacts[fact.name]}`);
        }
        if (fact.value !== undefined) {
            regionFacts[fact.name] = fact.value;
        }
    }
    /**
     * Removes a fact from the database.
     *
     * @param region the region for which the fact is to be removed.
     * @param name the name of the fact to remove.
     * @param value the value that should be removed (removal will fail if the value is specified, but does not match the current stored value).
     * @stability stable
     */
    static unregister(region, name, value) {
        const regionFacts = this.database[region] || {};
        if (name in regionFacts && value && regionFacts[name] !== value) {
            throw new Error(`Attempted to remove ${name} from ${region} with value ${value}, but the fact's value is ${regionFacts[name]}`);
        }
        delete regionFacts[name];
    }
}
exports.Fact = Fact;
_a = JSII_RTTI_SYMBOL_1;
Fact[_a] = { fqn: "@aws-cdk/region-info.Fact", version: "1.136.0" };
Fact.database = {};
/**
 * All standardized fact names.
 *
 * @stability stable
 */
class FactName {
    /**
     * The ARN of CloudWatch Lambda Insights for a version (e.g. 1.0.98.0).
     *
     * @stability stable
     */
    static cloudwatchLambdaInsightsVersion(version, arch) {
        // if we are provided an architecture use that, otherwise
        // default to x86_64 for backwards compatibility
        const suffix = version.split('.').join('_') + `_${arch !== null && arch !== void 0 ? arch : 'x86_64'}`;
        return `cloudwatch-lambda-insights-version:${suffix}`;
    }
    /**
     * The name of the regional service principal for a given service.
     *
     * @param service the service name, either simple (e.g: `s3`, `codedeploy`) or qualified (e.g: `s3.amazonaws.com`). The `.amazonaws.com` and `.amazonaws.com.cn` domains are stripped from service names, so they are canonicalized in that respect.
     * @stability stable
     */
    static servicePrincipal(service) {
        return `service-principal:${service.replace(/\.amazonaws\.com(\.cn)?$/, '')}`;
    }
}
exports.FactName = FactName;
_b = JSII_RTTI_SYMBOL_1;
FactName[_b] = { fqn: "@aws-cdk/region-info.FactName", version: "1.136.0" };
/**
 * The name of the partition for a region (e.g: 'aws', 'aws-cn', ...).
 *
 * @stability stable
 */
FactName.PARTITION = 'partition';
/**
 * The domain suffix for a region (e.g: 'amazonaws.com`).
 *
 * @stability stable
 */
FactName.DOMAIN_SUFFIX = 'domainSuffix';
/**
 * Whether the AWS::CDK::Metadata CloudFormation Resource is available in-region or not.
 *
 * The value is a boolean
 * modelled as `YES` or `NO`.
 *
 * @stability stable
 */
FactName.CDK_METADATA_RESOURCE_AVAILABLE = 'cdk:metadata-resource:available';
/**
 * The endpoint used for hosting S3 static websites.
 *
 * @stability stable
 */
FactName.S3_STATIC_WEBSITE_ENDPOINT = 's3-static-website:endpoint';
/**
 * The endpoint used for aliasing S3 static websites in Route 53.
 *
 * @stability stable
 */
FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID = 's3-static-website:route-53-hosted-zone-id';
/**
 * The hosted zone ID used by Route 53 to alias a EBS environment endpoint in this region (e.g: Z2O1EMRO9K5GLX).
 *
 * @stability stable
 */
FactName.EBS_ENV_ENDPOINT_HOSTED_ZONE_ID = 'ebs-environment:route-53-hosted-zone-id';
/**
 * The prefix for VPC Endpoint Service names, cn.com.amazonaws.vpce for China regions, com.amazonaws.vpce otherwise.
 *
 * @stability stable
 */
FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'vpcEndpointServiceNamePrefix';
/**
 * The account for ELBv2 in this region.
 *
 * @stability stable
 */
FactName.ELBV2_ACCOUNT = 'elbv2Account';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the AWS Deep Learning Containers images in a given region.
 *
 * @stability stable
 */
FactName.DLC_REPOSITORY_ACCOUNT = 'dlcRepositoryAccount';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the AWS App Mesh Envoy Proxy images in a given region.
 *
 * @stability stable
 */
FactName.APPMESH_ECR_ACCOUNT = 'appMeshRepositoryAccount';
/**
 * The CIDR block used by Kinesis Data Firehose servers.
 *
 * @stability stable
 */
FactName.FIREHOSE_CIDR_BLOCK = 'firehoseCidrBlock';
//# sourceMappingURL=data:application/json;base64,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