"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudArtifact = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const metadata_1 = require("./metadata");
/**
 * Represents an artifact within a cloud assembly.
 *
 * @stability stable
 */
class CloudArtifact {
    /**
     * @stability stable
     */
    constructor(assembly, id, manifest) {
        this.assembly = assembly;
        this.id = id;
        jsiiDeprecationWarnings._aws_cdk_cx_api_CloudAssembly(assembly);
        this.manifest = manifest;
        this.messages = this.renderMessages();
        this._dependencyIDs = manifest.dependencies || [];
    }
    /**
     * Returns a subclass of `CloudArtifact` based on the artifact type defined in the artifact manifest.
     *
     * @param assembly The cloud assembly from which to load the artifact.
     * @param id The artifact ID.
     * @param artifact The artifact manifest.
     * @returns the `CloudArtifact` that matches the artifact type or `undefined` if it's an artifact type that is unrecognized by this module.
     * @stability stable
     */
    static fromManifest(assembly, id, artifact) {
        jsiiDeprecationWarnings._aws_cdk_cx_api_CloudAssembly(assembly);
        switch (artifact.type) {
            case cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK:
                return new cloudformation_artifact_1.CloudFormationStackArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.CDK_TREE:
                return new tree_cloud_artifact_1.TreeCloudArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.ASSET_MANIFEST:
                return new asset_manifest_artifact_1.AssetManifestArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.NESTED_CLOUD_ASSEMBLY:
                return new nested_cloud_assembly_artifact_1.NestedCloudAssemblyArtifact(assembly, id, artifact);
            default:
                return undefined;
        }
    }
    /**
     * Returns all the artifacts that this artifact depends on.
     *
     * @stability stable
     */
    get dependencies() {
        if (this._deps) {
            return this._deps;
        }
        this._deps = this._dependencyIDs.map(id => {
            const dep = this.assembly.tryGetArtifact(id);
            if (!dep) {
                throw new Error(`Artifact ${this.id} depends on non-existing artifact ${id}`);
            }
            return dep;
        });
        return this._deps;
    }
    /**
     * @returns all the metadata entries of a specific type in this artifact.
     * @stability stable
     */
    findMetadataByType(type) {
        const result = new Array();
        for (const path of Object.keys(this.manifest.metadata || {})) {
            for (const entry of (this.manifest.metadata || {})[path]) {
                if (entry.type === type) {
                    result.push({ path, ...entry });
                }
            }
        }
        return result;
    }
    renderMessages() {
        const messages = new Array();
        for (const [id, metadata] of Object.entries(this.manifest.metadata || {})) {
            for (const entry of metadata) {
                let level;
                switch (entry.type) {
                    case cxschema.ArtifactMetadataEntryType.WARN:
                        level = metadata_1.SynthesisMessageLevel.WARNING;
                        break;
                    case cxschema.ArtifactMetadataEntryType.ERROR:
                        level = metadata_1.SynthesisMessageLevel.ERROR;
                        break;
                    case cxschema.ArtifactMetadataEntryType.INFO:
                        level = metadata_1.SynthesisMessageLevel.INFO;
                        break;
                    default:
                        continue;
                }
                messages.push({ level, entry, id });
            }
        }
        return messages;
    }
    /**
     * An identifier that shows where this artifact is located in the tree of nested assemblies, based on their manifests.
     *
     * Defaults to the normal
     * id. Should only be used in user interfaces.
     *
     * @stability stable
     */
    get hierarchicalId() {
        var _b;
        return (_b = this.manifest.displayName) !== null && _b !== void 0 ? _b : this.id;
    }
}
exports.CloudArtifact = CloudArtifact;
_a = JSII_RTTI_SYMBOL_1;
CloudArtifact[_a] = { fqn: "@aws-cdk/cx-api.CloudArtifact", version: "1.136.0" };
// needs to be defined at the end to avoid a cyclic dependency
const asset_manifest_artifact_1 = require("./artifacts/asset-manifest-artifact");
const cloudformation_artifact_1 = require("./artifacts/cloudformation-artifact");
const nested_cloud_assembly_artifact_1 = require("./artifacts/nested-cloud-assembly-artifact");
const tree_cloud_artifact_1 = require("./artifacts/tree-cloud-artifact");
//# sourceMappingURL=data:application/json;base64,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