"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const default_1 = require("../lib/default");
const aws_entities_1 = require("./aws-entities");
const fact_tables_1 = require("./fact-tables");
async function main() {
    checkRegions(fact_tables_1.APPMESH_ECR_ACCOUNTS);
    checkRegions(fact_tables_1.DLC_REPOSITORY_ACCOUNTS);
    checkRegions(fact_tables_1.ELBV2_ACCOUNTS);
    checkRegions(fact_tables_1.FIREHOSE_CIDR_BLOCKS);
    checkRegions(fact_tables_1.ROUTE_53_BUCKET_WEBSITE_ZONE_IDS);
    checkRegionsSubMap(fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS);
    const lines = [
        "import { Fact, FactName } from './fact';",
        '',
        '/* eslint-disable quote-props */',
        '/* eslint-disable max-len */',
        '',
        '/**',
        ' * Built-in regional information, re-generated by `npm run build`.',
        ' *',
        ` * @generated ${new Date().toISOString()}`,
        ' */',
        'export class BuiltIns {',
        '  /**',
        '   * Registers all the built in regional data in the RegionInfo database.',
        '   */',
        '  public static register(): void {',
    ];
    const defaultMap = 'default';
    for (const region of aws_entities_1.AWS_REGIONS) {
        let partition = fact_tables_1.PARTITION_MAP[defaultMap].partition;
        let domainSuffix = fact_tables_1.PARTITION_MAP[defaultMap].domainSuffix;
        for (const key in fact_tables_1.PARTITION_MAP) {
            if (region.startsWith(key)) {
                partition = fact_tables_1.PARTITION_MAP[key].partition;
                domainSuffix = fact_tables_1.PARTITION_MAP[key].domainSuffix;
            }
        }
        registerFact(region, 'PARTITION', partition);
        registerFact(region, 'DOMAIN_SUFFIX', domainSuffix);
        registerFact(region, 'CDK_METADATA_RESOURCE_AVAILABLE', fact_tables_1.AWS_CDK_METADATA.has(region) ? 'YES' : 'NO');
        registerFact(region, 'S3_STATIC_WEBSITE_ENDPOINT', fact_tables_1.AWS_OLDER_REGIONS.has(region)
            ? `s3-website-${region}.${domainSuffix}`
            : `s3-website.${region}.${domainSuffix}`);
        registerFact(region, 'S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID', fact_tables_1.ROUTE_53_BUCKET_WEBSITE_ZONE_IDS[region] || '');
        registerFact(region, 'EBS_ENV_ENDPOINT_HOSTED_ZONE_ID', fact_tables_1.EBS_ENV_ENDPOINT_HOSTED_ZONE_IDS[region] || '');
        registerFact(region, 'ELBV2_ACCOUNT', fact_tables_1.ELBV2_ACCOUNTS[region]);
        registerFact(region, 'DLC_REPOSITORY_ACCOUNT', fact_tables_1.DLC_REPOSITORY_ACCOUNTS[region]);
        registerFact(region, 'APPMESH_ECR_ACCOUNT', fact_tables_1.APPMESH_ECR_ACCOUNTS[region]);
        const firehoseCidrBlock = fact_tables_1.FIREHOSE_CIDR_BLOCKS[region];
        if (firehoseCidrBlock) {
            registerFact(region, 'FIREHOSE_CIDR_BLOCK', `${fact_tables_1.FIREHOSE_CIDR_BLOCKS[region]}/27`);
        }
        const vpcEndpointServiceNamePrefix = `${domainSuffix.split('.').reverse().join('.')}.vpce`;
        registerFact(region, 'VPC_ENDPOINT_SERVICE_NAME_PREFIX', vpcEndpointServiceNamePrefix);
        for (const service of aws_entities_1.AWS_SERVICES) {
            registerFact(region, ['servicePrincipal', service], default_1.Default.servicePrincipal(service, region, domainSuffix));
        }
        for (const version in fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS) {
            for (const arch in fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS[version]) {
                registerFact(region, ['cloudwatchLambdaInsightsVersion', version, arch], fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS[version][arch][region]);
            }
        }
    }
    lines.push('  }');
    lines.push('');
    lines.push('  private constructor() {}');
    lines.push('}');
    await fs.writeFile(path.resolve(__dirname, '..', 'lib', 'built-ins.generated.ts'), lines.join('\n'));
    function registerFact(region, name, value) {
        const factName = typeof name === 'string' ? name : `${name[0]}(${name.slice(1).map(s => JSON.stringify(s)).join(', ')})`;
        lines.push(`    Fact.register({ region: ${JSON.stringify(region)}, name: FactName.${factName}, value: ${JSON.stringify(value)} });`);
    }
}
/**
 * Verifies that the provided map of region to fact does not contain an entry
 * for a region that was not registered in `AWS_REGIONS`.
 */
function checkRegions(map) {
    const allRegions = new Set(aws_entities_1.AWS_REGIONS);
    for (const region of Object.keys(map)) {
        if (!allRegions.has(region)) {
            throw new Error(`Un-registered region fact found: ${region}. Add to AWS_REGIONS list!`);
        }
    }
}
/**
 * Verifies that the provided map of <KEY> to region to fact does not contain an entry
 * for a region that was not registered in `AWS_REGIONS`.
 */
function checkRegionsSubMap(map) {
    const allRegions = new Set(aws_entities_1.AWS_REGIONS);
    for (const key of Object.keys(map)) {
        for (const subKey of Object.keys(map[key])) {
            for (const region of Object.keys(map[key][subKey])) {
                if (!allRegions.has(region)) {
                    throw new Error(`Un-registered region fact found: ${region}. Add to AWS_REGIONS list!`);
                }
            }
        }
    }
}
main().catch(e => {
    // eslint-disable-next-line no-console
    console.error(e);
    process.exit(-1);
});
//# sourceMappingURL=data:application/json;base64,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