from typing import List
from .connector.amqp_client import Amqp_Connector
from .connector.mqtt_client import Mqtt_Connector
from .connector.prom_connector import Prom_Connector
from .probes import Gauge, Counter, Summary, Histogram
import json, uuid
import threading
from threading import Thread
import time
from datetime import datetime

class Qoa_Client(object):
    # Init QoA Client
    def __init__(self, client_conf: dict, connector_conf: dict):
        '''
        Client configuration contains the information about the client and its configuration in form of dictionary
        Example: 
        { 
            "client_id": "aaltosea1",
            "component_id": "data_processing"
        }
        The `connector_conf` is the dictionary containing multiple connector configuration (amqp, mqtt, kafka)
        Example: 
        {
            "amqp_connector":{
                "class": "amqp",
                "conf":{
                    "end_point": "localhost",
                    "exchange_name": "qoa4ml",
                    "exchange_type": "topic",
                    "out_routing_key": "qoa.report.ml"
                }
            }
        }
        '''

        self.config = client_conf
        self.metrics = {}
        self.metrics["default"] = {}
        self.connector = {}
        # Init all connectors in for loop 
        for key in connector_conf:
            self.connector[key] = self.init_connector(connector_conf[key])
        
        # Set default connector for sending monitoring data if not specify
        self.default_connector = list(self.connector.keys())[0]
        # self.add_metric(metric_conf)
        self.lock = threading.Lock()
        

    def init_connector(self, configuration: dict):
        if configuration["class"] == "amqp":
            return Amqp_Connector(configuration["conf"])
        if configuration["class"] == "mqtt":
            return Mqtt_Connector(configuration["conf"])
        # if configuration["class"] == "kafka":
        #     return Kafka_Connector(configuration["conf"])
        
    def add_metric(self, metric_conf: dict, category=None):
        # Add multiple metrics 
        if (category == None):
            # category = "default"
            pass
        else:
            if category in self.metrics:
                pass
            else:
                print(category)
                self.metrics[category] = {}
            for key in metric_conf:
                print(key)
                if key in self.metrics[category]:
                    pass
                else:
                    print(metric_conf[key])
                    self.metrics[category][key] = self.init_metric(key, metric_conf[key])

    def init_metric(self, name, configuration: dict):
        # init individual metrics
        if configuration["class"] == "Gauge":
            return Gauge(name, configuration["description"], configuration["default"])
        elif configuration["class"] == "Counter":
            return Counter(name, configuration["description"], configuration["default"])
        elif configuration["class"] == "Summary":
            return Summary(name, configuration["description"], configuration["default"])
        elif configuration["class"] == "Histogram":
            return Histogram(name, configuration["description"], configuration["default"])

    def get(self):
        # TO DO:
        return self.config

    def get_metric(self, key=None, category=None):
        # TO DO:
        if (category == None) & (key == None):
            metrics = {}
            for category in self.metrics:
                metrics.update(self.metrics[category])
            return metrics
        if (key == None) & (category != None):
            return self.metrics[category]
        if (isinstance(key, list)) & (category!=None):
            return dict((k, self.metrics[category][k]) for k in key)
        if (key!=None) & (category!=None): 
            return self.metrics[category][key]

    def set(self, key, value):
        # TO DO:
        try:
            self.config[key] = value
        except Exception as e:
            print("{} not found - {}".format(key,e))
    
    
    def generate_report(self, metric:list=None, category=None):
        report = self.config
        report['timestamp'] = str(datetime.fromtimestamp(time.time()))
        if (category == None):
            for category_key in self.metrics:
                report[category_key] = {}
                if (metric == None):
                    metric = list(self.metrics[category_key].keys())
                for key in metric:
                    report[category_key][key] = self.metrics[category_key][key].get_val()
        else:
            report[category] = {}
            if (metric == None):
                metric = list(self.metrics[category].keys())
            for key in metric:
                report[category][key] = self.metrics[category][key].get_val()
        return report
    

    def asyn_report(self, metrics:list=None, report:dict = None, connectors:list=None, category=None):
        if (report == None):
            report = self.generate_report(metrics, category)
        body_mess = json.dumps(report)
        self.lock.acquire()
        if (connectors == None):
            self.connector[self.default_connector].send_data(body_mess,str(uuid.uuid4()))
        else:
            for connector in connectors:
                print(connector)
        self.lock.release()

    def report(self, metrics:list=None, report: dict = None, connectors:list=None, category=None):
        sub_thread = Thread(target=self.asyn_report, args=(metrics,report, connectors, category))
        sub_thread.start()


    def __str__(self):
        return str(self.config) + '\n' + str(self.connector)